<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Catalog_Model_Attributes_Product extends MageQL_Catalog_Model_Attributes_Abstract {
    const SUPPORTED_TYPE_IDS = [
        "bundle",
        "configurable",
        "simple",
        "virtual",
    ];

    protected function getBaseAttributes(): array {
        return [
            "attribute_set_id",
            "entity_id",
            "has_options",
            "sku",
            "type_id",
        ];
    }

    protected function getEntityType(): string {
        return "catalog_product";
    }

    protected function getFieldAttributeMap(): array {
        return [
            "url" => ["url_key", "url_path"],
        ];
    }

    protected function filterAttributeData($a) {
        $a = parent::filterAttributeData($a);

        // Categories do not have any special types
        $a["apply_to"] = array_intersect($a["apply_to"], self::SUPPORTED_TYPE_IDS);

        // If the attribute applies to all supported types we include it as a default
        if(empty(array_diff(self::SUPPORTED_TYPE_IDS, $a["apply_to"]))) {
            $a["apply_to"] = [];
        }

        return $a;
    }

    public function resizeImage($src, string $attr, string $image, array $args): string {
        $helper = Mage::helper("catalog/image");

        $helper->init($src, $attr, $image);
        $helper->keepFrame($args["fill"] ?? false);

        if(array_key_exists("height", $args) || array_key_exists("width", $args)) {
            $helper->resize($args["width"] ?? null, $args["height"] ?? null);
        }

        return (string)$helper;
    }

    /**
     * Returns all attributes which can filter products.
     */
    public function getFilterableAttributes(): array {
        $attrs = [];

        foreach($this->getAttributes() as $field => $a) {
            if($a["filterable"]) {
                $attrs[$field] = $a;
            }
        }

        return $attrs;
    }

    public static function resolveAttribute($src, array $args, $ctx, ResolveInfo $info): Mage_Eav_Model_Entity_Attribute{
        $attributeConfig = Mage::getSingleton("mageql_catalog/attributes_product");
        $attr = $attributeConfig->getAttributes()[$info->fieldName];
        $eav = Mage::getSingleton("eav/config");

        return $eav->getAttribute(Mage_Catalog_Model_Product::ENTITY, $attr["code"]);
    }

    public static function resolveLabel($attr, array $args, $ctx, ResolveInfo $info): string {
        return $attr->getStoreLabel() ?: $attr->getFrontend()->getLabel() ?: $attr->getLabel();
    }

    public static function resolveValues($attr, array $args, $ctx, ResolveInfo $info): ?array {
        if( ! $attr->usesSource()) {
            return null;
        }

        $resource  = Mage::getResourceSingleton("catalog/product_flat");
        $session = Mage::getSingleton("customer/session");
        $attrValue = $attr->getAttributeCode()."_value";
        $db = $resource->getReadConnection();

        $select = $db->select();

        $select->distinct();
        $select->from(["p" => $resource->getFlatTableName()], [$attrValue]);

        $select->join(
            ["f" => "catalog_product_index_price"],
            sprintf(
                "p.entity_id = f.entity_id AND f.website_id = %d AND f.customer_group_id = %d",
                $ctx->getStore()->getWebsiteId(),
                $session->getCustomerGroupId()), []);

        $select->where("p.status = ?", Mage_Catalog_Model_Product_Status::STATUS_ENABLED);
        $select->where("p.type_id IN (?)", ["simple", "virtual"]);
        $select->where("p.$attrValue IS NOT NULL");

        $values = $db->query($select)->fetchAll(PDO::FETCH_COLUMN);

        sort($values);

        return array_filter($values);
    }
}
