<?php

declare(strict_types=1);

namespace Fixtures;

use Mage;
use Mage_Eav_Model_Entity_Setup;
use Mage_Catalog_Model_Product_Status;
use Mage_Catalog_Model_Product_Visibility;
use Crossroads\Magento\Test\Integration\MagentoManager;

class Products {
    public function load(): void {
        MagentoManager::initAdmin();

        $this->createAttributes();

        MagentoManager::reset();
        MagentoManager::initAdmin();

        $this->createSimpleProduct();
        $this->createVirtualProduct();
        $this->createConfigurableProduct();
        $this->assignConfigurableProductLinks();

        MagentoManager::reindex();

        MagentoManager::reset();
    }

    public function createAttribute(string $code, array $newOptions) {
        $installer = new Mage_Eav_Model_Entity_Setup("core_setup");
        $installer->startSetup();

        $attr = Mage::getModel("eav/entity_attribute")->loadByCode("catalog_product", $code);

        $attr->setIsVisibleOnFront(true);
        $attr->setUsedInProductListing(true);
        $attr->save();

        $existing = [];
        $options = Mage::getModel("eav/entity_attribute_source_table")
            ->setAttribute($attr)
            ->getAllOptions(false);

        foreach($options as $option) {
            if(in_array($option["label"], $newOptions)) {
                array_push($existing, $option["label"]);
            }
        }

        $insertOptions = array_diff($newOptions, $existing);

        if( ! empty($insertOptions)) {
            $installer->addAttributeOption([
                "attribute_id" => $attr->getId(),
                "values" => $insertOptions,
            ]);
        }

        $attributeSetId = $installer->getAttributeSetId("catalog_product", "Default");

        $installer->addAttributeToSet("catalog_product", $attributeSetId, "General", $code, 10);

        $installer->endSetup();
    }

    public function createAttributes() {
        $this->createAttribute("color", ["Red","Black", "Yellow"]);
        $this->createAttribute("manufacturer", ["Manufacturer A", "Manufacturer B"]);
    }

    public function getAttributeValue($code, $option) {
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", $code);
        $attrOptions = $attr->getSource()->getAllOptions();

        foreach($attrOptions as $o) {
            if($o["label"] === $option) {
                return $o["value"];
            }
        }

        return null;
    }

    public function createSimpleProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $model = Mage::getModel("catalog/product");
        $category = MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category");
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer A");

        $model->loadByAttribute("sku", "test-simple");
        $model->addData([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "attribute_set_id" => $model->getDefaultAttributeSetId(),
            "type_id" => "simple",
            "sku" => "test-simple",
            "name" => "Test for Simple Product",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "meta_title" => "Product meta title",
            "meta_description" => "Product meta description",
            "meta_keyword" => "Product,meta,keywords",
            "description" => "The long product description",
            "short_description" => "The shortest product description",
            "weight" => 3.14,
            "small_image" => "product_small_image.jpg",
            "small_image_label" => "Small Product Image",
            "thumbnail" => "product_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Product Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "first_gallery.jpg", "position" => 0, "label" => "First Gallery", "disabled" => 0],
                    ["file" => "second_gallery.jpg", "position" => 1, "label" => "Second Gallery", "disabled" => 0],
                    ["file" => "disabled_gallery.jpg", "position" => 2, "label" => "Disabled Gallery", "disabled" => 1],
                ],
            ],
            "category_ids" => [$category->getId()],
        ]);

        $model->save();

        Mage::getModel("cataloginventory/stock_item")
            ->addData([
                "stock_id" => 1,
                "use_config_manage_stock" => 0,
                "manage_stock" => 1,
                "min_sale_qty" => 1,
                "max_sale_qty" => 2,
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->assignProduct($model)->save();
    }

    public function createVirtualProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $model = Mage::getModel("catalog/product");
        $category = MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category");

        $model->loadByAttribute("sku", "test-virtual");
        $model->addData([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "attribute_set_id" => $model->getDefaultAttributeSetId(),
            "type_id" => "virtual",
            "sku" => "test-virtual",
            "name" => "Test for Virtual Product",
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 9.99,
            "meta_title" => "Product meta title virtual",
            "meta_description" => "Product meta description virtual",
            "meta_keyword" => "Product,meta,keywords,Virtual",
            "description" => "The long product description for virtual",
            "short_description" => "Virtual short",
            "small_image" => "virtual_small.jpg",
            "small_image_label" => "Small Virtual Product Image",
            "thumbnail" => "virtual_thumbnail.jpg",
            "thumbnail_label" => "Thumbnail Virtual Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "virtual_first_gallery.jpg", "position" => 0, "label" => "First Virtual Gallery", "disabled" => 0],
                    ["file" => "virtual_second_gallery.jpg", "position" => 1, "label" => "Second Virtual Gallery", "disabled" => 0],
                    ["file" => "virtual_disabled_gallery.jpg", "position" => 2, "label" => "Disabled Virtual Gallery", "disabled" => 1],
                ],
            ],
            "category_ids" => [$category->getId()],
        ]);

        $model->save();

        Mage::getModel("cataloginventory/stock_item")
            ->addData([
                "stock_id" => 1,
                "use_config_manage_stock" => 0,
                "manage_stock" => 1,
                "min_sale_qty" => 1,
                "max_sale_qty" => 2,
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->assignProduct($model)->save();
    }

    public function createConfigurableProduct() {
        $store = Mage::getModel("core/store")->load(MagentoManager::TESTING_STORE);
        $category = MagentoManager::loadCategoryByAttribute("varchar", "url_key", "test-category");
        $manufacturer = $this->getAttributeValue("manufacturer", "Manufacturer B");
        $attr = Mage::getSingleton("eav/config")->getAttribute("catalog_product", "color");
        $attrOptions = $attr->getSource()->getAllOptions();
        $attrRed = null;
        $attrBlack = null;

        foreach($attrOptions as $o) {
            switch($o["label"]) {
            case "Red":
                $attrRed = $o["value"];
                break;
            case "Black":
                $attrBlack = $o["value"];
                break;
            }
        }

        $model = Mage::getModel("catalog/product");
        $child1 = Mage::getModel("catalog/product");
        $child2 = Mage::getModel("catalog/product");

        $child1->loadByAttribute("sku", "test-config-child-1");
        $child1->addData([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "attribute_set_id" => $child1->getDefaultAttributeSetId(),
            "type_id" => "simple",
            "sku" => "test-config-child-1",
            "name" => "Red Child",
            "manufacturer" => $manufacturer,
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 12.34,
            "color" => $attrRed,
            "meta_title" => "Meta title Child 1 for config product",
            "meta_description" => "Product meta description config child 1",
            "meta_keyword" => "Product,meta,keywords,config,child,1",
            "description" => "The long product description for config child 1",
            "short_description" => "Config short child 1",
            "small_image" => "config_small_child1.jpg",
            "small_image_label" => "Small Config Product Image Child 1",
            "thumbnail" => "config_thumb_child1.jpg",
            "thumbnail_label" => "Thumbnail Thumbnail Image Child 1",
            "media_gallery" => [
                "images" => [
                    ["file" => "config_first_gallery_child1.jpg", "position" => 0, "label" => "First Config Gallery Child 1", "disabled" => 0],
                    ["file" => "config_second_gallery_child1.jpg", "position" => 1, "label" => "Second Config Galler Child 1y", "disabled" => 0],
                    ["file" => "config_disabled_gallery_child1.jpg", "position" => 2, "label" => "Disabled Config Gallery Child 1", "disabled" => 1],
                ],
            ],
            "category_ids" => [],
        ]);

        $child1->save();

        Mage::getModel("cataloginventory/stock_item")
            ->addData([
                "stock_id" => 1,
                "use_config_manage_stock" => 0,
                "manage_stock" => 1,
                "min_sale_qty" => 1,
                "max_sale_qty" => 2,
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->assignProduct($child1)->save();

        $child2->loadByAttribute("sku", "test-config-child-2");
        $child2->addData([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "attribute_set_id" => $child2->getDefaultAttributeSetId(),
            "type_id" => "simple",
            "sku" => "test-config-child-2",
            "name" => "Black Child",
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_NOT_VISIBLE,
            "price" => 22.34,
            "color" => $attrBlack,
            "meta_title" => "Meta title Child 2 for config product",
            "meta_description" => "Product meta description config child 2",
            "meta_keyword" => "Product,meta,keywords,config,child,2",
            "description" => "The long product description for config child 2",
            "short_description" => "Config short child 2",
            "small_image" => "config_small_child2.jpg",
            "small_image_label" => "Small Config Product Image Child 2",
            "thumbnail" => "config_thumb_child2.jpg",
            "thumbnail_label" => "Thumbnail Thumbnail Image Child 2",
            "media_gallery" => [
                "images" => [
                    ["file" => "config_first_gallery_child2.jpg", "position" => 0, "label" => "First Config Gallery Child 2", "disabled" => 0],
                    ["file" => "config_second_gallery_child2.jpg", "position" => 1, "label" => "Second Config Galler Child 2", "disabled" => 0],
                    ["file" => "config_disabled_gallery_child2.jpg", "position" => 2, "label" => "Disabled Config Gallery Child 2", "disabled" => 1],
                ],
            ],
            "category_ids" => [],
        ]);

        $child2->save();

        Mage::getModel("cataloginventory/stock_item")
            ->addData([
                "stock_id" => 1,
                "use_config_manage_stock" => 0,
                "manage_stock" => 1,
                "min_sale_qty" => 1,
                "max_sale_qty" => 2,
                "is_in_stock" => 1,
                "qty" => 999,
            ])
            ->assignProduct($child2)->save();

        $model->loadByAttribute("sku", "test-config");
        $model->addData([
            "store_id" => $store->getId(),
            "website_ids" => [$store->getWebsiteId()],
            "attribute_set_id" => $model->getDefaultAttributeSetId(),
            "type_id" => "configurable",
            "sku" => "test-config",
            "name" => "Test for Configurable Product",
            "status" => Mage_Catalog_Model_Product_Status::STATUS_ENABLED,
            "tax_class" => 1, // Default
            "visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
            "price" => 12.34,
            "meta_title" => "Product meta title config",
            "meta_description" => "Product meta description config",
            "meta_keyword" => "Product,meta,keywords,config",
            "description" => "The long product description for config",
            "short_description" => "Config short",
            "small_image" => "config_small.jpg",
            "small_image_label" => "Small Config Product Image",
            "thumbnail" => "config_thumb.jpg",
            "thumbnail_label" => "Thumbnail Thumbnail Image",
            "media_gallery" => [
                "images" => [
                    ["file" => "config_first_gallery.jpg", "position" => 0, "label" => "First Config Gallery", "disabled" => 0],
                    ["file" => "config_second_gallery.jpg", "position" => 1, "label" => "Second Config Gallery", "disabled" => 0],
                    ["file" => "config_disabled_gallery.jpg", "position" => 2, "label" => "Disabled Config Gallery", "disabled" => 1],
                ],
            ],
            "category_ids" => [$category->getId()],
        ]);

        $model->save();

        Mage::getModel("cataloginventory/stock_item")
            ->addData([
                "stock_id" => 1,
                "use_config_manage_stock" => 0,
                "manage_stock" => 1,
                "is_in_stock" => 1,
            ])
            ->assignProduct($model)->save();

        // Clear the super attribute links before saving them since magento has trouble updating them:
        $resource = Mage::getSingleton('core/resource');

        $resource->getConnection("core_write")
            ->delete($resource->getTableName("catalog/product_super_attribute"), "product_id = ".$model->getId());
        $resource->getConnection("core_write")
            ->delete($resource->getTableName("catalog/product_super_link"), "parent_id = ".$model->getId());

        $model->getTypeInstance()->setUsedProductAttributeIds([$attr->getId()]);
        $cfgData = $model->getTypeInstance()->getConfigurableAttributesAsArray();

        $model->setCanSaveConfigurableAttributes(true);
        $model->setConfigurableAttributesData($cfgData);

        $childProducts = [];
        $childProducts[$child1->getId()] = [
            0 => [
                "label" => "Red",
                "attribute_id" => $attr->getId(),
                "value_index" => $attrRed,
                "is_percent" => "0",
                // Value for price, should be ignored with simple-configurable-products:
                "pricing_value" => "21",
            ]
        ];
        $childProducts[$child2->getId()] = [
            0 => [
                "label" => "Black",
                "attribute_id" => $attr->getId(),
                "value_index" => $attrBlack,
                "is_percent" => "0",
                // Value for price, should be ignored with simple-configurable-products:
                "pricing_value" => "22",
            ]
        ];

        $model->setConfigurableProductsData($childProducts);
        $model->save();
    }

    public function assignConfigurableProductLinks() {
        // We need to load it again since the child-product links will cause issues
        $model = Mage::getModel("catalog/product")->loadByAttribute("sku", "test-config");

        $simpleId = $model->getIdBySku("test-simple");
        $virtualId = $model->getIdBySku("test-virtual");

        $upSell = [
            $simpleId => [ "position" => "" ],
        ];
        $crossSell = [
            $virtualId => [ "position" => "" ],
        ];
        $related = [
            $virtualId => [ "position" => 1 ],
            $simpleId => [ "position" => 2 ],
        ];

        $model->setUpSellLinkData($upSell);
        $model->setCrossSellLinkData($crossSell);
        $model->setRelatedLinkData($related);

        $model->save();
    }
}
