<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;

class MageQL_Catalog_Model_Product extends Mage_Core_Model_Abstract {
    public static function resolveBySku($src, array $args, $ctx, ResolveInfo $info): ?Mage_Catalog_Model_Product {
        $product = Mage::getModel("catalog/product");

        $id = $product->getIdBySku($args["sku"]);

        if($id) {
            // TODO: Limit the loaded attributes to the ones we need
            $product->load($id);
        }

        if( ! $product->getId()) {
            return null;
        }

        // Magento event to log product visit
        Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);

        return $product;
    }

    public static function resolveByRoute($rewrite) {
        $product = Mage::getModel("catalog/product");

        // TODO: Limit the loaded attributes to the ones we need
        $product->load($rewrite->getProductId());

        if( ! $product->getId()) {
            return null;
        }

        // Magento event to log product visit
        Mage::dispatchEvent("catalog_controller_product_view", ["product" => $product]);

        return $product;
    }

    public static function resolvePaginatedItems($collection, array $args, $ctx, ResolveInfo $info) {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        // 2 levels deep to get attributes (1 product, 2 attributes)
        $fields = $info->getFieldSelection(2);
        // Merge attributes
        $fields = array_merge($fields, $fields["attributes"] ?? []);

        // We use all attribute sets since we cannot be certain about which sets we will get
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        foreach($toSelect as $col) {
            $collection->addAttributeToSelect($col);
        }

        return $collection->getItems();
    }

    public static function resolveConfigurationOptionAttributes($src, array $args, $ctx, ResolveInfo $info) {
        // This will be cached on the product
        return $src->getTypeInstance(true)->getConfigurableAttributes($src);
    }

    public static function resolveConfigurationOptionItems($product, array $args, $ctx, ResolveInfo $info) {
        $instance = $product->getTypeInstance(true);
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        $prodAttrs = $instance->getConfigurableAttributes($product);
        // Get 3 levels deep, since that is the attribute nesting
        $fields = $info->getFieldSelection(3)["product"] ?? [];
        $fields = array_merge($fields, $fields["attributes"] ?? []);

        // Local implementation of Mage_Catalog_Model_Product_Type_Configurable::getUsedProducts

        $collection = $instance->getUsedProductCollection($product);

        $collection->addFilterByRequiredOptions();
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();
        $collection->addUrlRewrite();
        // TODO: Can we sort for this?
        $collection->addAttributeToSort("position", "asc");

        // No limit

        // We use all attribute sets since we cannot be certain about which
        // sets we will get, even though they should be the same as parent
        $toSelect = $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );

        // Add the attributes we use for the configurating
        foreach($prodAttrs as $attr) {
            $collection->addAttributeToSelect($attr->getProductAttribute()->getAttributeCode());
        }

        foreach($toSelect as $col) {
            $collection->addAttributeToSelect($col);
        }

        return array_map(function($item) use($prodAttrs) {
            return [$item, $prodAttrs];
        }, $collection->getItems());
    }

    public static function resolveConfigurationOptionItemValues($src) {
        $items = [];
        $prod = $src[0];
        $attrs = $src[1];

        foreach($attrs as $attr) {
            $prodAttr = $attr->getProductAttribute();
            $model = $prodAttr->getSource();

            $items[] = [
                "attribute" => snakeCaseToCamel($prodAttr->getAttributeCode()),
                "value" => $model->getOptionText($prod->getData($prodAttr->getAttributeCode())),
            ];
        }

        return $items;
    }

    /**
     * @param $src Mage_Catalog_Model_Product
     * @param $args Array of settings: page, pageSize
     * @param $ctx MageQL_Core_Model_Context
     */
    public static function resolveRelatedProducts($src, array $args, $ctx) {
        return self::prepareProductLinkCollection(
            $src->getRelatedProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param $src Mage_Catalog_Model_Product
     * @param $args Array of settings: page, pageSize
     * @param $ctx MageQL_Core_Model_Context
     */
    public static function resolveUpSellProducts($src, array $args, $ctx) {
        return self::prepareProductLinkCollection(
            $src->getUpSellProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param $src Mage_Catalog_Model_Product
     * @param $args Array of settings: page, pageSize
     * @param $ctx MageQL_Core_Model_Context
     */
    public static function resolveCrossSellProducts($src, array $args, $ctx) {
        return self::prepareProductLinkCollection(
            $src->getCrossSellProductCollection(),
            $args,
            $ctx
        );
    }

    /**
     * @param $src any Ignored
     * @param $args Array of settings: page, pageSize
     * @param $ctx MageQL_Core_Model_Context
     */
    public static function resolveBestsellingProducts($src, array $args, $ctx) {
        $storeId = (int)$ctx->getStore()->getId();
        $collection = Mage::getModel("catalog/product")->getCollection();
        $fromDate = date("Y-m-01");

        $collection->getSelect()
            ->join(
                ["aggregation" => $collection->getResource()->getTable("sales/bestsellers_aggregated_monthly")],
                "e.entity_id = aggregation.product_id AND aggregation.store_id = $storeId AND aggregation.period = '$fromDate'",
                ["SUM(aggregation.qty_ordered) AS sold_quantity"]
            )
            ->group("e.entity_id")
            ->order(array("sold_quantity DESC", "e.created_at"));

        return self::preparePaginatedProductCollection(
            $collection,
            $args,
            $ctx
        );
    }

    public static function resolveProductsByFilter($src, array $args, $ctx, ResolveInfo $info) {
        $attributeConfig = Mage::getSingleton("mageql_catalog/attributes_product");
        $attr = $attributeConfig->getAttributes()[$info->fieldName];

        if( ! $attr) {
            Mage::throwException("Unknown attribute field $attr.");
        }

        $collection = Mage::getModel("catalog/product")->getCollection();
        $attribute = Mage::getModel("catalog/product")->getResource()->getAttribute($attr["code"]);

        if(array_key_exists("value", $args)) {
            $value = $attribute->usesSource() ?
                $attribute->getSource()->getOptionId($args["value"]) :
                $args["value"];

            $collection->addAttributeToFilter($attr["code"], $value);
        }
        else if($attr["backend_input"] === "datetime") {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "gteq" => date("Y-m-d H:i:s", strtotime($attr["from"])),
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "lteq" => date("Y-m-d H:i:s", strtotime($attr["to"])),
                ]);
            }
        }
        else {
            if( ! empty($attr["from"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "gteq" => (float)$attr["from"],
                ]);
            }

            if( ! empty($attr["to"])) {
                $collection->addAttributeToFilter([
                    "attribute" => $attr["code"],
                    "lteq" => (float)$qttr["to"],
                ]);
            }
        }

        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->addAttributeToSort("name", Varien_Data_Collection::SORT_ORDER_ASC);

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    public static function resolveProductsBySearch($src, array $args, $ctx, ResolveInfo $info) {
        $query = Mage::getModel("catalogsearch/query");
        $fulltext = Mage::getModel("catalogsearch/fulltext");
        $searchHelper = Mage::getResourceHelper('catalogsearch');
        $resource = $fulltext->getResource();
        $minLength = $ctx->getStore()->getConfig(Mage_CatalogSearch_Model_Query::XML_PATH_MIN_QUERY_LENGTH);

        if(strlen($args["term"]) < $minLength) {
            return null;
        }

        $query->setStoreId($ctx->getStore()->getId());
        $query->loadByQuery($args["term"]);

        if( ! $query->getId()) {
            $query->setQueryText($args["term"]);
            $query->setStoreId($ctx->getStore()->getId());
            $query->setPopularity(1);
        }
        else {
            $query->setPopularity($query->getPopularity() + 1);
        }

        $query->prepare();

        $resource->prepareResult($fulltext, $args["term"], $query);

        $collection = Mage::getModel("catalog/product")->getCollection();
        $foundData = $resource->getFoundData();

        ksort($foundData);
        natsort($foundData);

        $foundIds = array_keys($foundData);

        $collection->addIdFilter($foundIds);
        $collection->addUrlRewrite();

        // TODO: Will this make it too slow?
        // Sort by relevance
        $collection->getSelect()->order(new Zend_Db_Expr(
            $searchHelper->getFieldOrderExpression("e.entity_id", $foundIds)." ".Zend_Db_Select::SQL_ASC.
            ", e.entity_id ASC"
        ));

        $query->save();

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * Prepares a product-link collection like related, upsell and crossell for
     * use in a PaginatedProducts type.
     *
     * @param $src Mage_Catalog_Model_Resource_Product_Collection
     * @param $args Array of settings: page, pageSize
     * @param $ctx MageQL_Core_Model_Context
     */
    public static function prepareProductLinkCollection($collection, array $args, $ctx) {
        // Just plain URLs here since we do not have a category
        $collection->addUrlRewrite();
        $collection->setPositionOrder();

        return self::preparePaginatedProductCollection($collection, $args, $ctx);
    }

    /**
     * Prepares a basic product collection for use in a PaginatedProducts type,
     * does not add sort, url-rewrite, or any kind of filter to the collection
     * besides store and visibility.
     *
     * @param $src Mage_Catalog_Model_Resource_Product_Collection
     * @param $args array of settings: page, pageSize
     * @param $ctx MageQL_Core_Model_Context
     */
    public static function preparePaginatedProductCollection($collection, array $args, $ctx) {
        $visibility = ["in" => [
            Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG,
            Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH
        ]];

        // We do not yet select any attributes, that is done when we
        // fetch the data in the items resolver
        $collection->addAttributeToFilter("visibility", $visibility);
        $collection->addStoreFilter($ctx->getStore());
        $collection->addMinimalPrice();

        // We call this directly on the select to prevent magento from stopping us from
        // going off the end of the list. (setCurPage() prevents this, which is not correct
        // from an API PoV)
        $collection->getSelect()
                   ->limitPage(max($args["page"] ?? 1, 1), $args["pageSize"]);

        return $collection;
    }
}
