<?php

declare(strict_types=1);

use Elasticsearch\ClientBuilder;

class MageQL_Catalog_Model_ElasticSearch
{
    /** @var Elasticsearch\Client */
    private $client;
    /** @var Mage_Core_Model_Store */
    private $store;

    public function __construct($store)
    {
        $this->store = $store;
        $this->client = ClientBuilder::create()
        ->setHosts([$this->getHost()])
        ->setBasicAuthentication($this->getUsername(), $this->getPassword())
        ->build();
    }

    private function getHost(): ?string
    {
        return $this->store->getConfig('mageql/elasticsearch/host');
    }

    private function getUsername(): ?string
    {
        return $this->store->getConfig('mageql/elasticsearch/username');
    }

    private function getPassword(): ?string
    {
        return $this->store->getConfig('mageql/elasticsearch/password');
    }

    public function search(string $term): array
    {
        $search = $this->searchRaw($term);
        $results = [];

        if (!isset($search['hits']['hits'])) {
            // No hits
            return $results;
        }

        foreach ($search['hits']['hits'] as $hit) {
            $results[$hit['_source']['entity_id']] = $hit['_score'];
        }
        return $results;
    }

    private function searchRaw(string $term)
    {
        $params = [
            'index' => $this->store->getCode(),
            'body' => [
                'query' => [
                    'multi_match' => [
                        'query' => $term,
                        'type' => 'best_fields',
                        'fields' => ["sku^10", "name^10", "manufacturer_value", "short_description"],
                    ],
                ],
            ],
        ];
        $result = $this->client->search($params);
        return $result;
    }
}