<?php

declare(strict_types=1);

/**
 * @psalm-type ProductFilterInput array{
 *   code: string,
 *   value?: ?string,
 *   minValue?: ?float,
 *   maxValue?: ?float,
 *   incVat?: ?bool,
 * }
 */
class MageQL_Catalog_Model_Filter_Product_Price
    implements MageQL_Catalog_Model_Filter_Product_Interface {
    public function getFilterableBy(
        Mage_Catalog_Model_Resource_Product_Collection $collection
    ): array {
        return [
            "price" => new MageQL_Catalog_Model_Product_Filter_Price($collection),
        ];
    }

    /**
     * @param ProductFilterInput $filter
     */
    public function applyFilter(
        Mage_Catalog_Model_Resource_Product_Collection $collection,
        array $filter
    ): bool {
        if($filter["code"] !== "price") {
            return false;
        }

        // Minimum and maximum values have already been filtered to be Floats by GraphQL
        $minValue = $filter["minValue"] ?? null;
        $maxValue = $filter["maxValue"] ?? null;
        $taxSetting = Mage::getStoreConfig(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX) ? true : false;
        $incVat = $filter["incVat"] ?? $taxSetting;
        $select = $collection->getSelect();

        if ($taxSetting) {
            // Magento instance use prices including tax
            if ($incVat) {
                // User query includes tax
                if($minValue !== null) {
                    $select->where("price_index.min_price >= ?", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("price_index.min_price <= ?", $maxValue);
                }
            } else {
                // User query excludes tax
                $select->join(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
                $select->join(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

                if($minValue !== null) {
                    $select->where("price_index.min_price >= ? * (1 + (tcr.rate / 100))", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("price_index.min_price <= ? * (1 + (tcr.rate / 100))", $maxValue);
                }
            }
        } else {
            // Magento instance use prices excluding tax
            if ($incVat) {
                // User query includes tax
                $select->join(["tc" => "tax_class"], "tc.class_id = price_index.tax_class_id", []);
                $select->join(["tcr" => "tax_calculation_rate"], "tcr.code = tc.class_name", []);

                if($minValue !== null) {
                    $select->where("price_index.min_price >= ? / (1 + (tcr.rate / 100))", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("price_index.min_price <= ? / (1 + (tcr.rate / 100))", $maxValue);
                }
            } else {
                // User query excludes tax
                if($minValue !== null) {
                    $select->where("price_index.min_price >= ?", $minValue);
                }

                if($maxValue !== null) {
                    $select->where("price_index.min_price <= ?", $maxValue);
                }
            }
        }

        return true;
    }
}
