<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

class MageQL_Catalog_Helper_Data extends Mage_Core_Helper_Abstract {

    const CONFIG_PRODUCT_FILTERS = "mageql/catalog/product/filter";

    /**
     * List of fields containing key-maps of attributes based on their attribute names.
     *
     * @var Array<string>
     */
    protected $listAttributeFields = ["attributes"];

    /**
     * Returns true if the supplied product is visible on its own in the
     * given store.
     */
    public function isProductVisible(
        Mage_Catalog_Model_Product $product,
        Mage_Core_Model_Store $store
    ): bool {
        return $product->getId() &&
            ! $product->isDisabled() &&
            $product->isVisibleInSiteVisibility() &&
            in_array($store->getWebsiteId(), $product->getWebsiteIds());
    }

    /**
     * @return $this
     */
    public function addListAttributeField(string $field): self {
        $this->listAttributeFields[] = $field;

        return $this;
    }

    /**
     * @param Array<string> $path
     * @return Array<string>
     */
    public function getSelectedFields(ResolveInfo $info, array $path): array {
        $config = Mage::getSingleton("mageql_catalog/attributes_product");
        // One extra level for attributes
        /**
         * Recursive types do not work properly, use a single level
         *
         * @var array<string, bool | array<string, bool>>
         */
        $fields = $info->getFieldSelection(count($path) + 1);

        foreach($path as $p) {
            /**
             * @var array<string, bool | array<string, bool>>
             */
            $fields = $fields[$p] ?? [];
        }

        // Merge attributes
        foreach($this->listAttributeFields as $f) {
            if(array_key_exists($f, $fields) && is_array($fields[$f])) {
                $fields = array_merge($fields, $fields[$f]);
            }
        }

        // We use all attribute sets since we cannot be certain about which sets we will get
        return $config->getUsedAttributes(
            $config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST),
            $fields
        );
    }

    public function getCustomProductFilters(Mage_Core_Model_Store $store): array {
        $data = [];
        $items = $store->getConfig(self::CONFIG_PRODUCT_FILTERS) ?: [];

        if( ! is_array($items)) {
            throw new RuntimeException(sprintf(
                "%s: Config value '%s' must be an array/map.",
                __METHOD__,
                self::CONFIG_PRODUCT_FILTERS
            ));
        }

        foreach($items as $key => $item) {
            if(array_key_exists("model", $item)) {
                $model = Mage::getSingleton($item["model"]);

                if( ! $model instanceof MageQL_Catalog_Model_Filter_Product_Interface) {
                    throw new Exception(sprintf(
                        "%s: Model instantiated from config '%s' must implement %s",
                        __METHOD__,
                        sprintf("config/default/mageql/catalog/product/filter/%s/model", $key),
                        MageQL_Catalog_Model_Filter_Product_Interface::class
                    ));
                }

                $data[$key] = $model;

                continue;
            }

            throw new Exception(sprintf("%s: Missing model tag in 'mageql/catalog/product/filter/%s'.", __METHOD__, $key));
        }

        return $data;
    }
}
