<?php

declare(strict_types=1);

namespace Fixtures;

use Mage;
use Mage_Tax_Model_Class;
use Mage_Tax_Model_Calculation_Rate;
use Mage_Tax_Model_Calculation_Rule;
use Mage_Tax_Model_Config;
use Mage_Core_Model_Resource_Setup;
use Mage_Customer_Model_Group;
use Crossroads\Magento\Test\Integration\MagentoManager;

class TaxRates {
    const CUSTOMER_TAX_CLASS = "Test Customer Tax Class";
    const PRODUCT_TAX_CLASS = "Test Product Tax Class";

    public static function getTaxClass(string $name): Mage_Tax_Model_Class {
        $collection = Mage::getResourceModel("tax/class_collection");

        $collection->addFieldToFilter("class_name", $name);
        $collection->load();

        /**
         * @var Mage_Tax_Model_Class
         */
        return $collection->getFirstItem();
    }

    public function load(): void {
        MagentoManager::initAdmin();

        $customerClass = $this->createClass(self::CUSTOMER_TAX_CLASS, Mage_Tax_Model_Class::TAX_CLASS_TYPE_CUSTOMER);
        $productClass = $this->createClass(self::PRODUCT_TAX_CLASS, Mage_Tax_Model_Class::TAX_CLASS_TYPE_PRODUCT);
        $rate = $this->createRate("SE", "Test 25% SE", 25.00);

        $this->createRule("Test Product Rule", $customerClass, $productClass, $rate);

        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_TAX_CLASS, (int)$productClass->getId());
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_PRICE_INCLUDES_TAX, "1");
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_SHIPPING_INCLUDES_TAX, "1");
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_DISCOUNT_TAX, "1");
        $setupModel->setConfigData(Mage_Tax_Model_Config::CONFIG_XML_PATH_DEFAULT_COUNTRY, "SE");

        $notLoggedIn = Mage::getModel("customer/group");
        $notLoggedIn->load(Mage_Customer_Model_Group::NOT_LOGGED_IN_ID);
        $general = Mage::getModel("customer/group")
            ->getCollection()
            ->addFieldToFilter("customer_group_code", "General")
            ->load()
            ->getFirstItem();

        $notLoggedIn->addData([
            "tax_class_id" => $customerClass->getId(),
        ]);
        $notLoggedIn->save();
        $general->addData([
            "tax_class_id" => $customerClass->getId(),
        ]);
        $general->save();

        MagentoManager::reset();
    }

    public function createClass(string $name, string $type): Mage_Tax_Model_Class {
        $collection = Mage::getResourceModel("tax/class_collection");
        $collection->addFieldToFilter("class_name", $name);
        $collection->load();

        /**
         * @var Mage_Tax_Model_Class $taxClass
         */
        $taxClass = $collection->getFirstItem();

        $taxClass->addData([
            "class_name" => $name,
            "class_type" => $type,
        ]);

        $taxClass->save();

        return $taxClass;
    }

    public function createRate(string $countryId, string $code, float $rate): Mage_Tax_Model_Calculation_Rate {
        $calcRate = Mage::getModel("tax/calculation_rate")
            ->loadByCode($code);

        $calcRate->addData([
            "tax_country_id" => $countryId,
            "tax_region_id" => "0",
            "tax_postcode" => "*",
            "code" => $code,
            "rate" => $rate,
        ]);

        $calcRate->save();

        return $calcRate;
    }

    /**
     * @psalm-suppress PossiblyUnusedReturnValue
     */
    public function createRule(
        string $code,
        Mage_Tax_Model_Class $customerClass,
        Mage_Tax_Model_Class $productClass,
        Mage_Tax_Model_Calculation_Rate $rate
    ): Mage_Tax_Model_Calculation_Rule {
        /**
         * @var Mage_Tax_Model_Calculation_Rule $rule
         */
        $rule = Mage::getModel("tax/calculation_rule")
            ->getCollection()
            ->addFieldToFilter("code", $code)
            ->load()
            ->getFirstItem();

        $rule->addData([
            "code" => $code,
            "tax_customer_class" => [$customerClass->getId()],
            "tax_product_class" => [$productClass->getId()],
            "tax_rate" => [$rate->getId()],
            "priority" => "0",
            "position" => "0",
        ]);

        $rule->save();

        return $rule;
    }
}
