<?php

declare(strict_types=1);

use function MageQL\snakeCaseToCamel;

/**
 * Option of a configurable product with a child being simple or virtual product.
 */
class MageQL_Catalog_Model_Product_Configurable_Option {
    /**
     * @readonly
     * @var Mage_Catalog_Model_Product
     */
    protected $parent;

    /**
     * @readonly
     * @var Mage_Catalog_Model_Product
     */
    protected $child;

    /**
     * @readonly
     * @var Mage_Catalog_Model_Resource_Product_Type_Configurable_Attribute_Collection
     */
    protected $attributes;

    public function __construct(
        Mage_Catalog_Model_Product $parent,
        Mage_Catalog_Model_Product $child,
        Mage_Catalog_Model_Resource_Product_Type_Configurable_Attribute_Collection $attributes
    ) {
        if($parent->getTypeId() !== "configurable") {
            throw new Exception(sprintf("%s: Invalid product type %s for config parent %d", __METHOD__, $parent->getTypeId(), (int)$parent->getId()));
        }

        if( ! in_array($child->getTypeId(), ["simple", "virtual"])) {
            throw new Exception(sprintf("%s: Invalid product type %s for config child %d in parent %d", __METHOD__, $child->getTypeId(), (int)$child->getId(), (int)$parent->getId()));
        }

        $this->parent = $parent;
        $this->child = $child;
        $this->attributes = $attributes;
    }

    public function getPrice(): MageQL_Catalog_Model_Product_Price {
        return new MageQL_Catalog_Model_Product_Price($this->child, $this->child->getFinalPrice());
    }

    public function getParent(): Mage_Catalog_Model_Product {
        return $this->parent;
    }

    public function getProduct(): Mage_Catalog_Model_Product {
        return $this->child;
    }

    public function getAttributes(): Mage_Catalog_Model_Resource_Product_Type_Configurable_Attribute_Collection {
        return $this->attributes;
    }

    /**
     * @return Array<array{attribute:string, value:string}>
     */
    public function getConfigurationOptionItemValues(): array {
        $items = [];

        foreach($this->attributes as $attr) {
            $prodAttr = $attr->getProductAttribute();
            $model = $prodAttr->getSource();

            $items[] = [
                "attribute" => snakeCaseToCamel($prodAttr->getAttributeCode()),
                "value" => (string)$model->getOptionText($this->child->getData($prodAttr->getAttributeCode())),
            ];
        }

        return $items;
    }
}
