<?php

declare(strict_types=1);

use GraphQL\Deferred;
use GraphQL\Type\Definition\ResolveInfo;

/**
 * Container for deferred loading of product categories, stored in Magento registry.
 * Dependent on current store.
 */
class MageQL_Catalog_Model_Product_Categories {
    public static function registryKey(): string {
        return "mageql_catalog_products_categories";
    }

    public static function instance(): self {
        $children = Mage::registry(static::registryKey());

        if( ! $children) {
            $children = new self();

            Mage::register(static::registryKey(), $children);
        }

        return $children;
    }

    /**
     * Clears this deferred container by removing the global instance.
     */
    public static function clear(): void {
        Mage::unregister(static::registryKey());
    }

    private function __construct() {
        $category = Mage::getModel("catalog/category");

        $category->load(Mage::app()->getStore()->getRootCategoryId());

        $this->rootCategoryPath = $category->getPath() ?: "";
    }

    /**
     * Root category path to be able to filter descendants to avoid categories
     * from unrelated stores.
     *
     * @readonly
     * @var string
     */
    protected $rootCategoryPath;

    /**
     * Map of product id -> [ category id -> categories ].
     *
     * @var Array<int, Array<int, Mage_Catalog_Model_Category>>
     */
    protected $data = [];

    /**
     * @var Array<int>
     */
    protected $productIds = [];

    /**
     * @var Array<string>
     */
    protected $fields = [];

    /**
     * Queues the given product to fetch categories with the supplied list of fields.
     *
     * @param int $productId
     * @param Array<string> $fields
     */
    public function add($productId, array $fields): self {
        if( ! in_array($productId, $this->productIds)) {
            $this->productIds[] = $productId;
        }

        $this->fields = array_unique(array_merge($this->fields, $fields));

        return $this;
    }

    /**
     * Loads the children of the currently queued category ids, will skip
     * loading if no category ids have been queued since previous call.
     */
    public function load(): self {
        if(empty($this->productIds)) {
            // Nothing new to load
            return $this;
        }

        /** @var Mage_Catalog_Model_Resource_Category_Collection|Mage_Catalog_Model_Resource_Category_Flat_Collection */
        $categories = Mage::getModel("catalog/category")->getCollection();
        $mainTable = $categories instanceof Mage_Catalog_Model_Resource_Category_Flat_Collection ? "main_table" : "e";

        foreach($this->fields as $col) {
            $categories->addAttributeToSelect($col);
        }

        $categories->addAttributeToFilter("is_active", "1");
        $categories->addAttributeToFilter("include_in_menu", "1");

        $query = $categories->getSelect();

        $query->join(
            [ "cp" => $categories->getTable("catalog/category_product") ],
            "cp.category_id = $mainTable.entity_id",
            []
        );

        $query->columns([
            "mageql_product_ids" => new Zend_Db_Expr("GROUP_CONCAT(cp.product_id SEPARATOR ',')"),
        ]);

        $query->where("cp.product_id IN (?)", $this->productIds);
        $query->group([ "$mainTable.entity_id" ]);

        // Filter by parent category to avoid getting the wrong data
        $categories->addAttributeToFilter("path", [ "like" => sprintf("%s/%%", $this->rootCategoryPath) ]);

        $categories->addAttributeToSort("cp.position", "asc");
        $categories->addAttributeToSort("name", "asc");

        // Load and reset, merge since we might have loaded more data, ensure
        // that the categories are overwritten if we keep loading new ones since
        // they might have been updated or require different fields
        foreach($categories as $c) {
            foreach(array_map("intval", array_filter(explode(",", $c->getMageqlProductIds()))) as $pid) {
                $this->data[$pid] = $this->data[$pid] ?? [];

                $this->data[$pid][(int)$c->getId()] = $c;
            }
        }

        $this->productIds = [];
        $this->fields = [];

        return $this;
    }

    /**
     * Fetches all category children belonging to $parent if they have been loaded.
     *
     * @param int $productId
     * @return Array<Mage_Catalog_Model_Category>
     */
    public function get($productId): array {
        return $this->data[$productId] ?? [];
    }
}
