<?php

declare(strict_types=1);

namespace Crossroads\Magento\Test\Integration;

use Exception;
use Mage;
use Mage_Catalog_Helper_Category_Flat;
use Mage_Catalog_Helper_Product_Flat;
use Mage_Catalog_Model_Category;
use Mage_Core_Block_Template;
use Mage_Core_Controller_Response_Http;
use Mage_Core_Model_App;
use Mage_Core_Model_Locale;
use Mage_Core_Model_Resource_Setup;
use Mage_Core_Model_Store;
use Mage_Core_Model_Store_Group;
use Mage_Directory_Model_Currency;
use Varien_Profiler;

class MagentoManager {
    const TESTING_STORE = "testing";
    const HOST_VALUE = "https://example.com";

    /**
     * @var ?string
     */
    protected static $varDir = null;
    /**
     * @var string
     */
    protected static $cryptKey = "9c0fb682-ef68-4842-8ed8-0cde0a683610";

    public static function setVarDir(string $path): void {
        self::$varDir = $path;
    }

    public static function getVarDir(): string {
        if( ! self::$varDir) {
            self::$varDir = getcwd()."/var";
        }

        return self::$varDir;
    }

    public static function setCryptKey(string $key): void {
        self::$cryptKey = $key;
    }

    /**
     * Sets up integration testing and populates stores and categories.
     */
    public static function setUp(string $rootDir): void {
        self::setUpBasic($rootDir);

        self::initAdmin();

        self::configure();
        self::createStores();

        self::reset();
    }

    /**
     * Sets up integration testing, but does not populate any fixture-data.
     *
     * @psalm-suppress UnresolvableInclude
     */
    public static function setUpBasic(string $rootDir): void {
        // A fix for PHP serialize issues, see https://wiki.php.net/rfc/precise_float_value
        ini_set("serialize_precision", "-1");

        require_once $rootDir . "/app/Mage.php";

        umask(0);

        if( ! class_exists("Mage_Core_Controller_Response_Http", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Response", "Mage_Core_Controller_Response_Http");
        }

        if( ! class_exists("Mage_Core_Model_Cookie", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Cookie", "Mage_Core_Model_Cookie");
        }

        if( ! class_exists("Mage_Catalog_Model_Product_Image", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Catalog\\Image", "Mage_Catalog_Model_Product_Image");
        }

        if( ! class_exists("Mage_Catalog_Model_Product_Attribute_Backend_Media", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Catalog\\Product\\Attribute\\Media", "Mage_Catalog_Model_Product_Attribute_Backend_Media");
        }

        if( ! class_exists("Mage_Core_Model_Session_Abstract_Varien", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Session", "Mage_Core_Model_Session_Abstract_Varien");
        }

        if( ! class_exists("Varien_Profiler", false)) {
            class_alias("Crossroads\\Magento\\Test\\Integration\\Profiler", "Varien_Profiler");
        }

        Mage::reset();

        gc_collect_cycles();

        Mage::setIsDeveloperMode(true);
        Mage::register("isSecureArea", true);
        Mage::app("admin", "store", [
            "is_installed" => false,
            "config_model" => __NAMESPACE__ . "\\Config",
            "env_config" => self::getEnvConfig(),
        ])->getCacheInstance()->flush();

        self::createVarFolder();

        Mage_Core_Model_Resource_Setup::applyAllUpdates();

        self::initAdmin();

        Mage_Core_Model_Resource_Setup::applyAllDataUpdates();

        self::reset();
    }

    public static function logQueries(): void {
        if( ! Mage::getConfig()) {
            return;
        }

        $var = Mage::getBaseDir("var");
        $connection = Mage::getSingleton("core/resource")->getConnection("core_write");

        if( ! $connection) {
            throw new Exception("Failed to obtain core_write connection");
        }

        $profiler = $connection->getProfiler();
        $queries = $profiler->getQueryProfiles();

        if( ! empty($queries)) {
            $queries = array_map(function($q): string {
                $t = $q->getStartedMicrotime();
                $micro = sprintf("%06d",($t - floor($t)) * 1000000);
                $date = gmdate("Y-m-d\TH:i:s.".$micro."Z", (int)$t);
                $params = implode(", ", array_map(function(?string $v, string $k = ""): string {
                    return ($k ? $k.": " : "").$v;
                }, $q->getQueryParams()));

                return sprintf("[%s] %f: %s (%s)", $date, $q->hasEnded() ? $q->getElapsedSecs() : -1, $q->getQuery(), $params);
            }, $queries);

            if( ! file_exists($var.DS."log")) {
                mkdir($var.DS."log", 0777, true);
            }

            file_put_contents($var.DS."log".DS."query.log", implode("\n", $queries), FILE_APPEND);
        }
    }

    public static function createVarFolder(): void {
        $var = Mage::getBaseDir("var");

        if( ! file_exists($var)) {
            mkdir($var, 0777, true);
        }
    }

    public static function getSystemLog(): string {
        return self::readLogFile("system.log");
    }

    public static function getExceptionLog(): string {
        return self::readLogFile("exception.log");
    }

    public static function readLogFile(string $file): string {
        if( ! Mage::getConfig()) {
            throw new Exception("Magento has not been initialized");
        }

        return file_get_contents(Mage::getBaseDir("log").DS.$file);
    }

    public static function configure(): void {
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        // Clear store-specific config
        $setupModel->deleteConfigData(Mage_Core_Model_Store::XML_PATH_STORE_STORE_NAME);
        $setupModel->deleteConfigData(Mage_Core_Model_Store::XML_PATH_UNSECURE_BASE_URL);
        $setupModel->deleteConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_BASE_URL);

        // General configuration
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_USE_REWRITES, 1);
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_IN_FRONTEND, 0);
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_OFFLOADER_HEADER, "SSL_OFFLOADED");
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_STORE_STORE_NAME, "Default Magento");
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_UNSECURE_BASE_URL, "https://default.test/");
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_BASE_URL, "https://default.test/");
        $setupModel->setConfigData(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE, "sv_SE");
        $setupModel->setConfigData(Mage_Core_Model_Locale::XML_PATH_DEFAULT_COUNTRY, "SE");
        $setupModel->setConfigData(Mage_Core_Model_Locale::XML_PATH_DEFAULT_TIMEZONE, "CET");
        $setupModel->setConfigData(Mage_Directory_Model_Currency::XML_PATH_CURRENCY_BASE, "SEK");
        $setupModel->setConfigData(Mage_Directory_Model_Currency::XML_PATH_CURRENCY_DEFAULT, "SEK");
        $setupModel->setConfigData(Mage_Directory_Model_Currency::XML_PATH_CURRENCY_ALLOW, "SEK");
        $setupModel->setConfigData("general/country/allow", "DE,DK,SE,NO,FI,GB");
        $setupModel->setConfigData("design/head/default_title", "Default Title");
        $setupModel->setConfigData("design/head/title_prefix", "Default Magento |");
        $setupModel->setConfigData("shipping/origin/country_id", "SE");
        $setupModel->setConfigData("web/default/no_route", "cms/index/noRoute");
        $setupModel->setConfigData("web/url/redirect_to_base", 0);
        $setupModel->setConfigData(Mage_Core_Block_Template::XML_PATH_TEMPLATE_ALLOW_SYMLINK, "1");
        $setupModel->setConfigData("dev/log/file", "system.log");
        $setupModel->setConfigData("dev/log/exception_file", "exception.log");
    }

    public static function createStores(): void {
        $website = Mage::getModel("core/website");
        $group = Mage::getModel("core/store_group");
        $store = Mage::getModel("core/store");

        $website->load(static::TESTING_STORE);

        if($website->getDefaultGroupId()) {
            $group->load($website->getDefaultGroupId());
        }

        $store->load(static::TESTING_STORE);

        $website->addData([
            "code" => static::TESTING_STORE,
            "is_default" => 0,
            "name" => "Test Website",
            "sort_order" => 1,
        ])->save();

        $group->addData([
            "name" => "Test Store Group",
            "website_id" => $website->getId(),
        ])->save();

        $store->addData([
            "code" => static::TESTING_STORE,
            "group_id" => $group->getId(),
            "is_active" => 1,
            "name" => "Test Store",
            "sort_order" => 1,
            "website_id" => $website->getId(),
        ])->save();

        self::configureStore($store);

        self::createCategories($store, $group);
    }

    private static function configureStore(Mage_Core_Model_Store $store): void {
        $setupModel = new Mage_Core_Model_Resource_Setup("core_setup");

        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_STORE_STORE_NAME, "Testing Magento", "stores", $store->getId());
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_UNSECURE_BASE_URL, "https://example.com/", "stores", $store->getId());
        $setupModel->setConfigData(Mage_Core_Model_Store::XML_PATH_SECURE_BASE_URL, "https://example.com/", "stores", $store->getId());
        $setupModel->setConfigData("design/head/default_title", "Testing Title", "stores", $store->getId());
        $setupModel->setConfigData("design/head/title_prefix", "Testing Magento |", "stores", $store->getId());

        /*
        $taxClass = Mage::getModel("tax/class");

        $taxClass->load("Test Rate", "class_name");

        $taxClass->addData([
            "class_name" => "Test Rate",
            "class_type" => Mage_Tax_Model_Class::TAX_CLASS_TYPE_PRODUCT,
        ]);

        $taxClass->save();
         */
    }

    /**
     * Since we are not actually reindexing after every small modification we
     * have to load data by id indirectly, and use custom queries to obtain them.
     */
    public static function loadCategoryByAttribute(
        string $attrType,
        string $attr,
        string $value
    ): Mage_Catalog_Model_Category {
        $conn = Mage::getSingleton("core/resource")->getConnection("core_write");

        if( ! $conn) {
            throw new Exception("Failed to obtain core_write connection");
        }

        $id = $conn->query("SELECT v.entity_id FROM catalog_category_entity_$attrType v
            JOIN eav_attribute e ON e.attribute_id = v.attribute_id AND e.entity_type_id = v.entity_type_id
            WHERE e.attribute_code = ? AND v.value = ?", [$attr, $value])->fetchColumn();

        $category = Mage::getModel("catalog/category");

        if($id) {
            $category->load($id);
        }

        return $category;
    }

    public static function createCategories(
        Mage_Core_Model_Store $store,
        Mage_Core_Model_Store_Group $group
    ): void {
        $root = self::loadCategoryByAttribute("varchar", "name", "The Test Root");

        $root->addData([
            "url_key" => "",
            "name" => "The Test Root",
            "atttribute_set_id" => $root->getDefaultAttributeSetId(),
            "description" => "Testing root category",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for Root Category",
            "meta_description" => "Meta description for Root Category",
            "meta_keywords" => "Meta keywords for Root Category",
            "image" => "category_test_root_image.jpg",
            "is_active" => 1,
            "is_anchor" => 0,
            "store_id" => $store->getId(),
            "path" => $root->getPath() ?: "1",
            "parent_id" => 1,
        ]);

        $root->save();

        $group->setRootCategoryId($root->getId())->save();

        $category = self::loadCategoryByAttribute("varchar", "url_key", "test-category");

        $category->addData([
            "url_key" => "test-category",
            "name" => "The Test Category",
            "atttribute_set_id" => $category->getDefaultAttributeSetId(),
            "description" => "Testing A test category",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for Category",
            "meta_description" => "Meta description for Category",
            "meta_keywords" => "Meta keywords for Category",
            "image" => "category_test_image.jpg",
            "is_active" => 1,
            "is_anchor" => 0,
            "path" => $category->getPath() ?: $root->getPath(),
            "parent_id" => $root->getId(),
            "store_id" => $store->getId(),
        ]);

        $category->save();

        $inactiveCategory = self::loadCategoryByAttribute("varchar", "url_key", "test-inactive-category");

        $inactiveCategory->addData([
            "url_key" => "test-inactive-category",
            "name" => "The Inactive Test Category",
            "atttribute_set_id" => $inactiveCategory->getDefaultAttributeSetId(),
            "description" => "Testing an inactive test category",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for Inactive Category",
            "meta_description" => "Meta description for Inactive Category",
            "meta_keywords" => "Meta keywords for Inactive Category",
            "image" => "category_test_inactive_image.jpg",
            "is_active" => 0,
            "is_anchor" => 0,
            "path" => $inactiveCategory->getPath() ?: $root->getPath(),
            "parent_id" => $root->getId(),
            "store_id" => $store->getId(),
        ]);

        $inactiveCategory->save();

        $childCategory = self::loadCategoryByAttribute("varchar", "url_key", "test-child-category");

        $childCategory->addData([
            "url_key" => "test-child-category",
            "name" => "A Child Category",
            "atttribute_set_id" => $childCategory->getDefaultAttributeSetId(),
            "description" => "Test a child category type",
            "display_mode" => "PRODUCTS",
            "meta_title" => "Meta Title for a Child Category",
            "meta_description" => "Meta description for a Child Category",
            "meta_keywords" => "Meta keywords for a Child Category",
            "image" => "category_child_test_image.jpg",
            "is_active" => 1,
            "is_anchor" => 0,
            "path" => $childCategory->getPath() ?: $category->getPath(),
            "parent_id" => $category->getId(),
            "store_id" => $store->getId(),
        ]);

        $childCategory->save();
    }

    public static function reindex(): void {
        self::initAdmin();

        Mage::getSingleton("catalog/product_indexer_eav")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_price")->reindexAll();
        Mage::getSingleton("catalog/indexer_url")->reindexAll();
        Mage::getSingleton("catalog/product_indexer_flat")->reindexAll();
        Mage::getSingleton("catalog/category_indexer_flat")->reindexAll();
        Mage::getSingleton("catalog/category_indexer_product")->reindexAll();
        Mage::getResourceSingleton("cataloginventory/indexer_stock")->reindexAll();
        Mage::getSingleton("catalogsearch/indexer_fulltext")->reindexAll();
    }

    public static function getOptions(): array {
        return [
            "is_installed" => true,
            "config_model" => __NAMESPACE__ . "\\Config",
            "var_dir" => self::getVarDir(),
            "env_config" => self::getEnvConfig(),
        ];
    }

    public static function getEnvConfig(): array {
        return array_merge(
            [
                "RESOURCE_DEFAULT_SETUP" => "mysql://root@localhost/testing_magento",
                "MAGENTO_CRYPT_KEY" => self::$cryptKey,
                "MAGENTO_CACHE_BACKEND" => "Crossroads\\Magento\\Test\\Integration\\Cache\\Backend",
            ],
            getenv()
        );
    }

    public static function initAdmin(): void {
        Mage::reset();

        gc_collect_cycles();

        Mage::setIsDeveloperMode(true);
        Mage::register("isSecureArea", true);
        Mage::init("admin", "store", self::getOptions());
    }

    public static function init(
        string $storeCode = self::TESTING_STORE
    ): Mage_Core_Model_App {
        return Mage::app($storeCode, "store", self::getOptions());
    }

    public static function reset(): void {
        Mage::app("admin", "store", self::getOptions())->getCacheInstance()->flush();

        self::createVarFolder();

        Mage::reset();
        Varien_Profiler::clear();

        gc_collect_cycles();

        Mage::setIsDeveloperMode(true);
    }

    public static function runRequest(
        Request $request,
        string $storeCode = self::TESTING_STORE
    ): Response {
        if( ! $request->getHeader("Host")) {
            $request->setHeader("Host", self::HOST_VALUE);
        }

        $app = self::init($storeCode);

        $app->setRequest($request);
        $app->run([
            "scope_code" => $storeCode,
            "scope_type" => "store",
            "options" => self::getOptions(),
        ]);

        return $app->getResponse();
    }
}
