<?php

declare(strict_types=1);

namespace Crossroads\Magento\Test\Integration\Catalog;

use Mage;
use Mage_Core_Model_Abstract;
use Varien_Image;

/**
 * Catalog product link model
 *
 * @category   Mage
 * @package    Mage_Catalog
 * @author      Magento Core Team <core@magentocommerce.com>
 */
class Image extends Mage_Core_Model_Abstract {
    /**
     * @var ?int
     */
    protected $_width;
    /**
     * @var ?int
     */
    protected $_height;
    /**
     * @var int
     */
    protected $_quality = 90;

    /**
     * @var bool
     */
    protected $_keepAspectRatio  = true;
    /**
     * @var bool
     */
    protected $_keepFrame        = true;
    /**
     * @var bool
     */
    protected $_keepTransparency = true;
    /**
     * @var bool
     */
    protected $_constrainOnly    = false;
    /**
     * @var array{int, int, int}
     */
    protected $_backgroundColor  = array(255, 255, 255);

    /**
     * @var ?string
     */
    protected $_baseFile;
    /**
     * @var ?bool
     */
    protected $_isBaseFilePlaceholder;
    /**
     * @var ?string
     */
    protected $_newFile;
    /**
     * @var ?string
     */
    protected $_destinationSubdir;
    /**
     * @var ?int
     */
    protected $_angle;

    /**
     * @var ?string
     */
    protected $_watermarkFile;
    /**
     * @var ?string
     */
    protected $_watermarkPosition;
    /**
     * @var ?int
     */
    protected $_watermarkWidth;
    /**
     * @var ?int
     */
    protected $_watermarkHeigth;
    /**
     * @var int
     */
    protected $_watermarkImageOpacity = 70;

    /**
     * @return $this
     */
    public function setWidth(?int $width)
    {
        $this->_width = $width;
        return $this;
    }

    public function getWidth(): ?int
    {
        return $this->_width;
    }

    /**
     * @return $this
     */
    public function setHeight(?int $height)
    {
        $this->_height = $height;
        return $this;
    }

    public function getHeight(): ?int
    {
        return $this->_height;
    }

    /**
     * Set image quality, values in percentage from 0 to 100
     *
     * @param int $quality
     * @return $this
     */
    public function setQuality($quality)
    {
        $this->_quality = $quality;
        return $this;
    }

    /**
     * Get image quality
     *
     * @return int
     */
    public function getQuality()
    {
        return $this->_quality;
    }

    /**
     * @return $this
     */
    public function setKeepAspectRatio(bool $keep)
    {
        $this->_keepAspectRatio = $keep;
        return $this;
    }

    /**
     * @return $this
     */
    public function setKeepFrame(bool $keep)
    {
        $this->_keepFrame = $keep;
        return $this;
    }

    /**
     * @return $this
     */
    public function setKeepTransparency(bool $keep)
    {
        $this->_keepTransparency = $keep;
        return $this;
    }

    /**
     * @return $this
     */
    public function setConstrainOnly(bool $flag)
    {
        $this->_constrainOnly = $flag;
        return $this;
    }

    /**
     * @return $this
     */
    public function setBackgroundColor(array $rgbArray)
    {
        $this->_backgroundColor = $rgbArray;
        return $this;
    }

    /**
     * @return $this
     */
    public function setSize(string $size)
    {
        // determine width and height from string
        list($width, $height) = explode('x', strtolower($size), 2);
        foreach (array('width', 'height') as $wh) {
            $$wh  = (int)$$wh;
            if (empty($$wh))
                $$wh = null;
        }

        /**
         * @var ?int $width
         * @var ?int $height
         */

        // set sizes
        $this->setWidth($width)->setHeight($height);

        return $this;
    }

    protected function _checkMemory(string $file = null): bool {
        return true;
    }

    /**
     * Convert array of 3 items (decimal r, g, b) to string of their hex values
     *
     * @param array $rgbArray
     * @return string
     */
    protected function _rgbToString($rgbArray)
    {
        $result = array();
        foreach ($rgbArray as $value) {
            if (null === $value) {
                $result[] = 'null';
            }
            else {
                $result[] = sprintf('%02s', dechex($value));
            }
        }
        return implode($result);
    }

    /**
     * Set filenames for base file and new file
     *
     * @param string $file
     * @return $this
     */
    public function setBaseFile($file)
    {
        $this->_isBaseFilePlaceholder = false;

        if (($file) && (0 !== strpos($file, '/', 0))) {
            $file = '/' . $file;
        }
        $baseDir = Mage::getSingleton('catalog/product_media_config')->getBaseMediaPath();

        if ('/no_selection' == $file) {
            $file = null;
        }
        if (!$file) {
            $file = '/placeholder.jpg';
            $this->_isBaseFilePlaceholder = true;
        }

        $this->_baseFile = $baseDir . $file;

        // build new filename (most important params)
        $path = array(
            Mage::getSingleton('catalog/product_media_config')->getBaseMediaPath(),
            'cache',
            Mage::app()->getStore()->getId(),
            $this->getDestinationSubdir()
        );
        if((!empty($this->_width)) || (!empty($this->_height)))
            $path[] = "{$this->_width}x{$this->_height}";

        // add misk params as a hash
        $miscParams = array(
                ($this->_keepAspectRatio  ? '' : 'non') . 'proportional',
                ($this->_keepFrame        ? '' : 'no')  . 'frame',
                ($this->_keepTransparency ? '' : 'no')  . 'transparency',
                ($this->_constrainOnly ? 'do' : 'not')  . 'constrainonly',
                $this->_rgbToString($this->_backgroundColor),
                'angle' . ($this->_angle ?: ''),
                'quality' . $this->_quality
        );

        // if has watermark add watermark params to hash
        if ($this->getWatermarkFile()) {
            $miscParams[] = $this->getWatermarkFile();
            $miscParams[] = $this->getWatermarkImageOpacity();
            $miscParams[] = $this->getWatermarkPosition();
            $miscParams[] = $this->getWatermarkWidth();
            $miscParams[] = $this->getWatermarkHeigth();
        }

        $path[] = md5(implode('_', $miscParams));

        // append prepared filename
        $this->_newFile = implode('/', $path) . $file; // the $file contains heading slash

        return $this;
    }


    public function getBaseFile(): ?string
    {
        return $this->_baseFile;
    }

    public function getNewFile(): ?string
    {
        return $this->_newFile;
    }

    /**
     * @param mixed $processor
     * @return $this
     */
    public function setImageProcessor($processor)
    {
        return $this;
    }

    /**
     * @return ?Varien_Image
     */
    public function getImageProcessor()
    {
        return null;
    }

    /**
     * @see Varien_Image_Adapter_Abstract
     * @return $this
     */
    public function resize()
    {
        return $this;
    }

    /**
     * @return $this
     */
    public function rotate(int $angle)
    {
        $this->_angle = intval($angle);
        return $this;
    }

    /**
     * Set angle for rotating
     *
     * This func actually affects only the cache filename.
     *
     * @param int $angle
     * @return $this
     */
    public function setAngle($angle)
    {
        $this->_angle = $angle;
        return $this;
    }

    /**
     * Add watermark to image
     * size param in format 100x200
     *
     * @param string $file
     * @param string $position
     * @param array $size
     * @param int $width
     * @param int $heigth
     * @param int $imageOpacity
     * @return $this
     */
    public function setWatermark($file, $position=null, $size=null, $width=null, $heigth=null, $imageOpacity=null)
    {
        if ($file) {
            $this->setWatermarkFile($file);
        } else {
            return $this;
        }

        if ($position)
           $this->setWatermarkPosition($position);
        if ($size)
            $this->setWatermarkSize($size);
        if ($width)
            $this->setWatermarkWidth($width);
        if ($heigth)
            $this->setWatermarkHeigth($heigth);
        if ($imageOpacity)
            $this->setImageOpacity($imageOpacity);

        return $this;
    }

    /**
     * @return $this
     */
    public function saveFile()
    {
        return $this;
    }

    /**
     * @return string
     */
    public function getUrl()
    {
        $baseDir = Mage::getBaseDir('media');
        $path = str_replace($baseDir . DS, "", $this->_newFile ?: "");
        return Mage::getBaseUrl('media') . str_replace(DS, '/', $path);
    }

    public function push(): void
    {
    }

    /**
     * @return $this
     */
    public function setDestinationSubdir(string $dir)
    {
        $this->_destinationSubdir = $dir;
        return $this;
    }

    /**
     * @return ?string
     */
    public function getDestinationSubdir()
    {
        return $this->_destinationSubdir;
    }

    public function isCached(): bool
    {
        return true;
    }

    /**
     * Set watermark file name
     *
     * @param string $file
     * @return $this
     */
    public function setWatermarkFile($file)
    {
        $this->_watermarkFile = $file;
        return $this;
    }

    /**
     * Get watermark file name
     *
     * @return ?string
     */
    public function getWatermarkFile()
    {
        return $this->_watermarkFile;
    }

    /**
     * Set watermark position
     *
     * @param string $position
     * @return $this
     */
    public function setWatermarkPosition($position)
    {
        $this->_watermarkPosition = $position;
        return $this;
    }

    /**
     * Get watermark position
     *
     * @return ?string
     */
    public function getWatermarkPosition()
    {
        return $this->_watermarkPosition;
    }

    /**
     * Set watermark image opacity
     *
     * @param int $imageOpacity
     * @return $this
     */
    public function setWatermarkImageOpacity($imageOpacity)
    {
        $this->_watermarkImageOpacity = $imageOpacity;
        return $this;
    }

    /**
     * Get watermark image opacity
     *
     * @return int
     */
    public function getWatermarkImageOpacity()
    {
        return $this->_watermarkImageOpacity;
    }

    /**
     * Set watermark size
     *
     * @param array|bool $size
     * @return $this
     */
    public function setWatermarkSize($size)
    {
        if(is_array($size)) {
            $this->setWatermarkWidth($size['width'])
                ->setWatermarkHeigth($size['heigth']);
        }

        return $this;
    }

    /**
     * Set watermark width
     *
     * @param int $width
     * @return $this
     */
    public function setWatermarkWidth($width)
    {
        $this->_watermarkWidth = $width;
        return $this;
    }

    /**
     * Get watermark width
     *
     * @return ?int
     */
    public function getWatermarkWidth()
    {
        return $this->_watermarkWidth;
    }

    /**
     * Set watermark heigth
     *
     * @param int $heigth
     * @return $this
     */
    public function setWatermarkHeigth($heigth)
    {
        $this->_watermarkHeigth = $heigth;
        return $this;
    }

    /**
     * Get watermark heigth
     *
     * @return ?int
     */
    public function getWatermarkHeigth()
    {
        return $this->_watermarkHeigth;
    }

    public function clearCache(): void
    {
        return;
    }

    protected function _fileExists(string $filename): bool {
        return true;
    }
}
