<?php

declare(strict_types=1);

namespace Crossroads\Magento\Test\Integration;

use Exception;
use Varien_Profiler;
use Mage_Core_Model_Config;
use Varien_Simplexml_Element;

class Config extends Mage_Core_Model_Config {
    protected $_isLocalConfigLoaded = false;
    protected static $testingConfig = [];

    public static function setConfig(array $config): void {
        self::$testingConfig = $config;
    }

    public static function addConfig(array $config): void {
        self::$testingConfig = mergeArrays(self::$testingConfig, $config);
    }

    public static function unsetConfigPath(string $path): void {
        $list = explode("/", $path);

        self::$testingConfig = excludePath(self::$testingConfig, $list);
    }

    public static function setConfigPath(string $path, ?string $value): void {
        $list = explode("/", $path);

        if(empty($list)) {
            throw new Exception("Empty path supplied");
        }

        $config = [
            "name" => array_pop($list),
            "value" => $value,
        ];

        foreach(array_reverse($list) as $key) {
            $config = [
                "name" => $key,
                "children" => [
                    $config,
                ],
            ];
        }

        self::addConfig([
            $config,
        ]);
    }

    public static function getConfig(): array {
        return self::$testingConfig;
    }

    public function loadBase(): self {
        $etcDir = $this->getOptions()->getEtcDir();
        $files = glob($etcDir.DS.'*.xml');

        foreach($files as $f) {
            // Make sure to skip local xml since we will be using a test db
            // and maybe different configurations depending on test case
            if(stripos($f, "local.xml") === false) {
                if( ! $this->getNode()) {
                    $this->loadFile($f);
                }
                else {
                    $merge = clone $this->_prototype;
                    $merge->loadFile($f);
                    $this->extend($merge);
                }
            }
        }

        foreach(self::$testingConfig as $cfg) {
            extendXml($this->getNode(), $cfg);
        }

        // TODO: Should be able to load a database cache if specified
        $this->_isLocalConfigLoaded = true;

        return $this;
    }

    public function loadModules(): self {
        Varien_Profiler::start('config/load-modules');
        $this->_loadDeclaredModules();

        $this->loadModulesConfiguration(['config.xml'], $this);

        /**
         * Prevent local.xml directives overwriting
         */
        $mergeConfig = clone $this->_prototype;
        if ($this->_isLocalConfigLoaded) {
            // Replaced
            foreach(self::$testingConfig as $cfg) {
                extendXml($this->getNode(), $cfg);
            }
        }

        $this->applyExtends();
        Varien_Profiler::stop('config/load-modules');
        return $this;
    }
}

function mergeArrays(array $target, array $source): array {
    if(empty($source)) {
        return $target;
    }

    foreach($source as $item) {
        $itemName = $item["name"];
        $itemChildren = $item["children"] ?? [];

        foreach($target as $k => $c) {
            if($c["name"] === $itemName) {
                if(array_key_exists("value", $item)) {
                    $c["value"] = $item["value"];

                    unset($c["children"]);
                }
                else {
                    $c["children"] = mergeArrays($c["children"] ?? [], $item["children"]);

                    unset($c["value"]);
                }

                if(array_key_exists("attributes", $item)) {
                    $c["attributes"] = array_merge($target["attributes"] ?? [], $item["attributes"]);
                }

                $target[$k] = $c;

                continue 2;
            }
        }

        array_push($target, $item);
    }

    return $target;
}

function excludePath(array $config, array $path): array {
    if(empty($config)) {
        return $config;
    }

    $key = array_shift($path);

    foreach($config as $k => $c) {
        if($c["name"] === $key) {
            if(empty($path)) {
                unset($config[$k]);
            }
            else if( ! empty($config[$k]["children"])) {
                $children = excludePath($c["children"], $path);

                if(empty($children)) {
                    unset($config[$k]);
                }
                else {
                    $config[$k]["children"] = $children;
                }
            }
        }
    }

    return $config;
}

function extendXml(Varien_Simplexml_Element $target, array $source): void {
    if(empty($source)) {
        return;
    }

    // this will be our new target node
    $targetChild = null;

    // name of the source node
    $sourceName = $source["name"];

    // here we have children of our source node
    $sourceChildren = $source["children"] ?? [];

    if(empty($sourceChildren)) {
        // handle string node
        if(isset($target->$sourceName)) {
            // if target already has children return without regard
            if ($target->$sourceName->hasChildren()) {
                return;
            }

            unset($target->$sourceName);
        }

        $targetChild = $target->addChild($sourceName, $target->xmlentities($source["value"]));
        $targetChild->setParent($target);

        foreach($source["attributes"] ?? [] as $key => $value) {
            $targetChild->addAttribute($key, $target->xmlentities($value));
        }

        return;
    }

    if(isset($target->$sourceName)) {
        $targetChild = $target->$sourceName;
    }

    if(is_null($targetChild)) {
        // if child target is not found create new and descend
        $targetChild = $target->addChild($sourceName);
        $targetChild->setParent($target);

        foreach($source["attributes"] ?? [] as $key => $value) {
            $targetChild->addAttribute($key, $target->xmlentities($value));
        }
    }

    // finally add our source node children to resulting new target node
    foreach($sourceChildren as $childKey => $childNode) {
        extendXml($targetChild, $childNode);
    }
}
