<?php

declare(strict_types=1);

namespace Crossroads\Magento\Test\Integration;

use Mage_Core_Controller_Request_Http;
use Zend_Controller_Exception;

/**
 * @psalm-suppress PropertyNotSetInConstructor
 */
class Request extends Mage_Core_Controller_Request_Http {
    /**
     * @var string|false
     */
    protected $_rawBody = false;
    /**
     * @var Array<string, string>
     */
    protected $headers = [];
    /**
     * @var Array<string, string>
     */
    protected $post = [];
    /**
     * @var Array<string, string>
     */
    protected $query = [];
    /**
     * @var string
     */
    protected $method = "GET";
    /**
     * @var string
     */
    protected $_baseUrl = "";

    public static function normalizeHeaderName(string $name): string {
        return strtoupper(str_replace('-', '_', $name));
    }

    /**
     * @param string $call
     * @param Array<string, string> $headers
     * @param string|false $body
     */
    public function __construct($call, array $headers = [], $body = false) {
        $parts = explode(" ", $call, 2);

        if(count($parts) === 2) {
            $this->method = $parts[0];
            $this->setRequestUri($parts[1]);
        }
        else {
            $this->setRequestUri($parts[0]);
        }

        $this->setPathInfo();

        $this->setHeaders($headers);

        $this->_rawBody = $body;
    }

    /**
     * @param ?string $key
     * @param ?mixed $default
     * @return ?mixed
     */
    public function getQuery($key = null, $default = null) {
        if($key === null) {
            return $this->query;
        }

        return (isset($this->query[$key])) ? $this->query[$key] : $default;
    }

    /**
     * @param array|string $spec
     * @param ?mixed $value
     * @return $this
     */
    public function setQuery($spec, $value = null) {
        if((null === $value) && !is_array($spec)) {
            throw new Zend_Controller_Exception('Invalid value passed to setQuery(); must be either array of values or key/value pair');
        }

        if(is_array($spec)) {
            foreach ($spec as $key => $value) {
                $this->setQuery($key, $value);
            }

            return $this;
        }

        $this->query[$spec] = $value;

        return $this;
    }

    public function getPost($key = null, $default = null) {
        if(null === $key) {
            return $this->post;
        }

        return (isset($this->post[$key])) ? $this->post[$key] : $default;
    }

    public function setPost($spec, $value = null) {
        if((null === $value) && !is_array($spec)) {
            throw new Zend_Controller_Exception('Invalid value passed to setPost(); must be either array of values or key/value pair');
        }

        if(is_array($spec)) {
            foreach ($spec as $key => $value) {
                $this->setPost($key, $value);
            }

            return $this;
        }

        $this->post[$spec] = $value;

        return $this;
    }

    /**
     * @return string
     */
    public function getRawBody() {
        return (string)$this->_rawBody;
    }

    /**
     * @param Array<string, string> $headers
     * @return $this
     */
    public function setHeaders(array $headers) {
        $this->headers = array_combine(array_map(__CLASS__."::normalizeHeaderName", array_keys($headers)), array_values($headers));

        return $this;
    }

    /**
     * @param string $name
     * @param string $value
     * @return $this
     */
    public function setHeader($name, $value) {
        $this->headers[self::normalizeHeaderName($name)] = $value;

        return $this;
    }

    /**
     * @return $this
     */
    public function setMethod(string $method) {
        $this->method = $method;

        return $this;
    }

    public function getMethod(): string {
        return $this->method;
    }

    /**
     * @param string $header
     * @return string|false
     */
    public function getHeader($header) {
        $temp = self::normalizeHeaderName($header);

        return isset($this->headers[$temp]) ? $this->headers[$temp] : false;
    }

    public function _canBeStoreCodeInUrl() {
        // Never
        return false;
    }

    public function isSecure() {
        // We are always secure during test ;)
        return true;
    }
}
