<?php

use \Klarna\XMLRPC\Flags;

class Crossroads_Klarna_Model_Klarna_Invoice extends Mage_Payment_Model_Method_Abstract
{
    protected $_code = "Crossroads_Klarna_Invoice";

    // Properties from mage model
    protected $_isGateway               = true;
    // Required to provide the ability to call the API, without this capture() would not be called
    protected $_isOnline                = true;
    // We do not perform separate reservations
    protected $_canUseForMultishipping  = false;
    protected $_canUseInternal          = true;
    protected $_canAuthorize            = true;
    // We do not allow refunds in Magento, this process is performed outside of Magento
    protected $_canRefund               = false;
    protected $_canCapture              = true;
    protected $_canVoid                 = true;
    protected $_canCancel               = true;
    protected $_canFetchTransactionInfo = true;

    protected $_infoBlockType = "Crossroads_Klarna/info_invoice";

    /**
     * Check whether payment method can be used
     *
     * @param  Mage_Sales_Model_Quote|null
     * @return bool
     */
    public function isAvailable($quote = null) {
        $enabledMethods    = Mage::helper("Crossroads_Klarna")->getEnabledMethods($quote);
        $minimumOrderTotal = Mage::getStoreConfig(Crossroads_Klarna_Helper_Data::CONFIG_PATH_MIN_ORDER_TOTAL, $quote ? $quote->getStoreId() : null);

        return count($enabledMethods) > 0 &&
            (!$quote || $quote->getBaseGrandTotal() >= $minimumOrderTotal) &&
            !$quote->hasVirtualItems() &&
            parent::isAvailable($quote);
    }

    /**
     * Called through `Mage_Sales_Model_Quote_Payment::importData` to set data for the payment method.
     *
     * @param  Varien_Object
     * @return Mage_Payment_Model_Info
     */
    public function assignData($data) {
        $this->getInfoInstance()->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_SSN, $data->getData("socialSecurityNumber"));
        $this->getInfoInstance()->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_PCLASS, $data->getData("klarnaPclass"));

        return parent::assignData($data);
    }

    /**
     * Authorization function for the payment, executed before the order is approved.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  float
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function authorize(Varien_Object $payment, $amount) {
        if( ! $payment->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_SSN)) {
            Mage::throwException("Crossroads_Klarna: Missing required additional information ".Crossroads_Klarna_Helper_Data::FIELD_SSN.".");
        }

        if( ! filter_var($payment->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_PCLASS), FILTER_VALIDATE_INT)) {
            Mage::throwException("Crossroads_Klarna: Missing required additional information ".Crossroads_Klarna_Helper_Data::FIELD_PCLASS.".");
        }

        $helper  = Mage::helper("Crossroads_Klarna/invoice");
        $order   = $payment->getOrder();
        $storeId = $order->getStoreId();

        // $rno is the reservation number
        // $status has three different values:
        //   Flags::ACCEPTED (1): The order is approved and can be activated
        //   Flags::PENDING  (2): The order is pending and cannot be activated yet
        //   Flags::DENIED   (3): The order is denied and should be cancelled
        list($rno, $status) = $helper->reserveAmount($order, $amount);

        $payment->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID, $rno);
        $payment->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_STATUS, $status);
        $payment->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_EID, $helper->getEid(Mage::app()->getStore($storeId)));

        $payment->setTransactionId($rno);
        // We only authorize the transaction here, it is not closed yet
        $payment->setIsTransactionClosed(0);

        switch($status) {
        case Flags::ACCEPTED:
            $payment->setIsTransactionApproved(true);
            break;
        case Flags::PENDING:
            $payment->setIsTransactionPending(true);
            break;
        case Flags::DENIED:
            $order->cancel();
            $order->save();

            throw Crossroads_API_ResponseException::create(403, "Crossroads_Klarna: Payment denied", null, 11004);
        }

        $payment->save();

        return $this;
    }

    /**
     * Captures amount from Klarna from a previous reservation made using authorize.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  float
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function capture(Varien_Object $payment, $amount) {
        $auth = $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH);

        if( ! $auth || ! $auth->getTxnId()) {
            Mage::throwException("Crossroads_Klarna: Failed to obtain authorization id for order '{$payment->getOrder()->getIncrementId()}'.");
        }

        $authTxnId = $auth->getTxnId();
        $helper    = Mage::helper("Crossroads_Klarna/invoice");
        $order     = $payment->getOrder();
        $rno       = $payment->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID);

        if( ! $rno) {
            Mage::throwException("Crossroads_Klarna: Missing required additional information ".Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID." in order {$order->getIncrementId()}.");
        }

        try {
            $invoiceNo = $helper->activateReservation($order, $rno);
        }
        catch(Exception $e) {
            // We cannot use Mage::throwException here since we also want to tie it to the original exception
            throw new Mage_Core_Exception($e->getMessage()." (#".$e->getCode().")", $e->getCode(), $e);
        }

        $payment->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_INVOICE_ID, $invoiceNo);
        $payment->setParentTransactionId($authTxnId);
        $payment->setTransactionId($invoiceNo);
        $payment->setIsTransactionPending(0);
        $payment->setIsTransactionClosed(1);

        return $this;
    }

    /**
     * Cancels a reservation with Klarna.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function cancel(Varien_Object $payment) {
        $auth      = $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH);
        $authTxnId = $auth->getTxnId();

        $helper = Mage::helper("Crossroads_Klarna/invoice");
        $order  = $payment->getOrder();
        $rno    = $payment->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID);

        if( ! $authTxnId) {
            Mage::throwException("Crossroads_Klarna: Failed to obtain authorization id for order '{$order->getIncrementId()}'.");
        }

        if( ! $rno) {
            Mage::throwException("Crossroads_Klarna: Missing required additional information ".Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID." in order {$order->getIncrementId()}.");
        }

        try {
            $invoiceNo = $helper->cancelReservation($order, $rno);
        }
        catch(Exception $e) {
            // We cannot use Mage::throwException here since we also want to tie it to the original exception
            throw new Mage_Core_Exception($e->getMessage()." (#".$e->getCode().")", $e->getCode(), $e);
        }

        $payment->setIsTransactionClosed(1);

        return $this;
    }

    public function void(Varien_Object $payment) {
        return $this->cancel($payment);
    }

    /**
     * Updates transaction information from Klarna.
     *
     * @param  Mage_Payment_Model_Info
     * @param  integer
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function fetchTransactionInfo(Mage_Payment_Model_Info $payment, $transactionId) {
        $order  = $payment->getOrder();
        $status = Mage::helper("Crossroads_Klarna/invoice")->fetchOrderStatus($payment);

        switch($status) {
        case Flags::ACCEPTED:
            $payment->setIsTransactionApproved(true);
            break;
        case Flags::PENDING:
            $payment->setIsTransactionPending(true);
            break;
        case Flags::DENIED:
            $payment->setIsTransactionDenied(true);

            $order->cancel();
            $order->save();
            break;
        }

        $payment->setAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_STATUS, $status);

        $payment->save();

        return parent::fetchTransactionInfo($payment, $transactionId);
    }
}
