<?php

use \Klarna\XMLRPC\Address;
use \Klarna\XMLRPC\Config;
use \Klarna\XMLRPC\Country;
use \Klarna\XMLRPC\Currency;
use \Klarna\XMLRPC\Flags;
use \Klarna\XMLRPC\Klarna;
use \Klarna\XMLRPC\Language;

class Crossroads_Klarna_Helper_Invoice extends Mage_Core_Helper_Abstract
{
    /**
     * Event triggered in reserveAmount after all items have been added to the Klarna object but before
     * shipping has been added.
     *
     * Params:
     *  * order:  Mage_Sales_Model_Order
     *  * klarna: \Klarna\XMLRPC\Klarna
     */
    const EVENT_RESERVE_AMOUNT_POST_ADD_ARTICLES = "crossroads_klarna_invoice_reserve_amount_post_add_articles";

    /**
     * Minimum total which is allowed to query Klarna with.
     */
    const MIN_TOTAL = 0.01;

    /**
     * Executes the supplied callback, catching all exceptions and if they are from the Klarna
     * API they will be converted into Crossroads_API_ResponseException with klarna information
     * since Klarna wants us to forward this information to the user.
     *
     * @param  callable
     * @throws \Klarna\XMLRPC\Exception\KlarnaException|Crossroads_API_ResponseException
     */
    protected static function encodeException($callback) {
        try {
            return $callback();
        }
        catch(Exception $e) {
            if($e->getCode()) {
                // Error-codes are only returned from the Klarna API, all internal errors in the
                // library do not have error-codes
                throw Crossroads_API_ResponseException::create(400, "Klarna error.", [
                    "klarnaErrorCode" => $e->getCode(),
                    "klarnaMessage"   => $e->getMessage()
                ], 11000, $e);
            }

            throw $e;
        }
    }

    /**
     * Splits a locale in the format sv_SE into [se, SE]
     *
     * @param  string
     * @return Array<string>
     */
    protected static function splitLocale($locale) {
        return explode("_", trim(strtr($locale, "-:", "__"), " \t\r\n_-"));
    }

    /**
     * Obtains the country code from a locale.
     *
     * @param  string
     * @return string
     */
    public static function localeToCountry($locale) {
        $parts = self::splitLocale($locale);

        return end($parts);
    }

    /**
     * Obtains the language code from a locale.
     *
     * @param  string
     * @return string
     */
    public static function localeToLanguage($locale) {
        $parts = self::splitLocale($locale);

        return reset($parts);
    }

    /**
     * Retrieves the checkout data for the key paymentMethodData, containing possible PClasses, currently
     * selected PClass, social security number (or date of birth if AT/DE/NL).
     *
     * @param  Mage_Sales_Model_Quote
     * @param  string
     * @return Array<mixed>
     */
    public function fetchCheckoutData($quote, $locale) {
        $lang           = self::localeToLanguage($locale);
        $total          = $quote->getBaseGrandTotal();
        $currency       = $quote->getBaseCurrencyCode();
        $selected       = $quote->getPayment()->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_PCLASS);
        $ssn            = $quote->getPayment()->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_SSN);
        $klarna         = $this->createKlarnaInstance($lang, $currency);
        $enabledMethods = Mage::helper("Crossroads_Klarna")->getEnabledMethods($quote);

        if($total < self::MIN_TOTAL) {
            return [
                "error"                => 2,
                "paymentMethods"       => [],
                "klarnaPclass"         => $selected,
                "socialSecurityNumber" => $ssn
            ];
        }

        try {
            $response = self::encodeException(function() use($klarna, $total, $currency, $locale) {
                return $klarna->checkoutService($total, $currency, $locale);
            });
        }
        // Klarna\XMLRPC throws an InvalidArgumentException if communication fails or amount is equal or below zero
        catch(InvalidArgumentException $e) {
            return [
                "error"                => 1,
                "paymentMethods"       => [],
                "klarnaPclass"         => $selected,
                "socialSecurityNumber" => $ssn
            ];
        }
        // CurlTransport throws a KlarnaException instead for connection issues
        catch(Exception $e) {
            if($e instanceof Crossroads_API_ResponseException) {
                // Rethrow since we need to display an error input
                throw $e;
            }

            // Log these since they might require someone to fix something
            Mage::logException($e);

            return [
                "error"                => 1,
                "paymentMethods"       => [],
                "klarnaPclass"         => $selected,
                "socialSecurityNumber" => $ssn
            ];
        }

        // TODO: Is this really necessary since if a non-200 occurs the response is empty from Klarna's API
        //       causing CurlTransport to crash with a KlarnaException due to parsed response being null.
        if($response->getStatus() < 200 || $response->getStatus() >= 300) {
            throw new Exception("Crossroads_Klarna: Failed call to Klarna API checkoutService:".print_r($response->getData(), true));
        }

        $data = $response->getData();

        // We remove and re-add the payment methods to make sure we get the right data format
        // as well as do not forget any extra data from Klarna.
        $excludePaymentMethods = array_diff_key($data, ["payment_methods" => true]);
        $paymentMethods        = array_key_exists("payment_methods", $data) ? $data["payment_methods"] : [];

        return array_merge($excludePaymentMethods, [
            "error"          => null,
            "klarnaPclass"   => $selected,
            // Exclude pclasses not present in enabled list
            "paymentMethods" => array_filter($paymentMethods, function($pclass) use($enabledMethods) {
                return array_key_exists("pclass_id", $pclass) && in_array($pclass["pclass_id"], $enabledMethods);
            }),
            "socialSecurityNumber" => $ssn
        ]);
    }

    /**
     * Retrieves addresses for the individual id (personal number) of the customer.
     *
     * NOTE: Only applicable to swedish inhabitants
     *
     * @param  string  Locale string for the user
     * @param  string  Currency code
     * @param  string  User personal number
     * @return Array<string>  Address map in the same format as address in checkout API
     */
    public function fetchAddresses($locale, $currency, $invId) {
        $lang   = self::localeToLanguage($locale);
        $klarna = $this->createKlarnaInstance($lang, $currency);

        $response = self::encodeException(function() use ($klarna, $invId) {
            return $klarna->getAddresses($invId);
        });

        return array_map(function($addr) {
            // Unused parts of $addr:
            //
            // * $addr->getCellno();
            // * $addr->getCareof();

            $street = implode(" ", array_filter(array_map("trim", [
                $addr->getStreet(),
                $addr->getHouseNumber(),
                $addr->getHouseExt()
            ])));

            return [
                "prefix"     => null,
                "firstname"  => $addr->getFirstName(),
                "middlename" => null,
                "lastname"   => $addr->getLastName(),
                "suffix"     => null,
                "company"    => $addr->getCompanyName(),
                "street"     => explode("\n", $street),
                "postcode"   => $addr->getZipCode(),
                "city"       => $addr->getCity(),
                "regionId"   => null,
                "countryId"  => $addr->getCountryCode(),
                "telephone"  => $addr->getCellno() ?: $addr->getTelno(),
                "fax"        => null,
                // Nonstandard, not part of Crossroads_API but still very useful, it will be
                // ignored if the address is sent as is to `PUT /checkout`:
                "email"      => $addr->getEmail()
            ];
        }, $response);
    }

    /**
     * Reserves an amount from the user given an order and an amount.
     *
     * NOTE: SSN (personal number/date of birth) and PClass information must be present in the
     *       order payment additional information. See Crossroads_Klarna_Model_Klarna_Invoice::assignData.
     *
     * @param  Mage_Sales_Model_Order
     * @param  double       Total base currency order amount
     * @return Array<mixed> A two-element array containing reservation number and reservation status
     */
    public function reserveAmount(Mage_Sales_Model_Order $order, $amount) {
        $store    = Mage::getModel("core/store")->load($order->getStoreId());
        $locale   = $store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);
        $currency = $order->getBaseCurrencyCode();
        $lang     = self::localeToLanguage($locale);
        $klarna   = $this->createKlarnaInstance($lang, $currency, $store);

        $quote  = $order->getQuote();
        $items  = $quote->getAllVisibleItems();
        $ssn    = $order->getPayment()->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_SSN);
        $pclass = $quote->getPayment()->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_PCLASS);

        if( ! $ssn) {
            // Field is required, and enforced by the observer in Crossroads_Klarna_Helper_Data::checkoutPostVerifyQuote
            throw new Exception("Crossroads_Klarna: Expected required additional information '".Crossroads_Klarna_Helper_Data::FIELD_SSN."' not present in order payment data. This should not happen if payment is triggered properly.");
        }

        if( ! $pclass) {
            throw new Exception("Crossroads_Klarna: Expected required additional information '".Crossroads_Klarna_Helper_Data::FIELD_PCLASS."' not present in order payment data. This should not happen if payment is triggered properly.");
        }

        foreach($items as $item) {
            $rowTotal = $item->getBaseRowTotal()
                  + $item->getBaseTaxAmount()
                  + $item->getBaseHiddenTaxAmount()
                  - $item->getBaseDiscountAmount();

            $klarna->addArticle(
                // Quantity, int
                (double)$item->getQty(),
                // Article number, string
                $item->getSku(),
                // Article name/title, string
                $item->getName(),
                // Price per item, float
                $rowTotal / $item->getQty(),
                // VAT, percentage int
                $item->getTaxPercent(),
                // Discount per item, float
                0, // $item->getDiscountAmount() / $item->getQty(),  // TODO: This seems like it can add additional discounts, investigate
                // Flags, int
                Flags::INC_VAT
            );
        }

        Mage::dispatchEvent(self::EVENT_RESERVE_AMOUNT_POST_ADD_ARTICLES, [
            "order"  => $order,
            "klarna" => $klarna
        ]);

        $shippingTaxRate = ($order->getBaseShippingInclTax() - $order->getBaseShippingTaxAmount()) > 0 ? $order->getBaseShippingTaxAmount() / ($order->getBaseShippingInclTax() - $order->getBaseShippingTaxAmount()) * 100 : 0;

        $klarna->addArticle(
            1,
            $order->getShippingMethod(),
            $order->getShippingDescription(),
            $order->getBaseShippingInclTax() - $order->getBaseShippingDiscountAmount(),
            $shippingTaxRate,
            0,
            Flags::INC_VAT | Flags::IS_SHIPMENT
        );

        $klarna->setAddress(Flags::IS_BILLING, $this->prepareAddress($order->getBillingAddress(), $quote->getCustomerEmail()));
        $klarna->setAddress(Flags::IS_SHIPPING, $this->prepareAddress($order->getShippingAddress(), $quote->getCustomerEmail()));

        // Order id association
        $klarna->setEstoreInfo($order->getIncrementId());

        return self::encodeException(function() use ($klarna, $order, $amount, $pclass) {
            return $klarna->reserveAmount(
                // PNO (Date of birth for AT/DE/NL)
                $order->getPayment()->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_SSN),
                // Gender (Flags)
                null,
                // Amount, float
                $amount,
                // Flags
                Flags::NO_FLAG,
                // Configurable from user input, limited by enabled variants in admin
                $pclass
            );
        });
    }

    /**
     * Converts a magento address object into a Klarna address object.
     *
     * @param  Mage_Sales_Model_Quote_Address
     * @return \Klarna\XMLRPC\Address
     */
    protected function prepareAddress($address, $email) {
        $street = is_array($address->getStreet()) ? implode("\n", $address->getStreet()) : $address->getStreet();

        $kAddr = new Address(
            $email,
            "", // telephone
            $address->getTelephone(), // cellno
            $address->getFirstname(),
            $address->getLastname(),
            "", // careof
            $street,
            $address->getPostcode(),
            $address->getCity(),
            $address->getCountryId(),
            "", // houseNo
            ""  // houseExt
        );

        if($address->getCompany()) {
            $kAddr->setCompanyName($address->getCompany());
        }

        return $kAddr;
    }

    /**
     * Activates an approved Klarna reservation, returning the invoice number.
     *
     * @param  Mage_Sales_Model_Order
     * @param  integer  Klarna reservation number
     * @return string
     */
    public function activateReservation(Mage_Sales_Model_Order $order, $reservationNumber) {
        $store    = Mage::getModel("core/store")->load($order->getStoreId());
        $locale   = $store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);
        $currency = $order->getBaseCurrencyCode();
        $lang     = self::localeToLanguage($locale);
        $klarna   = $this->createKlarnaInstance($lang, $currency, $store);

        $flags = $this->getMode($store) | Flags::RSRV_SEND_BY_EMAIL;

        list($result, $invoiceNo) = $klarna->activate($reservationNumber, "", $flags);

        if(strtoupper($result) !== "OK") {
            Mage::throwException("Attempt to activate Klarna reservation '$reservationNumber' responded with '$result'.");
        }

        return $invoiceNo;
    }

    /**
     * Cancels an approved Klarna reservation.
     *
     * @param  Mage_Sales_Model_Order
     * @param  integer  Klarna reservation number
     * @throws \Klarna\XMLRPC\Exception\KlarnaException
     */
    public function cancelReservation(Mage_Sales_Model_Order $order, $reservationNumber) {
        $store    = Mage::getModel("core/store")->load($order->getStoreId());
        $locale   = $store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);
        $currency = $order->getBaseCurrencyCode();
        $lang     = self::localeToLanguage($locale);
        $klarna   = $this->createKlarnaInstance($lang, $currency, $store);

        $klarna->cancelReservation($reservationNumber);
    }

    /**
     * Fetches order status from Klarna for the specified payment.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @return string  A \Klarna\XMLRPC\Flags flag, either ACCEPTED, PENDING or DENIED
     */
    public function fetchOrderStatus($payment) {
        $order    = $payment->getOrder();
        $store    = Mage::getModel("core/store")->load($order->getStoreId());
        $locale   = $store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);
        $currency = $order->getBaseCurrencyCode();
        $lang     = self::localeToLanguage($locale);
        $klarna   = $this->createKlarnaInstance($lang, $currency, $store);

        return $klarna->checkOrderStatus($payment->getAdditionalInformation(Crossroads_Klarna_Helper_Data::FIELD_RESERVATION_ID));
    }

    /**
     * Used in magento admin to always get a valid store. Falls back to Mage::app()->getStore() if
     * admin store or website is not set.
     *
     * @return Mage_Core_Model_Store
     */
    protected function getStore() {
        if($code = Mage::getSingleton("adminhtml/config_data")->getStore()) {
            return Mage::getModel("core/store")->load($code);
        }
        elseif($code = Mage::getSingleton("adminhtml/config_data")->getWebsite()) {
            $websiteId = Mage::getModel("core/website")->load($code)->getId();

            return Mage::app()->getWebsite($websiteId)->getDefaultStore();
        }

        return Mage::app()->getStore();
    }

    /**
     * Obtains a list of enabled payment classes from the klarna API.
     *
     * NOTE: Only use from admin.
     *
     * @return Array<\Klarna\XMLRPC\PClass>
     * @throws \Klarna\XMLRPC\Exception\KlarnaException
     */
    public function fetchPClasses() {
        $store = $this->getStore();

        $locale   = $store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);
        $currency = $store->getCurrentCurrencyCode();
        $lang     = self::localeToLanguage($locale);
        $country  = $store->getConfig("general/country/default");

        $klarna = $this->createKlarnaInstance($lang, $currency, $store, 5);

        return $klarna->getPClasses($country, $lang, $currency);
    }

    /**
     * Returns true if we have an eid and a secret.
     *
     * @return boolean
     */
    public function hasConfig() {
        $store = $this->getStore();

        return $this->getEid($store) && $this->getSecret($store);
    }

    /**
     * Creates a new instance of \Klarna\XMLRPC\Klarna.
     *
     * @param   string  2-letter ISO language code
     * @param   string  3-letter ISO currency code
     * @param   Mage_Core_Model_Store  If a specific store should be used, otherwise falls back to
     *                                 defined admin store or current store
     * @param   integer Request timeout in seconds
     * @returns \Klarna\XMLRPC\Klarna
     */
    protected function createKlarnaInstance($language, $currency, $store = null, $timeout = null) {
        $config   = new Config();
        $instance = new Klarna();
        $store    = $store ?: $this->getStore();

        $config["eid"]      = $this->getEid($store);
        $config["secret"]   = $this->getSecret($store);
        $config["country"]  = $this->getCountry($store);
        $config["language"] = $this->getLanguage($language);
        $config["currency"] = $this->getCurrency($currency);
        $config["mode"]     = $this->getMode($store);

        if($timeout) {
            $config["timeout"] = $timeout;
        }

        $instance->setConfig($config);

        return $instance;
    }

    public function getEid(Mage_Core_Model_Store $store) {
        return $store->getConfig("payment/Crossroads_Klarna_Invoice/seller_id");
    }

    protected function getSecret(Mage_Core_Model_Store $store) {
        return $store->getConfig("payment/Crossroads_Klarna_Invoice/secret");
    }

    protected function getMode(Mage_Core_Model_Store $store) {
        return $store->getConfig("payment/Crossroads_Klarna_Invoice/testdrive") ? Klarna::BETA : Klarna::LIVE;
    }

    protected function getCountry(Mage_Core_Model_Store $store) {
        $code = $store->getConfig("general/country/default");
        $id   = Country::fromCode($code);

        if($id === null) {
            throw new Exception("Crossroads_Klarna: Invalid country code '$code', could not convert to id using frommagento store country code.");
        }

        return $id;
    }

    protected function getLanguage($languageCode) {
        $id = Language::fromCode($languageCode);

        if($id === null) {
            throw new Exception("Crossroads_Klarna: Invalid language code '$languageCode', could not convert to id using \\Klarna\\XMLRPC\\Language::fromCode.");
        }

        return $id;
    }

    protected function getCurrency($currencyCode) {
        $id = Currency::fromCode($currencyCode);

        if($id === null) {
            throw new Exception("Crossroads_Klarna: Invalid currency code '$currencyCode', could not convert to id using \\Klarna\\XMLRPC\\Currency::fromCode.");
        }

        return $id;
    }
}
