<?php

class Crossroads_ImprovedAttributeValues_Model_Resource_Option_Label extends Mage_Core_Model_Resource_Db_Abstract {
    protected function _construct() {
        $this->_init("improvedattributevalues/option_label", "label_id");
    }

    public function getAttributeLabels($attribute) {
        if( ! $attribute->getAttributeId()) {
            return [];
        }

        $db = $this->getReadConnection();

        if( ! $db) {
            throw new Exception("Failed to obtain read connection");
        }

        $query = $db->select()
            ->from([ "s" => $this->getTable("core/store") ], ["store_id"])
            ->from([ "eao" => $this->getTable("eav/attribute_option") ], ["option_id"])
            ->join([ "eaov" => $this->getTable("eav/attribute_option_value") ], "eaov.option_id = eao.option_id AND eaov.store_id = 0", [ "store_name" => "value" ])
            ->joinLeft([ "l" => $this->getTable("improvedattributevalues/option_label") ], "s.store_id = l.store_id AND eao.option_id = l.option_id", [ "label_id", "title", "media" ])
            ->where("eao.attribute_id = ?", $attribute->getAttributeId())
            ->order(["eaov.value ASC", "s.store_id ASC"]);

        $stmt = $db->query($query);

        $ret = [];

        foreach($stmt as $row) {
            if( ! array_key_exists($row["option_id"], $ret)) {
                $ret[$row["option_id"]] = [
                    "store_label" => $row["store_name"],
                    "stores"      => [],
                ];
            }

            $ret[$row["option_id"]]["stores"][$row["store_id"]] = [
                "label_id" => $row["label_id"],
                "title"    => $row["title"],
                "media"    => $row["media"],
            ];
        }

        return $ret;
    }

    public function getLabels($storeId, $attributeEntityType = null, $attributeCode = null, $includeMedia = false, $onlyList = true) {
        $db = $this->getReadConnection();

        if( ! $db) {
            throw new Exception("Failed to obtain read connection");
        }

        $query = $db->select()
            ->from([ "a" => $this->getTable("eav/attribute") ], [ "attribute_id", "attribute_code" ])
            ->join([ "o" => $this->getTable("eav/attribute_option") ], "o.attribute_id = a.attribute_id", [])
            ->join([ "vg" => $this->getTable("eav/attribute_option_value") ], "vg.option_id = o.option_id AND vg.store_id = 0", ["option_id"])
            ->joinLeft([ "vs" => $this->getTable("eav/attribute_option_value") ], sprintf("vs.option_id = o.option_id AND vs.store_id = %d", $storeId), [])
            ->join([ "lg" => $this->getTable("improvedattributevalues/option_label") ], "lg.option_id = o.option_id AND lg.store_id = 0", [])
            ->joinLeft([ "ls" => $this->getTable("improvedattributevalues/option_label") ], sprintf("ls.option_id = o.option_id AND ls.store_id = %d", $storeId), [])
            ->columns([
                "value" => $db->getIfNullSql(
                    $db->quoteIdentifier("vs.value"),
                    $db->quoteIdentifier("vg.value")
                ),
                "labelId" => $db->getIfNullSql(
                    $db->quoteIdentifier("ls.label_id"),
                    $db->quoteIdentifier("lg.label_id")
                ),
                "title" => $db->getIfNullSql(
                    $db->quoteIdentifier("ls.title"),
                    $db->quoteIdentifier("lg.title")
                ),
                "media" => $db->getIfNullSql(
                    $db->quoteIdentifier("ls.media"),
                    $db->quoteIdentifier("lg.media")
                ),
            ]);

        if($attributeEntityType) {
            /**
             * @psalm-suppress ImplicitToStringCast
             */
            $query->where("a.entity_type_id = ?",
                $db->select()
                ->from([ "t" => $this->getTable("eav/entity_type") ], ["entity_type_id"])
                ->where("t.entity_type_code = ?", $attributeEntityType));
        }

        if($attributeCode) {
            $query->where("a.attribute_code = ?", $attributeCode);
        }

        if($includeMedia) {
            $query->columns([
                "dataUrl" => $db->select()
                    ->from([ "li" => $this->getTable("improvedattributevalues/option_label_index") ], [ "data" ])
                    ->where("li.label_id = labelId")
                    ->limit(1)
            ]);
        }

        if($onlyList) {
            $query->join([ "ca" => $this->getTable("catalog/eav_attribute") ], "ca.attribute_id = a.attribute_id", []);
            $query->where("ca.used_in_product_listing = 1");
        }

        return $db->query($query)->fetchAll();
    }

    public function attributeSaveAfter($event) {
        $db         = Mage::getSingleton('core/resource')->getConnection("core_write");
        $dataObject = $event->getDataObject();
        $options    = $dataObject->getOption();
        $ids        = $dataObject->getIavIconIds();
        $labels     = $dataObject->getIavIconLabels();
        $icons      = $dataObject->getIavIconMedia();
        $toInsert   = [];
        $toDelete   = [];

        if( ! $db) {
            throw new Exception("Failed to obtain write connection");
        }

        foreach($labels ?: [] as $optionId => $label) {
            if( ! array_key_exists($optionId, $options["value"]) || ! empty($options["delete"][$optionId])) {
                continue;
            }

            foreach($label as $storeId => $l) {
                if( ! empty($ids[$optionId][$storeId])) {
                    if(empty($l) && empty($icons[$optionId][$storeId])) {
                        $toDelete[] = $ids[$optionId][$storeId];

                        continue;
                    }

                    $db->update($this->getTable("improvedattributevalues/option_label"), [
                        "option_id"  => $optionId,
                        "store_id"   => $storeId,
                        "title"      => $l,
                        "media"      => empty($icons[$optionId][$storeId]) ? null : $this->unTmpImage($icons[$optionId][$storeId]),
                        "updated_at" => new Zend_Db_Expr("NOW()"),
                    ], [
                        "label_id = ?" => $ids[$optionId][$storeId],
                    ]);
                }
                else if( ! empty($l) || ! empty($icons[$optionId][$storeId])) {
                    $toInsert[] = [
                        "option_id" => $optionId,
                        "store_id"  => $storeId,
                        "title"     => $l,
                        "media"     => empty($icons[$optionId][$storeId]) ? null : $this->unTmpImage($icons[$optionId][$storeId]),
                    ];
                }
            }
        }

        if( ! empty($toInsert)) {
            $db->insertMultiple($this->getTable("improvedattributevalues/option_label"), $toInsert);
        }

        if( ! empty($toDelete)) {
            $db->delete($this->getTable("improvedattributevalues/option_label"), ["label_id IN (?)" => $toDelete]);
        }
    }

    public function unTmpImage($file) {
        if(strlen(trim($file)) === 0) {
            return null;
        }

        $config = Mage::getSingleton("improvedattributevalues/media_config");

        if( ! file_exists($config->getTmpMediaPath($file))) {
            // Nothing to do, file should be uploaded
            return $file;
        }

        $ioObject      = new Varien_Io_File();
        $destDirectory = dirname($config->getMediaPath($file));

        try {
            $ioObject->open(array('path'=>$destDirectory));
        } catch (Exception $e) {
            $ioObject->mkdir($destDirectory, 0777, true);
            $ioObject->open(array('path'=>$destDirectory));
        }

        if (strrpos($file, '.tmp') == strlen($file)-4) {
            $file = substr($file, 0, strlen($file)-4);
        }

        $destFile = $this->_getUniqueFileName($file, $ioObject->dirsep());

        $storageHelper = Mage::helper('core/file_storage_database');

        if ($storageHelper->checkDbUsage()) {
            $storageHelper->renameFile($config->getTmpMediaShortUrl($file), $config->getMediaShortUrl($destFile));

            $ioObject->rm($config->getTmpMediaPath($file));
            $ioObject->rm($config->getMediaPath($destFile));
        } else {
            $ioObject->mv($config->getTmpMediaPath($file), $config->getMediaPath($destFile));
        }

        return str_replace($ioObject->dirsep(), '/', $destFile);
    }

    /**
     * Check whether file to move exists. Getting unique name
     *
     * @param string $file
     * @param string $dirsep
     * @return string
     */
    protected function _getUniqueFileName($file, $dirsep) {
        if(Mage::helper("core/file_storage_database")->checkDbUsage()) {
            return Mage::helper("core/file_storage_database")->getUniqueFilename(
                Mage::getSingleton("catalog/product_media_config")->getBaseMediaUrlAddition(),
                $file
            );
        }

        return dirname($file) . $dirsep . Mage_Core_Model_File_Uploader::getNewFileName(
            Mage::getSingleton("improvedattributevalues/media_config")->getMediaPath($file)
        );
    }

    /**
     * @param $attrIds Array<int>
     */
    public function updateAttributeLabels($attrIds): void {
        $db    = Mage::getSingleton('core/resource')->getConnection("core_write");

        if( ! $db) {
            throw new Exception("Failed to obtain write connection");
        }

        $query = $db->select()
            ->from([ "l" => $this->getTable("improvedattributevalues/option_label") ], ["label_id", "media"])
            ->where("l.media IS NOT NULL")
            ->where("l.label_id IN (?)", $attrIds);

        $this->updateMedia($db, $db->query($query));
    }

    protected function updateMedia($db, $rows): void {
        $config = Mage::getSingleton("improvedattributevalues/media_config");
        $size   = 64;

        foreach($rows as $row) {
            $imgFile = $config->getMediaPath($row["media"]);
            $tmpFile = $config->getTmpMediaPath($row["media"]);

            if(file_exists($imgFile)) {
                $img = new Varien_Image($imgFile);

                $img->constrainOnly(true);
                $img->keepAspectRatio(true);
                $img->keepTransparency(true);
                $img->resize($size, $size);

                $img->save($tmpFile);

                $data = file_get_contents($tmpFile);
                $meta = getimagesize($tmpFile);
                $uri  = sprintf("data:%s;base64,%s", $meta["mime"], base64_encode($data));

                $db->query(sprintf("INSERT INTO %s (label_id, data) VALUES (?, ?) ON DUPLICATE KEY UPDATE data = ?, updated_at = NOW()", $this->getTable("improvedattributevalues/option_label_index")), [
                    $row["label_id"],
                    $uri,
                    $uri,
                ]);
            }
        }
    }

    public function reindexAll() {
        $db    = Mage::getSingleton('core/resource')->getConnection("core_write");

        if( ! $db) {
            throw new Exception("Failed to obtain write connection");
        }

        $query = $db->select()
            ->from([ "l" => $this->getTable("improvedattributevalues/option_label") ], ["label_id", "media"])
            ->where("l.media IS NOT NULL");

        $this->updateMedia($db, $db->query($query));
    }
}
