<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use function MageQL\snakeCaseToCamel;
use function MageQL\camelToSnakeCase;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;
use MageQL\Type\InterfaceBuilder;
use MageQL\Type\ObjectBuilder;

/**
 * @psalm-type Attribute array{
 *   code:string,
 *   backend_type:string,
 *   input:string,
 *   label:string,
 *   required:bool,
 *   not_system:bool,
 *   attribute_set:Array<string>,
 *   attribute_set_type:Array<string>,
 *   apply_to:Array<string>,
 *   show_in_detail:bool,
 *   show_in_list:bool,
 *   filterable:bool
 * }
 */
class Crossroads_ImprovedAttributeValues_Model_Schema extends MageQL_Core_Model_Schema_Abstract {
    /**
     * @var MageQL_Catalog_Model_Attributes_Product
     */
    protected $config;

    /**
     * @var ?Array<array{ title: string, media: ?string, value: string, option_id: string }>
     */
    protected $labels = null;

    public function __construct(array $args) {
        parent::__construct($args);

        MageQL_Catalog_Model_Product::$LIST_ATTRIBUTE_FIELDS[] = "attrLabels";

        $this->config = Mage::getSingleton("mageql_catalog/attributes_product");
    }

    /**
     * @return Array<array{ title: string, media: ?string, value: string, option_id: string }>
     */
    public function getAttrLabels(): array {
        if($this->labels === null) {
            $this->labels = Mage::getResourceModel("improvedattributevalues/option_label")
                ->getLabels(Mage::app()->getStore()->getId(), "catalog_product");
        }

        return $this->labels;
    }

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder {
        switch($typeName) {
        case "AttributeDescription":
            return $this->object("Description for a detailed product attribute.");

        case "AttributeDescriptions":
            return $this->object("Descriptions available for a detailed product.");

        case "AttributeLabel":
            return $this->object("An attribute label");

        case "AttributeLabelsDetail":
            return $this->object("Label types available for a detailed product.")
                ->setResolveField("MageQL\\forwardResolver");

        case "AttributeLabelsList":
            return $this->object("Label types available for a list product.")
                ->setResolveField("MageQL\\forwardResolver");

        default:
            return null;
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array {
        switch($typeName) {
        case "AttributeDescription":
            return [
                "title" => $this->field("String!", "The attribute value."),
                "description" => $this->field("String!", "The description for the attribute value."),
                "icon" => $this->field("String", "Any image link for the description."),
            ];

        case "AttributeDescriptions":
            return array_map([$this, "createDescription"], array_filter($this->config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL), [$this, "hasLabels"]));

        case "AttributeLabel":
            return [
                "title" => $this->field("String!", "The title for the label."),
                "icon" => $this->field("String", "Any icon link for the label."),
            ];

        case "AttributeLabelsDetail":
            return array_map([$this, "createLabels"], array_filter($this->config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_DETAIL), [$this, "hasLabels"]));

        case "AttributeLabelsList":
            return array_map([$this, "createLabels"], array_filter($this->config->getAttributesByArea(MageQL_Catalog_Model_Attributes_Abstract::AREA_LIST), [$this, "hasLabels"]));

        case "ListProduct":
            return [
                "attrLabels" => $this->field("AttributeLabelsList!", "Map of attributes with metadata for the product")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        case "ProductDetail":
            return [
                "attrLabels" => $this->field("AttributeLabelsDetail!", "Map of attributes with metadata for the product")
                    ->setResolver("MageQL\\forwardResolver"),
                "attrDescriptions" => $this->field("AttributeDescriptions!", "Map of attributes with descriptions for the values")
                    ->setResolver([$this, "resolveDescriptions"]),
            ];

        case "ProductOption":
            return [
                "attrLabels" => $this->field("AttributeLabelsList!", "Map of attributes with metadata for the product option")
                    ->setResolver("MageQL\\forwardResolver"),
            ];

        default:
            return [];
        }
    }

    /**
     * @param Attribute $attr
     */
    public function createLabels(array $attr): FieldBuilder {
        $isMulti = $attr["input"] === "multiselect";
        $required = $attr["required"] && ! $attr["not_system"] ? "!" : "";
        $type = $isMulti ? "[AttributeLabel!]!" : "AttributeLabel$required";

        return $this->field($type, sprintf("Attribute %s for %s", $isMulti ? "label" : "labels", $attr["label"]))
            ->setResolver($isMulti ?
                [$this, "resolveLabels"] :
                [$this, "resolveLabel"]
            );
    }

    /**
     * @param Attribute $attr
     */
    public function createDescription(array $attr): FieldBuilder {
        $isMulti = $attr["input"] === "multiselect";
        $required = $attr["required"] && ! $attr["not_system"] ? "!" : "";
        $type = $isMulti ? "[AttributeDescription!]!" : "AttributeDescription$required";

        return $this->field($type, sprintf("Attribute %s for %s", $isMulti ? "description" : "descriptions", $attr["label"]))
            ->setResolver([$this, "resolveDescription"]);
    }

    /**
     * @param Attribute $attr
     */
    public function hasLabels(array $attr): bool {
        return in_array($attr["input"], ["select", "multiselect"]);
    }

    public function resolveLabels(
        Mage_Catalog_Model_Product $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        $labels = [];
        $image = Mage::helper("improvedattributevalues/image");
        $attributeCode = camelToSnakeCase($info->fieldName);
        $value = $src->getData($attributeCode);

        foreach($this->getAttrLabels() as $a) {
            if($a["attribute_code"] === $attributeCode && $a["option_id"] === $value) {
                $labels[] = [
                    "title" => $value,
                    "icon" => $a["media"] ? $image->resizeIcon($a["media"]) : null,
                ];
            }
        }

        return $labels;
    }

    public function resolveLabel(
        Mage_Catalog_Model_Product $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): ?array {
        $attributeCode = camelToSnakeCase($info->fieldName);
        $value = $src->getData($attributeCode);
        $image = Mage::helper("improvedattributevalues/image");

        foreach($this->getAttrLabels() as $a) {
            if($a["attribute_code"] === $attributeCode && $a["option_id"] === $value) {
                return [
                    "title" => $a["title"],
                    "icon" => $a["media"] ? $image->resizeIcon($a["media"]) : null,
                ];
            }
        }

        return null;
    }

    public function resolveDescriptions(
        Mage_Catalog_Model_Product $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): array {
        return Mage::getResourceModel("improvedattributevalues/option_description")
            ->getProductDescriptions($src, $ctx->getStore()->getId());
    }

    public function resolveDescription(
        array $src,
        array $unusedArgs,
        MageQL_Core_Model_Context $ctx,
        ResolveInfo $info
    ): ?array {
        $attributeCode = camelToSnakeCase($info->fieldName);

        if(empty($src[$attributeCode])) {
            return null;
        }

        return $src[$attributeCode];
    }
}
