<?php

declare(strict_types=1);

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;

class Crossroads_Goyada_Model_Schema extends MageQL_Core_Model_Schema_Abstract {

    const SUCCESS = "success";

    public function getTypeBuilder(string $typeName, Registry $registry): ?AbstractBuilder
    {
        switch ($typeName) {
            case "GoyadaVoucher":
                return $this->object("An issued voucher")
                                ->setResolveField("GraphQL\Executor\Executor::defaultFieldResolver");

            case "GoyadaVoucherData":
                return $this->object("Data for an issued voucher")
                                ->setResolveField("GraphQL\Executor\Executor::defaultFieldResolver");

            default:
                return null;
        }
    }

    public function getTypeFields(string $typeName, Registry $registry): array
    {
        switch ($typeName) {
            case "OrderItem":
                return [
                    "goyadaVouchers" => $this->field("[GoyadaVoucher!]", "List of voucher codes, if any.")
                            ->setResolver("Crossroads_Goyada_Model_Schema::resolveGoyadaVouchers"),
                ];

            case "GoyadaVoucher":
                return [
                    "code" => $this->field("String!", "Voucher code"),
                    "data" => $this->field("[GoyadaVoucherData!]", "Voucher data"),
                    "image_url" => $this->field("String", "URL to voucher code as image (barcode, QRcode, etc), if any"),
                    "expiryDate" => $this->field("String", "Expiry date, if any")->setResolver("Crossroads_Goyada_Model_Schema::resolveDate"),
                ];

            case "GoyadaVoucherData":
                return [
                    "key" => $this->field("String!", "Voucher data key"),
                    "val" => $this->field("String", "Voucher data value"),
                ];

            default:
                return [];
        }
    }

    public static function resolveDate($param): ?string
    {
        if (!empty($param["expiryDate"])) {
            $time = strtotime($param["expiryDate"]);
            if (!empty($time)) {
                return date("Y-m-d", $time);
            }
        }
        return null;
    }

    /**
     * Load all vouchers for specified item
     * 
     * @param Mage_Sales_Model_Order_Item
     * @return ?Array<array{ code: string, data: ?array, image_url: ?string, expiryDate: ?string }>
     */
    public static function resolveGoyadaVouchers(Mage_Sales_Model_Order_Item $item): ?array
    {
        $vouchers = null;

        $collection = Mage::getModel('Crossroads_Goyada/voucher')->getCollection()->addItemFilter($item->getId())->addCodeFilter();
        $baseUrl = $item->getOrder()->getStore()->getBaseUrl();
        $url = Crossroads_Goyada_Helper_Data::GOYADA_IMAGE_URL;

        if (!empty($collection) && count($collection) > 0) {
            $vouchers = [];
            foreach ($collection as $voucher) {

                $voucherData = $voucher->getVoucherData();
                if (!empty($voucherData)) {
                    $voucherData = json_decode($voucherData, true);
                }
                if (empty($voucherData)) {
                    $voucherData = null;
                }

                $vouchers[] = [
                    "code" => $voucher->getVoucherCode(),
                    "data" => $voucherData,
                    "image_url" => $voucher->getImageHash() ? sprintf("%s%s%s.png", $baseUrl, $url, $voucher->getImageHash()) : null,
                    "expiryDate" => $voucher->getValidTo()
                ];
            }
        }

        return $vouchers;
    }

}
