<?php

class Crossroads_Goyada_Helper_Data extends Mage_Core_Helper_Abstract {

    // Module config
    const LOG_FILENAME            = "goyada.log";
    const CONFIG_TEST             = "goyada/general/test";
    const CONFIG_CODE_IMAGE_URL   = "goyada/email/code_image_url";
    const CONFIG_API_URL          = "goyada/general/api_url";
    const CONFIG_API_TEST_URL     = "goyada/general/api_test_url"; // https://giftcardproxy-public-stage-stage.goyadacloud.com/gcp

    // Product attributes
    const GOYADA_PRODUCT          = "is_goyada_product";
    const GOYADA_PRODUCT_ID       = "goyada_product_id";
    const GOYADA_API_SERVICE      = "goyada_service"; // fashioncheque
    const GOYADA_API_TEST_SERVICE = "goyada_test_service"; // fashionchequeTest
    const GOYADA_EMAIL_TEMPLATE   = "goyada_email_template";
    const GOYADA_EMAIL_BCC        = "goyada_email_bcc";
    const GOYADA_EMAIL_TEXT       = "gift_card_text";
    const GOYADA_EMAIL_TERMS      = "gift_card_terms";
    const GOYADA_IMAGE_TYPE       = "serial_code_image_type";
    const GOYADA_IMAGE_PARAMS     = "serial_code_image_params";

    const GOYADA_API_DATA          = "goyada_api_data";

    // Magento access control
    const GOYADA_ACL_ORDER_ITEM_LIST = "admin/sales/goyada_items";
    const GOYADA_ACL_ORDER_ITEM_ISSUE = "admin/sales/order/actions/goyada_issue";
    const GOYADA_ACL_ORDER_ITEM_EMAIL = "admin/sales/order/actions/goyada_email";

    // Other value keys for product
    const GOYADA_API_URL          = "goyada_api_url";

    // Email setup
    const USE_EMAIL_QUEUE = true; // Magento >= 1.9 uses this
    const EMAIL_EVENT_TYPE = "goyada";

    // URLs to controllers that fetches code image by hash (barcode, QRcode, etc)
    const CODE_IMAGE_URLS = [
        0 => "api" . DS . "goyada_image" . DS . "get" . DS . "hash" . DS,
        1 => "goyada" . DS . "image" . DS . "get" . DS . "hash" . DS
    ];

    public static $issueVoucherOrderStatuses = [
        Mage_Sales_Model_Order::STATE_PROCESSING,
        Mage_Sales_Model_Order::STATE_COMPLETE
    ];

    protected $voucherItemListCache = [];
    protected $currentStoreId = null;

    /**
     * Extract Goyada specific product data
     *
     * @param  Mage_Catalog_Model_Product
     * @param  Mage_Sales_Model_Order
     * @return array
     */
    public function getProductConfig($store, $product, $item = null)
    {
        $test = $store->getConfig(self::CONFIG_TEST) ? true : false;

        if ($test) {
            $url = $store->getConfig(self::CONFIG_API_TEST_URL);
            $url .= "?service=" . $product->getData(self::GOYADA_API_TEST_SERVICE);
        } else {
            $url = $store->getConfig(self::CONFIG_API_URL);
            $url .= "?service=" . $product->getData(self::GOYADA_API_SERVICE);
        }

        $bcc = $product->getData(self::GOYADA_EMAIL_BCC);
        $imgParams = $product->getData(self::GOYADA_IMAGE_PARAMS);

        return [
            "test" => $test,
            self::GOYADA_API_SERVICE => $product->getData($test ? self::GOYADA_API_TEST_SERVICE : self::GOYADA_API_SERVICE),
            self::GOYADA_PRODUCT_ID => $product->getData(self::GOYADA_PRODUCT_ID),
            self::GOYADA_API_URL => $url,
            self::GOYADA_API_DATA => $this->getAPIData($store, $product, $item),
            self::GOYADA_EMAIL_TEMPLATE => $product->getData(self::GOYADA_EMAIL_TEMPLATE),
            self::GOYADA_EMAIL_BCC => empty($bcc) ? [] : array_filter(array_map("trim", explode(",", $bcc))),
            self::GOYADA_EMAIL_TEXT => $product->getData(self::GOYADA_EMAIL_TEXT),
            self::GOYADA_EMAIL_TERMS => $product->getData(self::GOYADA_EMAIL_TERMS),
            self::GOYADA_IMAGE_TYPE => $product->getData(self::GOYADA_IMAGE_TYPE),
            self::GOYADA_IMAGE_PARAMS => empty($imgParams) ? [] : json_decode($imgParams, true)
        ];
    }

    /**
     * fetch API data and expand if needed
     *
     * @param  Mage_Sales_Model_Order
     * @param  Mage_Catalog_Model_Product
     * @return array
     */
    public function getAPIData($store, $product, $item = null)
    {
        $APIData = $product->getData(self::GOYADA_API_DATA);
        if (!empty($APIData) && !is_array($APIData)) {
            $APIData = unserialize($APIData);
        }

        if (!empty($APIData)) {
            foreach (array_keys($APIData) as $rowIndex) {
                if ($item && stripos($APIData[$rowIndex]["val"], "%price") === 0) {
                    $APIData[$rowIndex]["val"] = $item->getPriceInclTax();
                } elseif ($item && stripos($APIData[$rowIndex]["val"], "%multiplier=") === 0) {
                    if (strlen($APIData[$rowIndex]["val"]) > 12) {
                        $multiplier = floatval(substr($APIData[$rowIndex]["val"], 12));
                        if ($multiplier > 0) {
                            $APIData[$rowIndex]["val"] = $multiplier * $item->getPriceInclTax();
                        }
                    }
                } elseif (stripos($APIData[$rowIndex]["val"], "%currency") === 0) {
                    $APIData[$rowIndex]["val"] = $store->getConfig("currency/options/base");
                }
                if (stripos($APIData[$rowIndex]["val"], "%") === 0) {
                    $APIData[$rowIndex]["val"] = str_replace("%", "", $APIData[$rowIndex]["val"]);
                }
            }
        }

        return $APIData;

    }

    /**
     * Get data for all Goyada items in specified order
     *
     * @param  Mage_Sales_Model_Order->id
     * @return array
     */
    public function getOrderData($orderId)
    {
        $sqlQuery = "
            SELECT
                o.entity_id,
                o.increment_id,
                o.state,
                o.`status`,
                p.method AS payment_method,
                i.product_id,
                e.sku
            FROM sales_flat_order o
            JOIN sales_flat_order_payment p ON p.parent_id = o.entity_id
            JOIN sales_flat_order_item i ON i.order_id = o.entity_id
            JOIN catalog_product_entity e ON e.entity_id = i.product_id
            JOIN catalog_product_entity_int ei ON ei.entity_id = e.entity_id AND ei.store_id = 0
            JOIN eav_attribute a ON a.attribute_id = ei.attribute_id
            WHERE o.entity_id = :entityId AND a.attribute_code = :attributeCode AND ei.`value` = :attributeValue
        ";
        $params = [
            "entityId" => $orderId,
            "attributeCode" => "is_goyada_product",
            "attributeValue" => 1
        ];
        return Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $params);
    }

    /**
     * Get item data prepared for voucher purchase
     *
     * @param  Mage_Sales_Model_Order
     * @return array
     */
    public function getVoucherItemList($order)
    {
        $orderId = $order->getId();
        if (!array_key_exists($orderId, $this->voucherItemListCache)) {

            $itemList = [];
            $parentList = [];
            $storeId = $order->getStoreId();
            $store = Mage::app()->getStore($storeId);
            $orderItems = $order->getItemsCollection()->getItems();

            // Let's find out if order contains any voucher products.
            foreach ($orderItems as $item) {
                $itemId = intval($item->getId());

                // Search for vouchers
                if ($item->getProductType() == "virtual") {
                    $productId = intval($item->getProductId());
                    $product = Mage::getModel("catalog/product")->setStoreId($storeId)->load($productId);
                    if ($product->getData(self::GOYADA_PRODUCT)) {

                        // Found voucher product
                        $parentItemId = intval($item->getParentItemId());
                        if (!empty($parentItemId)) {
                            // Product is child, save parent and look up data for it later
                            $itemList[$itemId]["parent_item_id"] = $parentItemId;
                            $parentList[$parentItemId] = $itemId;
                        } else {
                            $itemList[$itemId]["parent_item_id"] = null;
                        }

                        // Fetch all Goyada specific attribute values
                        $itemList[$itemId]["product_config"] = $this->getProductConfig($store, $product, $item);

                        // Set data as if voucher have fixed value, change later if not so
                        $itemList[$itemId]["item_id"] = $itemId;
                        $itemList[$itemId]["product_id"] = $productId;
                        $itemList[$itemId]["qty"] = intval($item->getQtyOrdered());
                        $itemList[$itemId]["sku"] = $product->getSku();
                        $itemList[$itemId]["name"] = $product->getName();
                    }
                }
            }

            // Search for parent product
            if (!empty($parentList)) {
                foreach ($orderItems as $item) {
                    $itemId = intval($item->getId());
                    if (array_key_exists($itemId, $parentList)) {
                        $childId = $parentList[$itemId];
                        // Extract parent qty ordered
                        $parentQty = intval($item->getQtyOrdered());
                        if ($parentQty > 0) {
                            // This takes care of bundles where customer orders `x` amount of bundles where qty for child is `y`.
                            // The actual amount of vouchers to purchase is then calculated as `y` multiplied by `x`.
                            $itemList[$childId]["qty"] *= $parentQty;
                        }
                    }
                }
            }

            $this->voucherItemListCache[$orderId] = $itemList;
        }

        return $this->voucherItemListCache[$orderId];
    }

    /**
     * Extract image information from product
     *
     * @param  Mage_Catalog_Model_Product->id
     * @param  Mage_Core_Model_Store->id
     * @return array
     */
    public function getImagesForEmail($productId, $storeId = 0)
    {
        $retval = array("top" => null, "bottom" => null);
        $sqlQuery = "SELECT T1.value, T2.store_id, T2.label FROM catalog_product_entity_media_gallery T1 JOIN catalog_product_entity_media_gallery_value T2 ON T2.value_id = T1.value_id WHERE T1.entity_id = ? AND T2.label != ''";
        $images = Mage::getSingleton("core/resource")->getConnection("core_read")->fetchAll($sqlQuery, $productId);
        if ($images) {
            foreach ($images as $image) {
                $labelData = explode("_", $image["label"]);
                if (!empty($labelData)) {
                    $label = $labelData[0];
                    if (array_key_exists($label, $retval)) {
                        if ($image["store_id"] == $storeId || ($image["store_id"] == 0 && empty($retval[$label]))) {
                            if ($label == "top") {
                                $retval["background-color"] = empty($labelData[1]) ? "ffffff" : $labelData[1];
                                $retval["border-color"] = empty($labelData[2]) ? $retval["background-color"] : $labelData[2];
                                $retval["font-color"] = empty($labelData[3]) ? "000000" : $labelData[3];
                            } elseif ($label == "bottom") {
                                $retval["font-color"] = empty($labelData[1]) ? "000000" : $labelData[1];
                            }
                            $retval[$label] = Mage::getStoreConfig("web/unsecure/base_url", $storeId) . "media/catalog/product" . $image["value"];
                        }
                    }
                }
            }
        }

        return $retval;
    }

    /**
     * Send email to customer, for all product in order containing voucher codes.
     *
     * @param  Mage_Sales_Model_Order
     * @return void
     */
    public function sendCustomerEmailForOrder($order)
    {
        $itemList = $this->getVoucherItemList($order);

        foreach ($itemList as $itemData) {
            $this->sendCustomerEmailForItem($order, $itemData);
        }
    }

    /**
     * Send email to customer, for specified product containing voucher codes.
     *
     * https://stackoverflow.com/questions/22039156/cant-pass-variable-in-block-directive-from-transactional-email-template
     * {{block type="mymodule/sales_order_email_description" template="email/order/description.phtml" order=$order}}
     * http://excellencemagentoblog.com/blog/2011/11/25/magento-advanced-transactional-email-templates/
     * https://magento.stackexchange.com/questions/3769/pass-data-to-getchildhtml-or-call-method-on-child-block
     * https://magento.stackexchange.com/questions/141586/how-to-use-custom-variables-in-phtml-file-for-email
     *
     * @param  Mage_Sales_Model_Order
     * @param  array
     * @param  ?array|null
     * @return void
     */
    public function sendCustomerEmailForItem($order, $itemData, $codes = null)
    {
        $incrementId = $order->getIncrementId();
        $storeId = $order->getStoreId();

        $customerEmail = $order->getCustomerEmail();
        if (empty($customerEmail)) {
            Mage::helper("Integration")->sendAdminEmail("Goyada order missing email address", "Order {$incrementId} is missing customer email address.");
            Mage::throwException("Order {$incrementId} is missing customer email address.");
        }

        $customerName = $order->getBillingAddress()->getName() ?: ($order->getShippingAddress()->getName() ?: 0);
        if (empty($customerName)) {
            Mage::helper("Integration")->sendAdminEmail("Goyada order missing customer name", "Order {$incrementId} is missing customer name.");
            Mage::throwException("Order {$incrementId} is missing customer name.");
        }

        if (empty($codes)) {
            $codes = Mage::getModel("Crossroads_Goyada/voucher")->getCollection()->addItemFilter($itemData["item_id"])->addCodeFilter();
        }

        $codeQty = count($codes);
        if (empty($codeQty)) {
            Mage::throwException("Order {$incrementId} is missing vouchers for product [{$itemData["sku"]}].");
        }

        $mailTemplate = $itemData["product_config"][self::GOYADA_EMAIL_TEMPLATE];
        if (is_numeric($mailTemplate)) {
            $emailTemplate = Mage::getModel("core/email_template")->load($mailTemplate);
        } else {
            $emailTemplate = Mage::getModel("core/email_template")->loadDefault($mailTemplate);
        }
        $emailTemplate->setSenderName(Mage::getStoreConfig("trans_email/ident_sales/name", $storeId));
        $emailTemplate->setSenderEmail(Mage::getStoreConfig("trans_email/ident_sales/email", $storeId));

        if (!empty($itemData["product_config"][self::GOYADA_EMAIL_BCC])) {
            $emailTemplate->addBcc($itemData["product_config"][self::GOYADA_EMAIL_BCC]);
        }

        $emailTemplateVariables = [
            "order" => $order,
            "product" => Mage::getModel("catalog/product")->setStoreId($storeId)->load($itemData["product_id"]),
            "codes" => $codes,
            "store_name" => Mage::getStoreConfig("general/store_information/name", $storeId),
            self::GOYADA_EMAIL_TEXT => $itemData["product_config"][self::GOYADA_EMAIL_TEXT],
            self::GOYADA_EMAIL_TERMS => $itemData["product_config"][self::GOYADA_EMAIL_TERMS]
        ];

        $imageData = $this->getImagesForEmail($itemData["product_id"], $storeId);
        if (!empty($imageData)) {
            $emailTemplateVariables["image_top"] = $imageData["top"] ?? "";
            $emailTemplateVariables["image_bottom"] = $imageData["bottom"] ?? "";
            $emailTemplateVariables["background-color"] = $imageData["background-color"] ?? "";
            $emailTemplateVariables["border-color"] = $imageData["border-color"] ?? "";
            $emailTemplateVariables["font-color"] = $imageData["font-color"] ?? "";
        }

        if (self::USE_EMAIL_QUEUE) {
            /** @var $emailQueue Mage_Core_Model_Email_Queue */
            $emailQueue = Mage::getModel("core/email_queue");
            $emailQueue->setEventType(self::EMAIL_EVENT_TYPE)
                    ->setEntityType(Mage_Sales_Model_Order::ENTITY)
                    ->setEntityId($order->getId())
                    ->setIsForceCheck(false);
            $emailTemplate->setQueue($emailQueue);
        }

        $emailTemplate->send($customerEmail, $customerName, $emailTemplateVariables);

        // Mark each code as sent
        foreach ($codes as $code) {
            $code->setData("sent_at", date("Y-m-d H:i:s"))->save();
        }

        // Prepare comment for order about this sending of code(s)
        $codeWord = $codeQty > 1 ? "codes" : "code";
        $this->addRegisterMessage("goyada_order_comment", "- Successfully sent {$codeQty} {$codeWord} for [{$itemData["sku"]}] {$itemData["name"]}\n");
        $this->log("Successfully sent {$codeQty} {$codeWord} for [{$itemData["sku"]}] '{$itemData["name"]}' in order {$incrementId}");
    }

    /**
     * Create image if code is set to show that
     *
     * @param  Crossroads_Goyada_Model_Voucher->id
     * @param  array
     * @param  string
     * @return void
     */
    public function createCodeImage($voucherId, $itemData, $renderCode)
    {
        if (!empty($itemData["product_config"][self::GOYADA_IMAGE_PARAMS]["code_prefix"])) {
            $renderCode = $itemData["product_config"][self::GOYADA_IMAGE_PARAMS]["code_prefix"] . $renderCode;
        }
        if (!empty($itemData["product_config"][self::GOYADA_IMAGE_PARAMS]["code_suffix"])) {
            $renderCode = $renderCode . $itemData["product_config"][self::GOYADA_IMAGE_PARAMS]["code_suffix"];
        }

        switch ($itemData["product_config"][self::GOYADA_IMAGE_TYPE]) {
            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_BARCODE:
                $this->createBARcode($voucherId, $renderCode);
                break;

            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_QRCODE:
                $this->createQRcode($voucherId, $renderCode);
                break;

            case Crossroads_Serialcodes_Helper_Data::SERIALCODES_IMAGE_AZTECCODE:
                $this->createAZTECcode($voucherId, $renderCode);
                break;

            default:
                break;
        }
    }

    /**
     * Create bar-code
     *
     * @param  Crossroads_Goyada_Model_Voucher->id
     * @param  string
     * @return void
     */
    public function createBARcode($voucherId, $renderCode)
    {
        require_once(Mage::getBaseDir("lib") . "/barcodegen.1d-php5.v5.1.0/class/BCGFontFile.php");
        require_once(Mage::getBaseDir("lib") . "/barcodegen.1d-php5.v5.1.0/class/BCGColor.php");
        require_once(Mage::getBaseDir("lib") . "/barcodegen.1d-php5.v5.1.0/class/BCGDrawing.php");
        require_once(Mage::getBaseDir("lib") . "/barcodegen.1d-php5.v5.1.0/class/BCGcode128.barcode.php");

        try {

            // Loading Font
            $font = new BCGFontFile(Mage::getBaseDir("lib") . "/barcodegen.1d-php5.v5.1.0/font/Arial.ttf", 18);

            // The arguments are R, G, B for color.
            $color_black = new BCGColor(0, 0, 0);
            $color_white = new BCGColor(255, 255, 255);

            $drawException = null;
            try {
                $codeImage = new BCGcode128();
                $codeImage->setScale(2); // Resolution
                $codeImage->setThickness(30); // Thickness
                $codeImage->setForegroundColor($color_black); // Color of bars
                $codeImage->setBackgroundColor($color_white); // Color of spaces
                $codeImage->setFont($font); // Font (or 0)
                $codeImage->parse($renderCode); // Code
                $codeImage->clearLabels(); // Remove plain text code below barcode
            } catch (Exception $exception) {
                $drawException = $exception;
            }

            $newData = array(
                "voucher_id" => $voucherId,
                "image_hash" => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                "image_data" => null
            );

            $fileName = Mage::getBaseDir("media") . "/barcodes/{$newData["image_hash"]}.png";

            $drawing = new BCGDrawing($fileName, $color_white);

            if ($drawException) {
                $drawing->drawException($drawException);
            } else {
                $drawing->setBarcode($codeImage);
                $drawing->draw();
            }

            $drawing->finish(BCGDrawing::IMG_FORMAT_PNG);
            $newData["image_data"] = file_get_contents($fileName);
            unlink($fileName);

            $voucherImage = Mage::getModel("Crossroads_Goyada/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();
            $this->log("BAR-code created for {$voucherId}");

        } catch (Exception $exception) {
            $this->log("Exception creating BAR-code for {$voucherId}");
            Mage::logException($exception);
        }
    }

    /**
     * Create QR-code
     *
     * @param  Crossroads_Goyada_Model_Voucher->id
     * @param  string
     * @return void
     */
    public function createQRcode($voucherId, $renderCode)
    {
        require_once(Mage::getBaseDir("lib") . "/phpqrcode/qrlib.php");

        try {
            $newData = array(
                "voucher_id" => $voucherId,
                "image_hash" => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                "image_data" => null
            );

            $fileName = Mage::getBaseDir("media") . "/barcodes/{$newData["image_hash"]}.png";

            QRcode::png($renderCode, $fileName, QR_ECLEVEL_L, 8);
            $newData["image_data"] = file_get_contents($fileName);
            unlink($fileName);

            $voucherImage = Mage::getModel("Crossroads_Goyada/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();
            $this->log("QR-code created for {$voucherId}");
        } catch (Exception $exception) {
            $this->log("Exception creating QR-code for {$voucherId}");
            Mage::logException($exception);
        }
    }

    /**
     * Create aztec-code
     *
     * @param  Crossroads_Goyada_Model_Voucher->id
     * @param  string
     * @return void
     */
    public function createAZTECcode($voucherId, $renderCode)
    {
        require_once Mage::getBaseDir("lib") . DS . "Metzli" . DS . "Metzli.php";
        $renderFactor = 8; // 8 = 152px x 152px when aztec code is compact

        try {
            $aztecCode = MetzliEncoder::encode($renderCode);
            $renderer = new MetzliPngRenderer($renderFactor);

            $newData = array(
                "voucher_id" => $voucherId,
                "image_hash" => md5("SomeStartSalt{$renderCode}SomeEndSalt"),
                "image_data" => $renderer->render($aztecCode)
            );

            $voucherImage = Mage::getModel("Crossroads_Goyada/voucher_image");
            $voucherImage->addData($newData);
            $voucherImage->save();
            $this->log("AZTEC-code created for {$voucherId}");
        } catch (Exception $exception) {
            $this->log("Exception creating AZTEC-code for {$voucherId}");
            Mage::logException($exception);
        }
    }

    /**
     * Save voucher
     *
     * @param  array
     * @param  array
     * @param  string
     * @return Crossroads_Goyada_Model_Voucher->id
     */
    public function saveVoucher($itemData, $result, $callData)
    {
        $voucher = Mage::getModel("Crossroads_Goyada/voucher");
        $newData = [
            "item_id" => $itemData["item_id"],
            "voucher_action" => $itemData["product_config"][self::GOYADA_API_SERVICE],
            "voucher_product_id" => $itemData["product_config"][self::GOYADA_PRODUCT_ID],
            "voucher_data" => json_encode($itemData["product_config"][self::GOYADA_API_DATA]),
            "voucher_code" => !empty($result["cardNumber"]) ? $result["cardNumber"] : null,
            "valid_to" => !empty($result["expiryDate"]) ? $result["expiryDate"] : null,
            "api_url" => $callData["url"],
            "api_result" => $callData["result"],
        ];
        $voucher->addData($newData);
        $voucher->save();

        return $voucher->getId();
    }

    /**
     * Add message to registry
     *
     * @param  string
     * @param  string
     * @return void
     */
    public function addRegisterMessage($key, $msg)
    {
        $oldMessage = Mage::registry($key);
        $newMessage = $oldMessage . $msg;

        // If previously set, nothing will be registered if third parameter is true
        Mage::register($key, $newMessage, true);

        // Check if new message was registered
        if (Mage::registry($key) !== $newMessage) {
            // If not, key was registered before, unregister and try again.
            Mage::unregister($key);
            Mage::register($key, $newMessage, true);
        }
    }

    /**
     * Purchase voucher
     *
     * @param  Mage_Sales_Model_Order
     * @param  [?Mage_Sales_Model_Order_Item->id]
     * @return void
     */
    public function purchaseVoucher($order, $itemIds = [])
    {
        $itemList = $this->getVoucherItemList($order);

        foreach ($itemList as $itemId => $itemData) {

            // Only process vouchers for specific items, if supplied
            if (!empty($itemIds) && !in_array($itemId, $itemIds)) {
                continue;
            }

            // Make sure item has Goyada product id
            if (!empty($itemData["product_config"][self::GOYADA_PRODUCT_ID])) {

                // Make sure we only get missing codes
                $existingCodes = Mage::getModel("Crossroads_Goyada/voucher")
                        ->getCollection()
                        ->addItemFilter($itemId)
                        ->addCodeFilter();

                $existingCodeQty = count($existingCodes);
                if ($existingCodeQty <= $itemData["qty"]) {
                    $itemData["qty"] = $itemData["qty"] - $existingCodeQty;
                }

                if ($itemData["qty"] < 1) {
                    Mage::getSingleton("adminhtml/session")->addSuccess("All codes already issued for [{$itemData["sku"]}]");
                    continue;
                }

                // Can only fetch one code per API call, iterate to fetch them all
                for ($iterations = $itemData["qty"]; $iterations > 0; $iterations--) {
                    $callData = $this->_callAPI($order, $itemData);
                    if (empty($callData["result"])) {
                        Mage::throwException("Got no response from Goyada endpoint");
                    }
                    $result = json_decode($callData["result"], true);

                    if (empty($result["cardNumber"])) {
                        // Save voucher without code, since it also saves raw result if something goes wrong
                        $voucherId = $this->saveVoucher($itemData, $result, $callData);
                        Mage::throwException("Got this from Goyada endpoint: " . $callData["result"]);
                    } else {
                        // Try to extract expiryDate
                        $validTo = null;
                        if (!empty($result["expiryDate"])) {
                            $expiryDate = strtotime($result["expiryDate"]);
                            if (!empty($expiryDate)) {
                                $validTo = date("Y-m-d", $expiryDate);
                            }
                        }
                        $result["expiryDate"] = $validTo;
                        $voucherId = $this->saveVoucher($itemData, $result, $callData);
                    }

                    // Only create image if product is configured for it and we got voucher code in result
                    if (!empty($voucherId) && !empty($itemData["product_config"][self::GOYADA_IMAGE_TYPE]) && !empty($result["cardNumber"])) {
                        $this->createCodeImage($voucherId, $itemData, $result["cardNumber"]);
                    }

                    // Pause for a while if we are supposed to buy more than 1 voucher
                    if ($iterations > 1) {
                        usleep(100000);
                    }
                }

                $codeWord = $itemData["qty"] > 1 ? "codes" : "code";
                $this->addRegisterMessage("goyada_order_comment", "- Successfully issued {$itemData["qty"]} {$codeWord} for [{$itemData["sku"]}] {$itemData["name"]}\n");
                $this->log("Successfully issued {$itemData["qty"]} {$codeWord} for [{$itemData["sku"]}] '{$itemData["name"]}' in order {$order->getIncrementId()}");

            } else {
                Mage::throwException("Product is missing Goyada product id");
            }
        }
    }

    /**
     * Make API call to Goyada endpoint
     *
     * @param  Mage_Sales_Model_Order
     * @param  array
     * @return array
     */
    protected function _callAPI($order, $itemData)
    {
        // call: https://giftcardproxy-public-stage-stage.goyadacloud.com/gcp?service=fashionchequeTest&productIdentifier=123456789&currency=EUR&price=1600
        // result: {"expiryDate":"2021-06-08T16:11:40.4577968+02:00","currency":"EUR","status":"OK","verificationCode":"978292","amount":"1600","cardNumber":"6299200000392978631"}
        $url = $itemData["product_config"][self::GOYADA_API_URL];
        $url .= "&productIdentifier=" . $itemData["product_config"][self::GOYADA_PRODUCT_ID];
        //$url .= "&transactionId=" . $order->getIncrementId();
        foreach ($itemData["product_config"][self::GOYADA_API_DATA] as $kv) {
            $url .= "&{$kv["key"]}={$kv["val"]}";
        }

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_VERBOSE, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, 20);

        $this->log("Call to URL: {$url}");
        $result = curl_exec($ch);
        $cErrNo = curl_errno($ch);

        if ($cErrNo) {
            $result = "Curl error #{$cErrNo}: " . curl_error($ch);
            $this->log($result, LOG_ERR);
        } else {
            $this->log("RAW answer: {$result}");
        }

        return [
            "url" => $url,
            "result" => $result
        ];

    }

    /**
     * Wrapper for logging
     *
     * @param string
     * @param int
     * @param string
     * @param boolean
     */
    public function log($msg, $logType = LOG_DEBUG, $logFilename = self::LOG_FILENAME, $force = true)
    {
        if (Mage::getStoreConfig("goyada/general/logging", Mage::app()->getStore()->getId()) || $force) {
            Mage::log($msg, $logType, $logFilename, $force);
        }
    }

}
