<?php

class Crossroads_Fees_Helper_Tax extends Mage_Core_Helper_Abstract {
    const CONFIG_TAX_CLASS_FALLBACK_SHIPPING    = "tax/class_fallbacks/shipping_tax_class";
    const CONFIG_TAX_CLASS_FALLBACK_PAYMENT_FEE = "tax/class_fallbacks/crossroads_fees_payment_tax_class";
    /**
     * Minimum subtotal considered to be an actual amount of currency.
     */
    const MIN_SUBTOTAL = 0.01;
    /**
     * Minimum allowed rate from the different dynamic tax classes, if the resulting rate is smaller than this it will fall back to the default tax class rate. Percentage rate, divide by 100 to get fraction.
     */
    const MIN_RATE     = 1;
    /**
     * Tax class used for fees and shipping fees to provide weighted average tax based on subtotal tax.
     *
     * It is negative since no tax class id will be less than zero.
     */
    const WEIGHTED_AVERAGE_TAX_CLASS_ID = -923;
    /**
     * Tax class used for fess and shipping fees to provide a tax based on the tax with the largest sum.
     *
     * Dominant moms, den momsen som har störst siffervärde.
     */
    const DOMINATING_TAX_CLASS_ID = -922;
    /**
     * Tax class used for fees and shipping fees to provide a tax based on the tax with the highest rate.
     */
    const HIGHEST_RATE_TAX_CLASS_ID = -921;
    /**
     * Shipping fee type, used to determine fallback tax rate if an average/dominating/highest rate cannot be found.
     */
    const TYPE_SHIPPING = 1;
    /**
     * Payment fee type, used to determine fallback tax rate if an average/dominating/highest rate cannot be found.
     */
    const TYPE_PAYMENT_FEE = 2;

    protected static $taxClassIdToCode = [
        self::WEIGHTED_AVERAGE_TAX_CLASS_ID => "Weighted average subtotal rate",
        self::DOMINATING_TAX_CLASS_ID       => "Dominating subtotal rate",
        self::HIGHEST_RATE_TAX_CLASS_ID     => "Highest subtotal rate"
    ];

    /**
     * Returns true if the supplied tax class id is a custom tax class maintained by this helper.
     *
     * @param  integer
     * @return boolean
     */
    public function isCustomTaxClassId($taxClassId) {
        switch($taxClassId) {
        case Crossroads_Fees_Helper_Tax::WEIGHTED_AVERAGE_TAX_CLASS_ID:
        case Crossroads_Fees_Helper_Tax::DOMINATING_TAX_CLASS_ID:
        case Crossroads_Fees_Helper_Tax::HIGHEST_RATE_TAX_CLASS_ID:
            return true;
        default:
            return false;
        }
    }

    /**
     * Returns the tax rate specification for the given type, address and request. The rate will fall back
     * to the tax class id set for the given type.
     *
     * Has the following two values in the return value:
     *
     * * rate         => integer for tax-rate
     * * appliedRates => array of applied rates, matches the return format for 'tax/calculation'::getAppliedRates()
     *
     * @param  integer  One of the TYPE_* constants from this class
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Varien_Object
     * @return array
     */
    public function getTaxRate($type, $address, $taxRequest) {
        if(!$taxRequest->getCountryId() || !$taxRequest->getCustomerClassId() || !$taxRequest->getProductClassId()) {
            throw new Exception("Crossroads_Fees: Tax request is missing mandatory parameters");
        }

        $classId = $taxRequest->getProductClassId();
        $rate    = $this->getRate($type, $address, $taxRequest);

        if($rate < 0) {
            return $this->getFallbackTaxRate($type, $address->getQuote()->getStore(), $taxRequest);
        }

        return [
            $rate,
            [[
                "id"      => self::$taxClassIdToCode[$classId],
                "percent" => $rate,
                "rates"   => [[
                    "code"      => self::$taxClassIdToCode[$classId],
                    "title"     => Mage::helper("crossroads_fees")->__(self::$taxClassIdToCode[$classId]),
                    "percent"   => $rate,
                    "position"  => 0,
                    "priority"  => 0,
                    "CFdynamic" => true
                ]]
            ]]
        ];
    }

    /**
     * Retrieves the tax-rate for the supplied fee-type, address and request. Throws an error if the
     * product class id is not one of the custom tax classes supported by this helper.
     *
     * @param  integer  One of the TYPE_* constants from this class
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Varien_Object
     * @return integer
     */
    protected function getRate($type, $address, $taxRequest) {
        switch($taxRequest->getProductClassId()) {
        case Crossroads_Fees_Helper_Tax::WEIGHTED_AVERAGE_TAX_CLASS_ID:
            return $this->getAverageTaxRate($type, $address);
        case Crossroads_Fees_Helper_Tax::DOMINATING_TAX_CLASS_ID:
            return $this->getDominatingTaxRate($type, $address);
        case Crossroads_Fees_Helper_Tax::HIGHEST_RATE_TAX_CLASS_ID:
            return $this->getHighestTaxRate($type, $address);
        default:
            throw new Exception("Crossroads_Fees: Tax class id not recognized: '{$taxRequest->getProductClassId()}'.");
        }
    }

    /**
     * Calculates the weighted average rate of the included tax rates, will return -1 if no rate
     * was possible to determine.
     *
     * @param  integer  One of the TYPE_* constants from this class
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Varien_Object
     * @return integer
     */
    public function getAverageTaxRate($type, $address) {
        $quote           = $address->getQuote();
        $subtotalInclTax = 0.0;
        $subtotal        = 0.0;

        foreach($quote->getAllItems() as $item) {
            // BaseRowTotal never includes tax
            $subtotalInclTax += $item->getBaseRowTotalInclTax();
            $subtotal        += $item->getBaseRowTotal();
        }

        if($subtotalInclTax < self::MIN_SUBTOTAL) {
            return -1;
        }

        $rate = max($subtotalInclTax / $subtotal - 1.0, 0.0) * 100;

        if($rate < self::MIN_RATE) {
            return -1;
        }

        return $rate;
    }

    /**
     * Determines the dominating rate based on subtotal tax-amount per rate, returning -1 if it
     * fails to determine any rate.
     *
     * @param  integer  One of the TYPE_* constants from this class
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Varien_Object
     * @return integer
     */
    public function getDominatingTaxRate($type, $address) {
        $quote     = $address->getQuote();
        $usedRates = [];

        foreach($quote->getAllItems() as $item) {
            $rates = $item->getAppliedRates();

            if(empty($rates)) {
                continue;
            }

            foreach($rates as $r) {
                if(empty($usedRates[$r["id"]])) {
                    $usedRates[$r["id"]] = [
                        "rate" => (double)$r["percent"],
                        "sum"  => 0.0
                    ];
                }

                if($usedRates[$r["id"]]["rate"] !== (double)$r["percent"]) {
                    throw new Exception("Mismatching rate for tax rate id '{$r["id"]}'.");
                }

                $usedRates[$r["id"]]["sum"] += (double)$item->getBaseTaxAmount();
            }
        }

        // Sort with highest absolute sum first
        uasort($usedRates, function($a, $b) {
            if($a["sum"] > $b["sum"]) {
                return -1;
            }
            else if($a["sum"] === $b["sum"]) {
                return 0;
            }
            else {
                return 1;
            }
        });

        $rate = current($usedRates);

        if( ! $rate || $rate["sum"] < self::MIN_SUBTOTAL || $rate["rate"] < self::MIN_RATE) {
            return -1;
        }

        return $rate["rate"];
    }

    /**
     * Determines the highest rate based, returns -1 if it fails to determine any rate.
     *
     * @param  integer  One of the TYPE_* constants from this class
     * @param  Mage_Sales_Model_Quote_Address
     * @param  Varien_Object
     * @return integer
     */
    public function getHighestTaxRate($type, $address) {
        $quote     = $address->getQuote();
        $usedRates = [];

        foreach($quote->getAllItems() as $item) {
            $rates = $item->getAppliedRates();

            if(empty($rates)) {
                continue;
            }

            foreach($rates as $r) {
                if(empty($usedRates[$r["id"]])) {
                    $usedRates[$r["id"]] = (double)$r["percent"];
                }

                if($usedRates[$r["id"]] !== (double)$r["percent"]) {
                    throw new Exception("Mismatching rate for tax rate id '{$r["id"]}'.");
                }
            }
        }

        // Sort with highest absolute sum first
        arsort($usedRates);

        $rate = current($usedRates);

        if( ! $rate || $rate["sum"] < self::MIN_SUBTOTAL || $rate["rate"] < self::MIN_RATE) {
            return -1;
        }

        return $rate["rate"];
    }

    /**
     * Uses the normal tax-calculation to determine the rate from a fallback-type setting.
     *
     * @param  integer  One of the TYPE_* constants from this class
     * @param  Varien_Object
     * @return integer
     */
    protected function getFallbackTaxRate($type, $store, $taxRequestFallback) {
        $calc    = Mage::getSingleton("tax/calculation");
        $request = clone $taxRequestFallback;

        switch($type) {
        case Crossroads_Fees_Helper_Tax::TYPE_SHIPPING:
            $request->setProductClassId($store->getConfig(Crossroads_Fees_Helper_Tax::CONFIG_TAX_CLASS_FALLBACK_SHIPPING));
            break;
        case Crossroads_Fees_Helper_Tax::TYPE_PAYMENT_FEE:
            $request->setProductClassId($store->getConfig(Crossroads_Fees_Helper_Tax::CONFIG_TAX_CLASS_FALLBACK_PAYMENT_FEE));
            break;
        default:
            throw new Exception("Crossroads_Fees: Unknown tax type '$type'.");
        }

        return [
            $calc->getRate($request),
            $calc->getAppliedRates($request)
        ];
    }
}
