<?php

/**
 * Class responsible for adding the payment amount and payment tax to the quote address total.
 */
class Crossroads_Fees_Model_Sales_Quote_Address_Total_Paymentfee extends Mage_Sales_Model_Quote_Address_Total_Abstract {
    protected $_code = "payment_fee";

    public function collect(Mage_Sales_Model_Quote_Address $address) {
        parent::collect($address);

        // Reset, since the address might now be empty
        $address->setPaymentFeeAmount(0);
        $address->setPaymentFeeTaxAmount(0);
        $address->setBasePaymentFeeAmount(0);
        $address->setBasePaymentFeeTaxAmount(0);
        $address->setPaymentFeeTitle(null);

        $calc      = Mage::getSingleton("tax/calculation");
        $helper    = Mage::helper("crossroads_fees/paymentfee");
        $taxHelper = Mage::helper("crossroads_fees/tax");
        $quote     = $address->getQuote();
        $store     = $quote->getStore();
        $method    = $quote->getPayment()->getMethod();

        // If it is empty or we do not yet have a method, do not add a fee
        if( ! count($this->_getAddressItems($address)) || !$helper->canApplyPaymentFee($method, $quote)) {
            return $this;
        }

        $includingTax  = $helper->getPaymentFeeIncludesTax($method, $quote);
        $baseAmount    = $helper->getBasePaymentFeeAmount($method, $quote);
        $feeTitle      = $helper->getPaymentFeeTitle($method, $quote);
        $taxRequest    = $helper->getTaxRequest($quote);

        if($taxHelper->isCustomTaxClassId($taxRequest->getProductClassId())) {
            list($taxRate, $appliedRates) = $taxHelper->getTaxRate(
                Crossroads_Fees_Helper_Tax::TYPE_PAYMENT_FEE,
                $address,
                $taxRequest
            );
        }
        else {
            $taxRate      = $calc->getRate($taxRequest);
            $appliedRates = $calc->getAppliedRates($taxRequest);
        }

        $baseTaxAmount = $calc->calcTaxAmount($baseAmount, $taxRate, $includingTax);

        // Convert amounts to user-selected currency
        $amount    = $store->convertPrice($baseAmount, false);
        $taxAmount = $store->convertPrice($baseTaxAmount, false);

        // Save payment fee specifics
        //
        // We only need to set tax_amount since the total amount fields will be set below
        $address->setPaymentFeeTaxAmount($taxAmount);
        $address->setBasePaymentFeeTaxAmount($baseTaxAmount);
        $address->setPaymentFeeTitle($feeTitle);

        // Add to total
        //
        // This also sets payment_fee_amount and base_payment_fee_amount, since the total code
        // (payment_fee) is not used by anything else we do not replace anything, just add to the grand total.
        //
        // We subtract the tax if it is already included since we add the tax below.
        $address->setTotalAmount($this->getCode(), $includingTax ? $amount - $taxAmount : $amount);
        $address->setBaseTotalAmount($this->getCode(), $includingTax ? $baseAmount - $baseTaxAmount : $baseAmount);

        // Add to tax total
        //
        // This will be added to the total we set above.
        $address->addTotalAmount("tax", $taxAmount);
        $address->addBaseTotalAmount("tax", $baseTaxAmount);

        // Add the tax to applied taxes
        $this->_saveAppliedTaxes($address, $appliedRates, $taxAmount, $baseTaxAmount, $taxRate);
    }

    /**
     * Adds/updates applied taxes to include the amount and baseAmount, copied from Mage_Sales_Model_Quote_Address_Total_Tax.
     *
     * @param  Mage_Sales_Model_Quote_Address
     * @param  array
     * @param  double
     * @param  double
     * @param  integer
     * @return void
     */
    protected function _saveAppliedTaxes(Mage_Sales_Model_Quote_Address $address, $applied, $amount, $baseAmount, $rate)
    {
        $previouslyAppliedTaxes = $address->getAppliedTaxes();
        $process = count($previouslyAppliedTaxes);

        foreach ($applied as $row) {
            if (!isset($previouslyAppliedTaxes[$row['id']])) {
                $row['process'] = $process;
                $row['amount'] = 0;
                $row['base_amount'] = 0;
                $previouslyAppliedTaxes[$row['id']] = $row;
            }

            if (!is_null($row['percent'])) {
                $row['percent'] = $row['percent'] ? $row['percent'] : 1;
                $rate = $rate ? $rate : 1;

                $appliedAmount = $amount/$rate*$row['percent'];
                $baseAppliedAmount = $baseAmount/$rate*$row['percent'];
            } else {
                $appliedAmount = 0;
                $baseAppliedAmount = 0;
                foreach ($row['rates'] as $rate) {
                    $appliedAmount += $rate['amount'];
                    $baseAppliedAmount += $rate['base_amount'];
                }
            }


            if ($appliedAmount || $previouslyAppliedTaxes[$row['id']]['amount']) {
                $previouslyAppliedTaxes[$row['id']]['amount'] += $appliedAmount;
                $previouslyAppliedTaxes[$row['id']]['base_amount'] += $baseAppliedAmount;
            } else {
                unset($previouslyAppliedTaxes[$row['id']]);
            }
        }
        $address->setAppliedTaxes($previouslyAppliedTaxes);
    }

    public function fetch(Mage_Sales_Model_Quote_Address $address) {
        $address->addTotal([
            "code"           => $this->getCode(),
            "title"          => $address->getPaymentFeeTitle(),
            "value"          => $address->getPaymentFeeAmount(),
            "value_incl_tax" => $address->getPaymentFeeAmount() + $address->getPaymentFeeTaxAmount(),
            "value_excl_tax" => $address->getPaymentFeeAmount()
        ]);

        return $this;
    }
}
