<?php

/**
 * Helper for payment fees.
 */
class Crossroads_Fees_Helper_Paymentfee extends Mage_Core_Helper_Abstract {
    /**
     * Key to Magento configuration for specific payment fees for different payemnt methods,
     * their amounts, if they are enabled and any description.
     */
    const CONFIG_FEES         = "crossroads_fees/payment_fees/fee_amounts";
    /**
     * Key to Magento configuration, if the payment fee module is enabled, boolean.
     */
    const CONFIG_ENABLED      = "crossroads_fees/payment_fees/enabled";
    /**
     * Key to Magento configuration, if the payment fee includes tax.
     */
    const CONFIG_INCLUDES_TAX = "crossroads_fees/payment_fees/includes_tax";
    /**
     * Key to Magento configuration, which tax class the payment fees should use.
     */
    const CONFIG_TAX_CLASS    = "tax/classes/crossroads_fees_payment_tax_class";

    /**
     * Returns a normalized map of payment fee settings.
     *
     * Contains the following keys:
     *  * enabled: boolean, default: false
     *  * amount:  float,   default: 0.0
     *  * title:   string,  default: ""
     *
     * @param  Mage_Core_Model_Store
     * @param  string
     * @return array
     */
    protected function getFeeConfig($store, $paymentMethod) {
        $conf = unserialize($store->getConfig(self::CONFIG_FEES)) ?: [];

        if(array_key_exists($paymentMethod, $conf) && is_array($conf[$paymentMethod])) {
            return [
                "enabled" => array_key_exists("enabled", $conf[$paymentMethod]) ? (bool)$conf[$paymentMethod]["enabled"] : false,
                "amount"  => array_key_exists("amount",  $conf[$paymentMethod]) ? (double)$conf[$paymentMethod]["amount"] : 0.0,
                "title"   => array_key_exists("title",   $conf[$paymentMethod]) ? (string)$conf[$paymentMethod]["title"] : ""
            ];
        }

        return [
            "enabled" => false,
            "amount"  => 0.0,
            "title"   => ""
        ];
    }

    /**
     * Returns true if we can apply a payment fee on the supplied payment method and quote.
     *
     * @param  string
     * @param  Mage_Sales_Model_Quote
     * @return boolean
     */
    public function canApplyPaymentFee($paymentMethod, $quote) {
        $store = $quote->getStore();

        return $paymentMethod && $store->getConfig(self::CONFIG_ENABLED) && $this->getFeeConfig($store, $paymentMethod)["enabled"];
    }

    /**
     * Returns true if the payment fee returned from getBasePaymentFeeAmount() includes tax.
     *
     * @param  string
     * @param  Mage_Sales_Model_Quote
     * @return boolean
     */
    public function getPaymentFeeIncludesTax($paymentMethod, $quote) {
        return (bool)$quote->getStore()->getConfig(self::CONFIG_INCLUDES_TAX);
    }

    /**
     * Returns the payment fee for the supplied payment method in the current
     * store's base currency.
     *
     * @param  string
     * @param  Mage_Sales_Model_Quote
     * @return double|null
     */
    public function getBasePaymentFeeAmount($paymentMethod, $quote) {
        return $this->getFeeConfig($quote->getStore(), $paymentMethod)["amount"];
    }

    /**
     * Returns the tax request for the payment fee tax for the the current
     * store's base currency.
     *
     * @param  Mage_Sales_Model_Quote
     * @return double
     */
    public function getTaxRequest($quote) {
        $calc            = Mage::getSingleton("tax/calculation");
        $store           = $quote->getStore();
        $paymentTaxClass = $store->getConfig(self::CONFIG_TAX_CLASS);
        $taxRequest      = $calc->getRateRequest(
            $quote->getShippingAddress(),
            $quote->getBillingAddress(),
            $quote->getCustomerTaxClassId(),
            $store
        );

        $taxRequest->setProductClassId($paymentTaxClass);

        return $taxRequest;
    }

    /**
     * Returns the title of the payment fee, null if no fee is found for the given payment method.
     *
     * @param  string
     * @param  Mage_Sales_Model_Quote
     * @return string|null
     */
    public function getPaymentFeeTitle($paymentMethod, $quote) {
        return $this->getFeeConfig($quote->getStore(), $paymentMethod)["title"];
    }
}
