<?php

class Crossroads_Elasticsearch_Products_FacetedController extends Crossroads_API_Controller_Resource {
    /**
     * Parameters:
     *  * params:   Key-value query parameters used for filter
     *  * response: The response body Varien_Object
     */
    const EVENT_PRODUCTS_FACETED_RESPONSE = "crossroads_elasticsearch_products_faceted_response";

    /**
     * Parameters:
     *  * data:   Varien_Object with sorted value, set to true if event has sorted.
     *  * params: Request parameters
     *  * query:  The elasticsearch query to sort.
     */
    const EVENT_SORT = "crossroads_elasticsearch_products_faceted_sort";

    /**
     * Event fired when constructing the query for filtering faceted products.
     *
     * Params:
     *
     *  * query:  The elasticsearch query
     *  * params: A Varien_Object of the parameters, remaining values will be used for filtering
     */
    const EVENT_FACETED_QUERY = "crossroads_elasticsearch_products_faceted_query";

    /**
     *
     * @apiParam {Integer} [limit=20]
     * @apiParam {Integer} [page=0]
     * @apiParam {String}  [sort]   Multiple sort-columns can be used in order with most significant first, default is _score (and will be added last as a discriminator, no need to add this manually)
     * @apiParam {Boolean} [aggregate=false]
     */
    public function getAll() {
        $helper = Mage::helper("Crossroads_Elasticsearch");
        $store  = Mage::app()->getStore();
        $req    = $this->getRequest();

        if( ! $helper->enabledInStore([Mage_Catalog_Model_Product::ENTITY], $store)) {
            return [403, [ "message" => "Elasticsearch extension is disabled for this store" ]];
        }

        $factory     = Mage::getModel("API/factory");
        $pageSize    = min((int)$req->getQuery("limit", "20"), 100);
        $page        = max((int)$req->getQuery("page", "1"), 1);
        $aggregate   = trim(strtolower($req->getQuery("aggregate", "false"))) === "true";
        $resource    = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes  = $resource->getFilterableProductAttributes($store);
        $bucketOrd   = $resource->getSortedAttributeValues($attributes, $store);
        $filterAttrs = array_filter($attributes, function($a) { return $a->getIsFilterable() && $a->hasKeywordField(); });
        $sortAttrs   = array_filter($attributes, function($a) { return $a->getUsedForSortBy(); });
        $client      = $helper->createClient();
        $params      = new Varien_Object(array_filter(array_diff_key($this->getRequest()->getParams(), [
            "limit"     => true,
            "page"      => true,
            "sort"      => true,
            "aggregate" => true,
        ])));
        $q = $helper->createProductQuery($store, Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_CATALOG);

        if($params->hasData("category_path")) {
            $path = $params->getData("category_path");
            $rewrite = Mage::getModel("core/url_rewrite")
                ->setStoreId($store->getId())
                ->loadByRequestPath($path);

            if($rewrite->getId()) {
                $category = Mage::getModel("catalog/category")
                    ->load($rewrite->getCategoryId());

                $match = $category->getPath();

                if($category->getIsAnchor()) {
                    $q->addFilter(["bool" => ["should" => [
                        ["term" => ["xes_paths" => $match]],
                        // Trailing slash to prevent other categories on the
                        // same level from matching if they share id-prefix
                        ["prefix" => ["xes_paths" => $match."/"]]
                    ]]]);
                }
                else {
                    $q->addFilter(["term" => ["xes_paths" => $match]]);
                }
            }
            else {
                // Dummy value for ensuring nothing matches
                $q->addFilter(["term" => ["xes_paths" => "999999/99999"]]);
            }

            $params->unsetData("category_path");
        }

        // TODO: Is this really necessary now? does not seem to be used
        Mage::dispatchEvent(self::EVENT_FACETED_QUERY, [
            "params" => $params,
            "query"  => $q,
        ]);

        $params = $params->getData();
        $clone  = clone $q;
        $query  = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->addSource([ "sku", "name" ])
            ->setPageSize($pageSize)
            ->setPage($page)
            ->setQuery($clone->setFacetFilter($filterAttrs, $params));

        if($aggregate) {
            $query->setAggregations(Mage::getModel("Crossroads_Elasticsearch/query_aggregation_attribute")
                ->setAttributes($filterAttrs)
                ->setParameters($params)
                ->setExcludeQueryConstructor(function($paramKey) use($q, $filterAttrs, $params) {
                    $clone = clone $q;

                    return $clone->setFacetFilter($filterAttrs, array_diff_key($params, [ $paramKey => true ]));
                }));
        }

        if($req->getQuery("sort")) {
            $obj = new Varien_Object([
                "sorted" => false,
            ]);

            // TODO: Can we merge this somehow into some default?
            Mage::dispatchEvent(self::EVENT_SORT, [
                "query"   => $query,
                "flags"   => $obj,
                "request" => $req,
                "sort"    => $req->getQuery("sort"),
            ]);

            if( ! $obj->getSorted()) {
                foreach(array_reverse(array_filter(array_map("trim", explode(",", $req->getQuery("sort"))))) as $sort) {
                    $query->addAttributeSort($sortAttrs, $sort);
                }
            }
        }

        $result = $client->search($query->toRequest());

        $numProducts = $result["hits"]["total"];
        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        $resp = new Varien_Object([
            "_time"    => $result["took"],
            "total"    => $numProducts,
            // TODO: Reuse this between search, recommended, faceted. Maybe create some sharing through collection model(s)?
            "products" => $factory->createProductListSerializer($store)->mapArray(
                $helper->fetchProducts($entityIds, $pageSize, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG)),
            "facets"   => $aggregate && array_key_exists("aggregations", $result) ? $helper->transformFacets($result["aggregations"], $filterAttrs, $params, $bucketOrd, $numProducts) : null,
        ]);

        Mage::dispatchEvent(self::EVENT_PRODUCTS_FACETED_RESPONSE, [
            "params"   => $params,
            "response" => $resp,
        ]);

        return [200, $resp->getData()];
    }

    public function getItem($id) {
        return [404];
    }
}
