<?php

/**
 * Indexes product (store id, entity id) -> list of store-based category names for the full hierarchy of the products.
 */
class Crossroads_Elasticsearch_Model_Resource_Product_Categories extends Crossroads_Elasticsearch_Model_Resource_Abstract {
    protected function _construct() {
        $this->_setResource("core_read");
    }

    public function addMapping($event) {
        $mapping = $event->getMapping();
        $locale  = $event->getLocale();

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_CATEGORIES, [
            "type"           => "keyword",
            // "copy_to" => "_all",
            "fields"         => [
                $locale => [
                    "type"     => "text",
                    "analyzer" => Mage::helper("Crossroads_Elasticsearch")->toLanguageAnalyzer($locale),
                    //"copy_to"  => "_all",
                    "store"    => true,
                ],
            ],
        ]);
        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_POSITIONS, [
            "type"       => "integer",
            "doc_values" => true,
        ]);
        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_PATHS, [
            "type"       => "keyword",
            "index"      => true,
            "doc_values" => true,
        ]);
    }

    public function indexAll($event) {
        $manager = $event->getManager();
        $indices = $event->getIndices();
        $stores  = $event->getStores();
        $types   = $event->getTypes();
        $sql     = $this->getReadConnection();
        $query   = $this->createProductCategoryQuery($sql, $stores);
        $stmt    = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    public function indexProducts($event) {
        $productIds = $event->getProductIds();
        $manager    = $event->getManager();
        $indices    = $event->getIndices();
        $stores     = $event->getStores();
        $types      = $event->getTypes();
        $sql        = $this->getReadConnection();
        $query      = $this->createProductCategoryQuery($sql, $stores, $productIds);
        $stmt       = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    protected function loadData($indices, $types, $manager, $stmt) {
        foreach($stmt as $row) {
            // Skip empty position values (0 = empty)
            $positions  = array_values(array_filter(array_unique(array_map(function($i) { return (int)$i; }, explode("±", $row["positions"])))));
            $categories = array_unique(explode("±", $row["category"]));
            $paths      = array_unique(explode("±", $row["paths"]));

            $doc = array_filter([
                Crossroads_Elasticsearch_Helper_Data::FIELD_CATEGORIES => $categories,
                Crossroads_Elasticsearch_Helper_Data::FIELD_POSITIONS  => empty($positions) ? null : $positions,
                Crossroads_Elasticsearch_Helper_Data::FIELD_PATHS      => $paths,
            ]);

            if(empty($doc)) {
                continue;
            }

            // Perform partial updates
            $manager->queue([ "update" => [
                "_index"   => $indices[$row["store_id"]],
                "_routing" => (string)$row["entity_id"],
                "_type"    => "product",
                "_id"      => (string)$row["entity_id"],
            ]], [ "doc" => $doc ]);
        }
    }

    protected function createProductCategoryQuery($sql, $stores, $productIds = null) {
        $eavConfig             = Mage::getSingleton("eav/config");
        $categoryEntityTypeId  = $eavConfig->getEntityType(Mage_Catalog_Model_Category::ENTITY)->getId();

        $select = $sql->select()
            ->from(["s" => $this->createEntityIdStoreQuery($sql, $stores, $productIds)], ["entity_id", "store_id", "website_id"])
            ->join(["p" => $this->getTable("catalog/product")], "p.entity_id = s.entity_id", ["type_id", "sku", "created_at", "updated_at"]);

        $categoriesFlatTree = $sql->select()
                ->from(["cc"   => $this->getTable("catalog/category")], ["entity_id"])
                // is_active filtering
                ->from(["iav"  => $this->getTable("eav/attribute")], [])
                ->join(["iavv" => $this->getValueTable("catalog/category", "int")], "iavv.attribute_id = iav.attribute_id AND iavv.entity_id = cc.entity_id", ["store_id"])
                // We join with the numbers 2-10 so we skip the root and main category since all products will have it
                ->join(["n"    => new Zend_Db_Expr("(select 2 `i` union select 3 union select 4 union select 5 union select 6 union select 7 union select 8 union select 9 union select 10)")], "n.i < LENGTH(cc.path) - LENGTH(REPLACE(cc.path, '/', '')) + 1", [])
                ->where("iav.attribute_code = ?", "is_active")
                ->where("iavv.value = ?", "1")
                ->columns(["category_id" => new Zend_Db_Expr("SUBSTRING_INDEX(SUBSTRING_INDEX(cc.path, '/', n.i + 1), '/', -1)")]);

        $select = $select
            ->join(["ccp" => $this->getTable("catalog/category_product")], "ccp.product_id = p.entity_id", [])
            ->join(["cid" => $categoriesFlatTree], "cid.entity_id = ccp.category_id AND (cid.store_id = s.store_id OR cid.store_id = 0)", [])
            ->join(["cat" => $this->getTable("catalog/category")], "cat.entity_id = ccp.category_id", [])
            ->from(["can" => $this->getTable("eav/attribute")], [])
            ->joinLeft(["cvn_global" => $this->getValueTable("catalog/category", "varchar")], "cvn_global.entity_id = cid.category_id AND cvn_global.attribute_id = can.attribute_id AND cvn_global.store_id = 0", [])
            ->joinLeft(["cvn_store" => $this->getValueTable("catalog/category", "varchar")], "cvn_store.entity_id = cid.category_id AND cvn_store.attribute_id = can.attribute_id AND cvn_store.store_id = s.store_id", [])
            ->columns([
                "category"  => "GROUP_CONCAT(DISTINCT " . $sql->getIfNullSql($sql->quoteIdentifier("cvn_store.value"), $sql->quoteIdentifier("cvn_global.value")) . " SEPARATOR '±')",
                "positions" => "GROUP_CONCAT(ccp.position SEPARATOR '±')",
                "paths"     => "GROUP_CONCAT(DISTINCT cat.path SEPARATOR '±')",
            ]);

        $select = $select->where("can.attribute_code = ?", "name")
            ->where("can.entity_type_id = ?", $categoryEntityTypeId);

        return $select->group(["s.store_id", "p.entity_id", "ccp.category_id"]);
    }
}
