<?php

class Crossroads_Elasticsearch_Model_Query {
    protected $pageSize = 20;
    protected $page     = 1;
    protected $type     = null;
    protected $index    = null;
    protected $source   = [];
    protected $sort     = ["_score"];
    protected $query    = null;
    protected $aggs     = null;

    public function setIndex($index, $type) {
        $this->index = $index;
        $this->type  = $type;

        return $this;
    }

    public function setPageSize($pageSize) {
        $this->pageSize = $pageSize;

        return $this;
    }

    public function setPage($page) {
        $this->page = $page;

        return $this;
    }

    public function setSource($source) {
        $this->source = $source;

        return $this;
    }

    public function addSource($source) {
        $this->source = array_values(array_merge($source, $this->source));

        return $this;
    }

    public function addSort($field, $order, $options = []) {
        array_unshift($this->sort, [ $field => array_merge($options, [ "order" => $order ]) ]);

        return $this;
    }

    public function addAttributeSort($attributes, $sort) {
        switch($sort) {
        case "position":
            return $this->addSort(Crossroads_Elasticsearch_Helper_Data::FIELD_POSITIONS, "asc", [ "mode" => "avg" ]);
        case "purchases":
            return $this->addSort(Crossroads_Elasticsearch_Helper_Data::FIELD_PURCHASES, "desc");
        case "popularity":
            return $this->addSort(Crossroads_Elasticsearch_Helper_Data::FIELD_POPULARITY, "desc");
        default:
            foreach($attributes as $attr) {
                if($attr->getCode() === $sort && ($attr->hasKeywordField() || $attr->getUsedForSortBy())) {
                    // We can easily sort on keyword fields for text
                    return $this->addSort($attr->getKeywordField(), "asc");
                }
            }

            // TODO: Error code
            throw Crossroads_API_ResponseException::create(400, sprintf("Unknown sort key '%s'.", $sort));
        }

        return $this;
    }

    public function setQuery($query) {
        $this->query = $query;

        return $this;
    }

    public function setAggregations($aggs) {
        $this->aggs = $aggs;

        return $this;
    }

    public function toRequest() {
        $body = [
            "size"    => $this->pageSize,
            "from"    => max(0, $this->page - 1) * $this->pageSize,
            "sort"    => $this->sort,
            "query"   => $this->query ? $this->query->toRequest() : null,
            "_source" => $this->source,
        ];

        if($this->aggs) {
            $body["aggs"] = call_user_func_array("array_merge", array_map(
                    function($a) { return is_array($a) ? $a : $a->toRequest(); },
                is_array($this->aggs) ? $this->aggs : [$this->aggs]));

            if(empty($body["aggs"])) {
                $body["aggs"] = new stdClass;
            }
        }

        return [
            "index" => $this->index,
            "type"  => $this->type,
            "body"  => $body,
        ];
    }
}
