<?php

class Crossroads_Elasticsearch_SearchController extends Crossroads_API_Controller_Resource {
    /**
     * Event fired when constructing the query for searching products.
     *
     * Params:
     *
     *  * query:  The elasticsearch query
     *  * params: A Varien_Object of the parameters, remaining values will be used for filtering
     */
    const EVENT_SEARCH_QUERY = "crossroads_elasticsearch_search_query";

    /**
     * Copied from Crossroads_API_SearchController
     */
    protected function getQueryModel($queryText) {
        $storeId = Mage::app()->getStore()->getId();
        $query   = Mage::getModel("catalogsearch/query");

        $query->setStoreId($storeId);
        $query->loadByQuery($queryText);

        if( ! $query->getId()) {
            $query->setQueryText($queryText);
            $query->setStoreId($storeId);
            $query->setPopularity(1);
        }
        else {
            $query->setPopularity($query->getPopularity() + 1);
        }

        $query->prepare();

        return $query;
    }

    /**
     * @apiParam {Integer} [limit=20]
     * @apiParam {Integer} [page=0]
     * @apiParam {String}  [sort]   Multiple sort-columns can be used in order with most significant first, default is _score (and will be added last as a discriminator, no need to add this manually)
     * @apiParam {String}  query
     * @apiParam {Boolean} [aggregate=false]
     */
    public function getAll() {
        $req         = $this->getRequest();
        $searchQuery = $req->getParam("query");
        $pageSize    = min((int)$req->getQuery("limit", "20"), 100);
        $page        = max((int)$req->getQuery("page", "1"), 1);
        $aggregate   = trim(strtolower($req->getQuery("aggregate"))) === "true";

        if ( ! $searchQuery) {
            // TODO: Error code
            throw Crossroads_API_ResponseException::create(400, "Missing 'query' parameter.", null);
        }

        if( ! $aggregate && $page < 2) {
            $mageQuery = $this->getQueryModel($searchQuery);
        }

        $store       = Mage::app()->getStore();
        $helper      = Mage::helper("Crossroads_Elasticsearch");
        $factory     = Mage::getModel("API/factory");
        $queryParser = Mage::getModel("Crossroads_Elasticsearch/query_parser");
        $resource    = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $client      = $helper->createClient();
        $attributes  = $resource->getSearchableProductAttributes($store);
        $bucketOrd   = $resource->getSortedAttributeValues($attributes, $store);
        $filterAttrs = array_filter($attributes, function($a) { return $a->getIsFilterableInSearch() && $a->hasKeywordField(); });
        $sortAttrs   = array_filter($attributes, function($a) { return $a->getUsedForSortBy(); });
        $params      = new Varien_Object(array_filter(array_diff_key($req->getParams(), [
            "query"     => true,
            "limit"     => true,
            "page"      => true,
            "aggregate" => true,
        ])));

        $parsedQuery = $queryParser->setAttributes($attributes)
            ->setLocaleCode(Mage::app()->getLocale()->getLocaleCode())
            ->parseQuery($searchQuery);

        $q = $helper->createProductQuery($store, Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_CATALOG)
                ->setSearchQuery($parsedQuery);

        Mage::dispatchEvent(self::EVENT_SEARCH_QUERY, [
            "params" => $params,
            "query"  => $q,
        ]);

        $params = $params->getData();
        $clone = clone $q;
        $query = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->addSource([ "sku", "name" ])
            ->setPageSize($pageSize)
            ->setPage($page)
            ->setQuery($clone->setFacetFilter($filterAttrs, $params));

        if($aggregate) {
            $query->setAggregations(Mage::getModel("Crossroads_Elasticsearch/query_aggregation_attribute")
                ->setAttributes($filterAttrs)
                ->setParameters($params)
                ->setExcludeQueryConstructor(function($paramKey) use($filterAttrs, $q, $params) {
                    $clone = clone $q;

                    return $clone->setFacetFilter($filterAttrs, array_diff_key($params, [ $paramKey => true ]));
                }));
        }

        if($req->getQuery("sort")) {
            foreach(array_reverse(array_filter(array_map("trim", explode(",", $req->getQuery("sort"))))) as $sort) {
                $query->addAttributeSort($sortAttrs, $sort);
            }
        }

        $result = $client->search($query->toRequest());

        $numProducts = $result["hits"]["total"];
        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        if( ! $aggregate && $page < 2) {
            // Update result count
            $mageQuery->setNumResults($result["hits"]["total"]);
            $mageQuery->save();
        }

        return [200, [
            // TODO: Reuse this between search, recommended, faceted. Maybe create some sharing through collection model(s)?
            "products"   => $factory->createProductListSerializer($store)->mapArray(
                $helper->fetchProducts($entityIds, $pageSize, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG)),
            "query"      => $searchQuery,
            "totalCount" => $result["hits"]["total"],
            // New stuff compared to original search:
            "facets"     => $aggregate && array_key_exists("aggregations", $result) ? $helper->transformFacets($result["aggregations"], $filterAttrs, $params, $bucketOrd, $numProducts) : null,
            "_time"      => $result["took"],
            "total"      => $result["hits"]["total"],
        ]];
    }

    public function getItem($id) {
        switch($id) {
        case "suggest":
            return $this->doSuggest();
        default:
            return [404];
        }
    }

    protected function doSuggest() {
        $searchQuery = $this->getRequest()->getParam("query");
        $limit       = min((int)$this->getRequest()->getParam("limit") ?: 10, 20);
        $store       = Mage::app()->getStore();
        $attributes  = Mage::getResourceModel("Crossroads_Elasticsearch/attribute")->getAutocompleteAttributes($store);
        $helper      = Mage::helper("Crossroads_Elasticsearch");
        $client      = $helper->createClient();
        $indexName   = $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX);

        if(empty($attributes)) {
            return [200, [
                "suggestions" => [],
                "_reason"     => "No available attributes.",
            ]];
        }

        $result = $client->search([
            "index" => $indexName,
            "type"  => "product",
            "body"  => [
                "_source" => [
                    "_score"
                ],
                "suggest" => array_merge([
                    "text" => $searchQuery,
                ], array_map(function($attr) use($limit, $searchQuery) {
                    return [
                        "prefix"     => $searchQuery,
                        "completion" => [
                            "field"  => $attr->getCompletionField(),
                            "skip_duplicates" => true,
                            "fuzzy" => [
                                "fuzziness"  => "AUTO",
                                "min_length" => 5,
                            ],
                            "size" => $limit,
                        ],
                    ];
                }, array_combine(array_map(function($a) { return $a->getCode(); }, $attributes), $attributes))),
            ]
        ]);

        return [200, $this->prepareAutocomplete($attributes, $searchQuery, $result, $limit)];
    }

    protected function prepareAutocomplete($attributes, $searchQuery, $data, $limit) {
        $response    = [];
        $keyedAttrs  = array_combine(array_map(function($a) { return $a->getCode(); }, $attributes), $attributes);
        $filterAttrs = array_filter($keyedAttrs, function($a) { return $a->getIsFilterableInSearch() && $a->isText(); });

        foreach($data["suggest"] as $key => $result) {
            foreach($result as $inner) {
                foreach($inner["options"] as $opt) {
                    $response[] = [
                        "text"  => $opt["text"],
                        "score" => $opt["_score"],
                        "key"   => array_key_exists($key, $filterAttrs) ? $key : "query",
                        "label" => array_key_exists($key, $filterAttrs) ? $filterAttrs[$key]->getFrontendLabel() : null,
                    ];
                }
            }
        }

        usort($response, function($a, $b) {
            // Reverse sort, higher score is better
            return $b["score"] - $a["score"];
        });

        return [
            "suggestions" => $response,
            "query"       => $searchQuery,
        ];
    }
}
