<?php

class Crossroads_Elasticsearch_Model_Product_Index {
    /**
     * Data:
     *  * mapping:    Varien_Object containing key-value mapping of the index
     *  * locale:     Locale code
     *  * attributes: List of attributes
     */
    const EVENT_MAPPINGS = "crossroads_elasticsearch_product_index_mappings";

    /**
     * Ensures that a given store has an alias backed with an index, returning the index name to use
     * as an index target.
     *
     * @param  Mage_Core_Model_Store
     * @param  Array<Crossroads_Elasticsearch_Model_Attribute>
     * @param  Elasticsearch
     * @return string
     */
    public function ensureAlias($store, $attributes, $client) {
        $aliasName = $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX);

        if( ! $client->indices()->existsAlias([ "name" => $aliasName ])) {
            $indexName = $this->createIndex($store, $attributes, $client);

            $client->indices()->putAlias([
                "name"  => $aliasName,
                "index" => $indexName
            ]);

            return $indexName;
        }

        $aliases = $client->indices()->getAlias([ "name" => $aliasName ]);

        // Extract actual index from alias
        foreach($aliases as $index => $aliasMapping) {
            if(array_key_exists($aliasName, $aliasMapping['aliases'])) {
                return $index;
            }
        }

        throw new Exception(sprintf("Crossroads_Elasticsearch: Expected to find index aliased by %s.", $aliasName));
    }


    public function createIndex($store, $attributes, $client) {
        $indexBaseName = $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX);
        $i             = 0;

        do {
            $indexName = sprintf("%s_%s_v%d", $indexBaseName, gmdate("Y-m-d"), ++$i);
        }
        while($client->indices()->exists([ "index" => $indexName ]));

        $client->indices()->create([
            "index" => $indexName,
            "body"  => [
                "settings" => [
                    "analysis"           => $this->getAnalysis($store, $attributes),
                    "number_of_shards"   => $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_INDEX_SHARDS),
                    "number_of_replicas" => $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_INDEX_REPLICAS),
                ],
                "mappings" => [
                    "product" => $this->getMappings($store, $attributes),
                ],
            ],
        ]);

        return $indexName;
    }

    public function updateAlias($store, $indexName, $client) {
        $aliasName = $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX);
        $actions   = [];

        if($client->indices()->existsAlias([ "name" => $aliasName ])) {
            $alias = $client->indices()->getAlias([ "name" => $aliasName ]);

            if($alias) {
                foreach(array_keys($alias) as $index) {
                    $actions[] = [ "remove" => [ "index" => $index, "alias" => $aliasName ] ];
                }
            }
        }

        $actions[] = [ "add" => [ "index" => $indexName, "alias" => $aliasName ] ];

        $client->indices()->updateAliases([ "body" => [ "actions" => $actions ] ]);
    }

    protected function getAnalysis($store, $attributes) {
        return [
            "char_filter" => [
                "sku_pattern_replace_filter" => [
                    "type"        => "pattern_replace",
                    "pattern"     => "(\\s|[-_\/\.\(\)])*",
                    "replacement" => ""
                ],
            ],
            "analyzer" => [
                "sku_analyzer" => [
                    "type"        => "custom",
                    "tokenizer"   => "keyword",
                    "char_filter" => ["sku_pattern_replace_filter"],
                    "filter"      => []
                ],
                "trigram" => [
                    "type"      => "custom",
                    "tokenizer" => "standard",
                    "filter"    => ["standard", "shingle"],
                ],
            ],
            "filter" => [
                "shingle" => [
                    "type"  => "shingle",
                    "min_shingle_size" => 2,
                    "max_shingle_size" => 3,
                ],
            ],
        ];
    }

    protected function getMappings($store, $attributes) {
        $locale = $store->getConfig(Mage_Core_Model_Locale::XML_PATH_DEFAULT_LOCALE);

        // Default props
        $props = new Varien_Object([
            // Child/Parent differentiating type
            "object_type" => [
                "type" => "join",
                "relations" => [
                    "product" => "order",
                ],
            ],

            // Product
            "store_id"  => [ "type" => "keyword" ],
            "entity_id" => [ "type" => "keyword" ],
            "type_id"   => [ "type" => "keyword", "index" => false ],
            "sku" => [
                "type"     => "text",
                "analyzer" => "sku_analyzer",
                "copy_to"  => "_all",
            ],
            "visibility" => [ "type" => "keyword" ],
            "website_id" => [ "type" => "keyword", "index" => false ],
            "updated_at" => [
                "type"    => "date",
                "copy_to" => "_all",
                "format"  => Crossroads_Elasticsearch_Model_Attribute::MYSQL_DATE_FORMAT,
            ],
            "created_at" => [
                "type"    => "date",
                "copy_to" => "_all",
                "format"  => Crossroads_Elasticsearch_Model_Attribute::MYSQL_DATE_FORMAT,
            ],
            "elasticsearch_limit_customers" => [
                "type" => "keyword",
            ],

            // Recommendation-specific stuff
            Crossroads_Elasticsearch_Helper_Data::FIELD_RECOMMENDED => [
                "type"  => "keyword",
            ],
            "xes_recommended_skus" => [
                "type"  => "keyword",
            ],
            "customer_group_id" => [
                "type"  => "keyword",
            ],
        ]);

        foreach($attributes as $attr) {
            $props->setData($attr->getCode(), $attr->getMapping($locale));
        }

        Mage::dispatchEvent(self::EVENT_MAPPINGS, [
            "mapping"    => $props,
            "locale"     => $locale,
            "attributes" => $attributes,
        ]);

        return [
            "properties" => $props->getData()
        ];
    }
}
