<?php

class Crossroads_Elasticsearch_Model_Resource_Product_Recommended extends Crossroads_Elasticsearch_Model_Resource_Abstract {
    const SEPARATOR               = "±";
    const EVENT_POST_CREATE_QUERY = "crossroads_elasticsearch_index_recommended_post_create_query";

    protected function _construct() {
        $this->_setResource("core_read");
    }

    public function addMapping($event) {
        $mapping = $event->getMapping();

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_RECOMMENDED_SKUS, [
            "type"  => "keyword",
            "index" => true,
        ]);

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_RECOMMENDED, [
            "type"  => "keyword",
            "index" => true,
        ]);

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_CUSTOMER_ID, [
            "type"  => "keyword",
            "index" => true,
        ]);

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_CUSTOMER_GROUP_ID, [
            "type"  => "keyword",
            "index" => true,
        ]);
    }

    public function indexAll($event) {
        $manager = $event->getManager();
        $indices = $event->getIndices();
        $stores  = $event->getStores();
        $types   = $event->getTypes();
        $sql     = $this->getReadConnection();
        $query   = $this->createRelatedProductsQuery($sql, $stores);
        $stmt    = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    public function indexProducts($event) {
        $productIds = $event->getProductIds();
        $manager    = $event->getManager();
        $indices    = $event->getIndices();
        $stores     = $event->getStores();
        $types      = $event->getTypes();
        $sql        = $this->getReadConnection();
        $query      = $this->createRelatedProductsQuery($sql, $stores, $productIds);
        $stmt       = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    protected function loadData($indices, $types, $manager, $stmt) {
        foreach($stmt as $row) {
            $doc = array_filter(array_merge([
                "object_type" => [
                    "name"   => "order",
                    "parent" => (string)$row["product_id"],
                ],
            ], array_map(function($e) {
                return array_unique(array_values(array_filter(explode(self::SEPARATOR, $e))));
            }, array_diff_key($row, [
                "product_id"  => true,
            ]))));

            if(empty($doc)) {
                continue;
            }

            // TODO: Store data in another index to to do aggregates on
            $manager->queue([ "index" => [
                "_index"   => $indices[$row["store_id"]],
                "_routing" => $row["product_id"],
                // A bit WTF, but ElasticSearch 6 only allows one type per index
                "_type"    => "product",
                "_id"      => $row["product_id"]."__".$row["order_id"],
            ]], $doc);
        }
    }

    public function createRelatedProductsQuery($sql, $stores, $productIds = null) {
        $daysToConsider = Mage::app()->getStore()->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_DAYS_CONSIDERED);
        $select = $sql->select();

        $select->from(["o" => $this->getTable("sales/order")], [
            "store_id" => "store_id",
            "order_id" => "entity_id",
            "created_at" => "created_at",
            "customer_id" => "customer_id",
        ]);

        $select->join(["i" => $this->getTable("sales/order_item")], "i.order_id = o.entity_id", [
            "product_id",
        ]);

        $select->columns([
            Crossroads_Elasticsearch_Helper_Data::FIELD_CUSTOMER_GROUP_ID => new Zend_Db_Expr("COALESCE(o.customer_group_id, 0)"),
            Crossroads_Elasticsearch_Helper_Data::FIELD_RECOMMENDED => "(SELECT GROUP_CONCAT(DISTINCT j.product_id SEPARATOR '±') FROM sales_flat_order_item j WHERE j.order_id = o.entity_id AND j.product_id <> i.product_id)",
            Crossroads_Elasticsearch_Helper_Data::FIELD_RECOMMENDED_SKUS => "(SELECT GROUP_CONCAT(DISTINCT j.sku SEPARATOR '±') FROM sales_flat_order_item j WHERE j.order_id = o.entity_id AND j.product_id <> i.product_id)",
        ]);

        $select->where("i.updated_at > DATE_SUB(NOW(), INTERVAL ? DAY)", $daysToConsider);

        Mage::dispatchEvent(self::EVENT_POST_CREATE_QUERY, [
            "resource"   => $this,
            "sql"        => $sql,
            "query"      => $select,
            "stores"     => $stores,
            "productIds" => $productIds,
        ]);

        return $select;
    }
}
