<?php

class Crossroads_Elasticsearch_Model_Query_Products {
    const VISIBILITY_CATALOG = "CATALOG";
    const VISIBILITY_SEARCH  = "SEARCH";

    protected $should = [
        ["term" => ["visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH]],
    ];

    protected $visibility = self::VISIBILITY_CATALOG;

    protected $filter = [
        [ "term" => [ "object_type" => "product" ] ]
    ];

    /** Facet filters */
    protected $facetFilters = [];

    protected $query = null;

    protected $customerId = null;

    /**
     * Set visibilty for products, if they are being shown in catalog or search. Default is catalog.
     *
     * Always includes BOTH.
     *
     * @param  string   One of the VISIBILITY_* constants
     */
    public function setVisibility($visibility) {
        $this->visibility = $visibility;

        return $this;
    }

    public function setStore($store) {
        if( ! $store->getConfig(Mage_CatalogInventory_Helper_Data::XML_PATH_SHOW_OUT_OF_STOCK)) {
            $this->filter[] = ["term" => [ Crossroads_Elasticsearch_Helper_Data::FIELD_IN_STOCK => true ]];
        }

        return $this;
    }

    public function setCustomerId($customerId) {
        $this->customerId = $customerId;

        return $this;
    }

    public function addFilter($filter) {
        $this->filter[] = $filter;

        return $this;
    }

    public function addFacetFilter($filter) {
        $this->facetFilters[] = $filter;

        return $this;
    }

    /**
    * Attributes should be fetched with the extra data for indexing status.
     */
    public function setFacetFilter($attributes, $params) {
        $filters = [];
        $wrong   = array_diff(array_keys($params), array_map(function($a) {
            return $a->getCode();
        }, array_filter($attributes, function($a) {
            return $a->hasKeywordField();
        })));

        if( ! empty($wrong)) {
            throw Crossroads_API_ResponseException::create(400, sprintf("Unknown parameter keys '%s'.", implode("', '", $wrong)));
        }

        foreach($attributes as $attr) {
            if(array_key_exists($attr->getCode(), $params) && $attr->hasKeywordField()) {
                $data     = $params[$attr->getCode()];
                $operator = $attr->getOperator();

                // Group filters with OR if they are arrays
                $filters[] = [ "bool" => [ $operator => array_map(function($d) use($attr) {
                    // Term queries for exact match filters
                    return [ "term" => [ $attr->getKeywordField() => (string)$d ] ];
                }, is_array($data) ? $data : [$data]) ] ];
            }
        }

        $this->facetFilters = array_merge($this->facetFilters, $filters);

        return $this;
    }

    public function setSearchQuery($query) {
        $this->query = $query;

        return $this;
    }

    protected function getShould() {
        $should = $this->should;

        switch($this->visibility) {
        case self::VISIBILITY_SEARCH:
            $should[] = ["term" => ["visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG]];

            break;
        case self::VISIBILITY_CATALOG:
            $should[] = ["term" => ["visibility" => Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG]];

            break;
        default:
            throw new Exception(sprintf("Invalid visibility '%s'.", $this->visibility));
        }

        return $should;
    }

    public function toRequest() {
        $query = [
            "filter"               => $this->filter,
            "should"               => $this->getShould(),
            "minimum_should_match" => 1,
        ];

        if( ! $this->query) {
            $query["must"] = $this->facetFilters;
        }
        else {
            $query["must"]   = $this->query;
            $query["filter"] = array_merge($query["filter"], $this->facetFilters);
        }

        return [
            "bool" => $query,
        ];
    }
}
