<?php

class Crossroads_Elasticsearch_Model_Query_Products_Recommended extends Crossroads_Elasticsearch_Model_Query_Products {
    protected $skus       = [];
    protected $complete   = false;
    protected $boosts     = [];
    protected $gaussScale = "365d";
    protected $minOrders  = 2;
    /**
     * Extra filter to optionally match.
     */
    protected $extraMust  = [];

    /**
     * Adds a term boost filter with a multiplier weight, if the term filter matches the score
     * will be multiplied by `$boost`.
     *
     * @param  array    Term filter (eg. [ "column" => value ]
     * @param  integer  Multiplier if the condition matches
     */
    public function addBoostFilter($filter, $boost) {
        $this->boosts[] = [
            "filter" => [ "term" => $filter ],
            "weight" => (int)$boost,
        ];

        return $this;
    }

    /**
     * Sets a list of SKUs which are to be used in the order filter, adding SKUs here will
     * result in only products being recommended which have been part of an order containing
     * at least one of the supplied SKUs.
     *
     * @param  Array<string>
     */
    public function setSkus($skus) {
        $this->skus = $skus;

        return $this;
    }

    /**
     * Appends the supplied SKUs to the list of skus which are to be used in the order filter,
     * see `setSkus`.
     *
     * @param  Array<string>
     */
    public function addSkus($skus) {
        $this->skus = array_unique(array_merge($this->skus, $skus));

        return $this;
    }

    /**
     * If set to true the order filter will not be wrapped in a constant score query,
     * this means that the filter will rank products contained in orders having most of the
     * supplied SKUs higher than orders just containing some of them. Default is false.
     *
     * @param  boolean
     */
    public function setComplete($flag) {
        $this->complete = $flag;

        return $this;
    }

    /**
     * Sets the middle of the slope for the gauss curve in days for date scaling.
     *
     * @param  integer
     */
    public function setGaussScale($days) {
        $this->gaussScale = sprintf("%dd", $days);

        return $this;
    }

    /**
     * Sets the number of minimum orders having to match for a recommendation to count.
     *
     * @param  integer
     */
    public function setMinimumOrders($num) {
        $this->minOrders = (int)$num;

        return $this;
    }

    public function toRequest() {
        $facetFilters = $this->facetFilters;

        if( ! empty($this->skus)) {
            $facetFilters[] = [
                "bool" => [
                    "must_not" => [
                        "terms" => [
                            "sku" => $this->skus,
                        ]
                    ]
                ]
            ];
        }

        $query = [
            "must" => [ [ "bool" => [
                "should" => array_merge($this->extraMust, [ [ "has_child" => [
                    "type"         => "order",
                    "score_mode"   => "sum",
                    "min_children" => $this->minOrders,
                    "query"        => $this->getChildQuery(),
                ] ] ]),
            ] ] ],
            "should"               => $this->getShould(),
            "filter"               => array_merge($this->filter, $facetFilters),
            "minimum_should_match" => 1,
        ];

        return [ "bool" =>  $query ];
    }

    protected function getChildQuery() {
        // We wrap it into a constant score question to make sure we sidestep the
        // Term Frequency/Inverse Document Frequency (TF/IDF) scoring since it will
        // regard large orders as irrelevant.
        $term = [ "terms" => [ Crossroads_Elasticsearch_Helper_Data::FIELD_RECOMMENDED_SKUS => $this->skus ] ];

        $all = empty($this->skus) ? [ "match_all" => new stdClass ] : ($this->complete ? $term : [ "constant_score" => [ "filter" => $term ] ]);

        // We apply a function query on it if we have a customer group
        return [
            "function_score" => [
                "query"     => $all,
                "functions" => $this->getDecayFunctions(),
            ],
        ];
    }

    protected function getDecayFunctions() {
        // TODO: Make it possible to boost products themselves
        return array_merge([
            // Decay order relevance over time
            [ "gauss" => [
                "created_at" => [
                    "origin" => gmdate("Y-m-d G:i:s"),
                    "scale"  => $this->gaussScale,
                ],
            ] ],
        ], $this->boosts);
    }

    public function addExtraMatch($match) {
        $this->extraMust[] = $match;

        return $this;
    }
}
