<?php

/**
 * Iterator over a result set of product attributes, returning product entities as arrays.
 *
 * Expects data from the query created by Crossroads_Elasticsearch_Model_Resource_Product::createProductQuery
 */
class Crossroads_Elasticsearch_Model_Resource_Product_DataIterator implements Iterator {
    protected $stmt;
    protected $attributes;
    protected $nextRow = null;
    protected $entity  = null;
    protected $mLUT    = [];

    /**
     * @param  Zend_Db_Statement_Interface  Result-set of product attribute data
     * @param  array<Crossroads_Elasticsearch_Model_Attribute>  Associative array of attribute data (key = attribute_id)
     * @param  Array<>
     */
    public function __construct(Zend_Db_Statement_Interface $stmt, $attributes, $multiselectLUT) {
        $this->stmt  = $stmt;
        $this->attrs = $attributes;
        $this->mLUT  = $multiselectLUT;
    }

    protected function loadNext() {
        $row = $this->nextRow ?: $this->stmt->fetch(PDO::FETCH_ASSOC);

        if( ! $row) {
            return null;
        }

        $storeId  = $row["store_id"];
        $entityId = $row["entity_id"];
        $entity   = [
            "object_type" => [ "name" => "product" ],
            "store_id"    => $row["store_id"],
            "entity_id"   => $row["entity_id"],
            "type_id"     => $row["type_id"],
            "sku"         => $row["sku"],
            "visibility"  => $row["visibility"],
            "created_at"  => $row["created_at"],
            "updated_at"  => $row["updated_at"],
            "website_id"  => $row["website_id"],
        ];

        $entity = $this->decodeRow($entity, $row, $this->attrs[$row["attribute_id"]]);

        while($row = $this->stmt->fetch(PDO::FETCH_ASSOC)) {
            if($row["store_id"] !== $storeId || $row["entity_id"] !== $entityId) {
                $this->nextRow = $row;

                return $this->finalizeEntity($entity);
            }

            $entity = $this->decodeRow($entity, $row, $this->attrs[$row["attribute_id"]]);
        }

        $this->nextRow = null;

        return $this->finalizeEntity($entity);
    }

    protected function decodeRow($entity, $row, $attr) {
        $store_id       = $entity["store_id"];
        $frontend_input = $attr->getFrontendInput();
        $attribute_code = $attr->getCode();
        $backend_type   = $attr->getBackendType();

        switch($frontend_input) {
        case "select":
        case "multiselect":
            if($attr->getBackendModel() !== "eav/entity_attribute_backend_array") {
                $value = $row["option_value"];

                break;
            }
            else {
                // Intentionally left empty, eav/entity_attribute_backend_array stores its
                // data as a comma-separated array of ids in the backend_type column instead
                // of any attribute options. Fall through on purpose.
            }
        default:
            $value = $row[$backend_type];
        }

        if($attr->getIsHtml()) {
            $value = html_entity_decode(strip_tags($value));
        }

        $value = trim($value);

        if(strlen($value) === 0) {
            return $entity;
        }

        if($frontend_input === "multiselect") {
            $arr = ! empty($entity[$attribute_code]) ? $entity[$attribute_code] : [];

            if($attr->getBackendModel() === "eav/entity_attribute_backend_array") {
                $ids   = array_map("intval", array_map("trim", explode(",", $value)));
                $value = [];

                foreach($ids as $id) {
                    if(array_key_exists((int)$store_id, $this->mLUT) &&
                       array_key_exists($attr->getId(), $this->mLUT[$store_id]) &&
                       array_key_exists($id, $this->mLUT[$store_id][$attr->getId()])) {
                        $value[$id] = $this->mLUT[$store_id][$attr->getId()][$id];
                    }
                }
            }
            else {
                $value = [$row["option_value_id"] => $value];
            }

            foreach($value as $k => $v) {
                // Prevent duplicates by using the value id
                $arr[$k] = $v;
            }

            $entity[$attribute_code] = $arr;
        }
        // Only insert if it is non-empty, we run this as an index which will replace
        // documents.
        else if($value !== "") {
            $entity[$attribute_code] = $value;
        }

        return $entity;
    }

    protected function finalizeEntity($entity) {
        foreach($this->attrs as $attr) {
            if($attr->getFrontendInput() === "multiselect") {
                $attribute_code = $attr->getCode();

                // We have to just get the values of the multiselect, we store ids to prevent multiple selects
                // of the same value from being entered
                if( ! empty($entity[$attribute_code])) {
                    $entity[$attribute_code] = array_filter(array_values($entity[$attribute_code]));
                }
                else {
                    unset($entity[$attribute_code]);
                }
            }
        }

        if(array_key_exists("elasticsearch_limit_customers", $entity)) {
            $entity["elasticsearch_limit_customers"] = array_filter(array_map("trim", explode(",", $entity["elasticsearch_limit_customers"])));
        }

        // If we have an autocomplete, make sure we *never* have zero data in an existing key
        foreach($this->attrs as $attr) {
            if($attr->getAutocomplete() && empty($entity[$attr->getCode()])) {
                unset($entity[$attr->getCode()]);
            }
        }

        return $entity;
    }

    public function current() {
        return $this->entity;
    }

    public function key() {
        return $this->entity["entity_id"];
    }

    public function next() {
        $this->entity = $this->loadNext();
    }

    public function rewind() {
        $this->entity = $this->loadNext();
    }

    public function valid() {
        return (bool)$this->entity;
    }
}
