<?php

class Crossroads_Elasticsearch_Model_Resource_Product_Purchases extends Crossroads_Elasticsearch_Model_Resource_Abstract {
    protected function _construct() {
        $this->_setResource("core_read");
    }

    public function addMapping($event) {
        $mapping = $event->getMapping();

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_PURCHASES, [
            "type"       => "integer",
            "null_value" => 0,
        ]);

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_POPULARITY, [
            "type"       => "double",
            "null_value" => 0,
        ]);
    }

    public function indexAll($event) {
        $manager = $event->getManager();
        $indices = $event->getIndices();
        $stores  = $event->getStores();
        $types   = $event->getTypes();
        $sql     = $this->getReadConnection();
        $query   = $this->createRelatedProductsQuery($sql, $stores);
        $stmt    = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    public function indexProducts($event) {
        $productIds = $event->getProductIds();
        $manager    = $event->getManager();
        $indices    = $event->getIndices();
        $stores     = $event->getStores();
        $types      = $event->getTypes();
        $sql        = $this->getReadConnection();
        $query      = $this->createRelatedProductsQuery($sql, $stores, $productIds);
        $stmt       = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    protected function loadData($indices, $types, $manager, $stmt) {
        foreach($stmt as $row) {
            $doc = array_filter([
                Crossroads_Elasticsearch_Helper_Data::FIELD_PURCHASES  => (int)$row["purchases"],
                Crossroads_Elasticsearch_Helper_Data::FIELD_POPULARITY => (double)$row["popularity"],
            ]);

            if(empty($doc)) {
                continue;
            }

            $manager->queue([ "update" => [
                "_index"   => $indices[$row["store_id"]],
                "_routing" => (string)$row["product_id"],
                "_type"    => "product",
                "_id"      => (string)$row["product_id"],
            ]], [ "doc" => $doc ]);
        }
    }

    public function createRelatedProductsQuery($sql, $stores, $productIds = null) {
        $daysToConsider = Mage::app()->getStore()->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_DAYS_CONSIDERED);
        $twoSigmaSquare = -(((double)$daysToConsider) ^ 2) / (2 * log(0.5));

        // We start with the sales_flat_order_items to prevent a lot of unnecessary temporaries, we already
        // have store_id and product_id in it
        $select = $sql->select()
            ->from(["i" => $this->getTable("sales/order_item")], ["store_id", "product_id"])
            ->join(["p" => $this->createEntityIdStoreQuery($sql, $stores, $productIds)], "p.entity_id = i.product_id AND p.store_id = i.store_id", [])
            ->where("i.created_at >= ?", date_format(date_sub(date_create(), date_interval_create_from_date_string("$daysToConsider days")), "Y-m-d"));

        $select = $select->columns([
                "purchases"   => "COUNT(DISTINCT i.order_id)",
                "popularity"  => "SUM(EXP( - POW(GREATEST(DATEDIFF(NOW(), i.created_at), 0), 2) / $twoSigmaSquare))",
            ]);

        return $select->group(["i.store_id", "i.product_id"]);
    }
}
