<?php

class Crossroads_Elasticsearch_Model_Indexer_Product extends Mage_Index_Model_Indexer_Abstract {
    const UPDATE_KEY                  = "Crossroads_Elasticsearch_update_product_id";
    const DELETE_KEY                  = "Crossroads_Elasticsearch_delete_product_id";
    const MASS_ACTION_PRODUCT_IDS_KEY = "Crossroads_Elasticsearch_mass_action_product_ids";
    const MASS_ACTION_TYPE_KEY        = "Crossroads_Elasticsearch_mass_action_type";

    // TODO: Make sure we match the category-product link update
    protected $_matchedEntities = [
        Mage_CatalogInventory_Model_Stock_Item::ENTITY => array(
            Mage_Index_Model_Event::TYPE_SAVE
        ),
        Mage_Catalog_Model_Product::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_DELETE,
            Mage_Index_Model_Event::TYPE_MASS_ACTION,
        ],
        // We need all the ones below since they affect the indexed data
        Mage_Catalog_Model_Resource_Eav_Attribute::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE,
            Mage_Index_Model_Event::TYPE_DELETE,
        ],
        Mage_Core_Model_Config_Data::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE
        ],
        Mage_Catalog_Model_Category::ENTITY => [
            Mage_Index_Model_Event::TYPE_SAVE
        ]
    ];

    public function getName() {
        return "Crossroads ElasticSearch Products";
    }

    public function getDescription() {
        return "Index process for indexing products in ElasticSearch";
    }

    protected function _registerEvent(Mage_Index_Model_Event $event) {
        switch($event->getEntity()) {
        case Mage_Catalog_Model_Product::ENTITY:
            $this->registerProduct($event);
            break;
        case Mage_Catalog_Model_Resource_Eav_Attribute::ENTITY:
            $this->registerAttribute($event);
            break;
        case Mage_Core_Model_Config_Data::ENTITY:
            $this->registerConfig($event);
            break;
        case Mage_Catalog_Model_Category::ENTITY:
            $this->registerCategory($event);
            break;
        case Mage_CatalogInventory_Model_Stock_Item::ENTITY:
            $this->registerStockItem($event);
            break;
        }
    }

    protected function registerProduct($event) {
        $dataObject = $event->getDataObject();

        switch($event->getType()) {
        case Mage_Index_Model_Event::TYPE_SAVE:
            $event->addNewData(self::UPDATE_KEY, $dataObject->getEntityId());
            break;
        case Mage_Index_Model_Event::TYPE_DELETE:
            $event->addNewData(self::DELETE_KEY, $dataObject->getEntityId());
            break;
        case Mage_Index_Model_Event::TYPE_MASS_ACTION:
            $event->addNewData(self::MASS_ACTION_PRODUCT_IDS_KEY, $dataObject->getProductIds());
            $event->addNewData(self::MASS_ACTION_TYPE_KEY, $dataObject->getActionType());
            break;
        }
    }

    protected function registerAttribute($event) {
        $attr = $event->getDataObject();

        // We do not need to consider crossroads_elasticsearch_boost since it is only a query-time
        // configuration
        if($attr->dataHasChangedFor("is_filterable_in_search") ||
           $attr->dataHasChangedFor("is_visible_in_advanced_search") ||
           $attr->dataHasChangedFor("is_searchable") ||
           $attr->dataHasChangedFor("frontend_input") ||
           $attr->dataHasChangedFor("backend_type") ||
           // is_html:
           $attr->dataHasChangedFor("is_html_allowed_on_front") ||
           $attr->dataHasChangedFor("is_wysiwyg_enabled")) {
            $process = $event->getProcess();
            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);
        }
    }

    protected function registerConfig($event) {
        if(stripos($event->getDataObject()->getPath(), "crossroads_elasticsearch") === 0) {
            $process = $event->getProcess();
            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);
        }
    }

    protected function registerCategory($event) {
        if($event->getType() !== Mage_Index_Model_Event::TYPE_SAVE) {
            return;
        }

        $category   = $event->getDataObject();
        $productIds = $category->getAffectedProductIds();

        if($productIds) {
            $event->addNewData(self::MASS_ACTION_PRODUCT_IDS_KEY, $productIds);
            $event->addNewData(self::MASS_ACTION_TYPE_KEY, "save");
        }
        else {
            $process = $event->getProcess();
            $process->changeStatus(Mage_Index_Model_Process::STATUS_REQUIRE_REINDEX);
        }
    }

    protected function registerStockItem(Mage_Index_Model_Event $event) {
        if($event->getType() !== Mage_Index_Model_Event::TYPE_SAVE) {
            return;
        }

        $event->addNewData(self::UPDATE_KEY, $event->getDataObject()->getProductId());
    }

    public function _processEvent(Mage_Index_Model_Event $event) {
        $dataObject = $event->getNewData();
        $resource   = Mage::getResourceModel("Crossroads_Elasticsearch/product");

        if( ! empty($dataObject[self::UPDATE_KEY])) {
            $resource->indexProducts([$dataObject[self::UPDATE_KEY]]);
        }

        if( ! empty($dataObject[self::DELETE_KEY])) {
            $resource->deleteProducts([$dataObject[self::DELETE_KEY]]);
        }

        if( ! empty($dataObject[self::MASS_ACTION_PRODUCT_IDS_KEY])) {
            switch($dataObject[self::MASS_ACTION_TYPE_KEY]) {
            case "remove":
                $resource->deleteProducts($dataObject[self::MASS_ACTION_PRODUCT_IDS_KEY]);
                break;
            default:
                $resource->indexProducts($dataObject[self::MASS_ACTION_PRODUCT_IDS_KEY]);
            }
        }
    }

    public function reindexAll() {
        Mage::getResourceModel("Crossroads_Elasticsearch/product")->reindexAll();
    }
}
