<?php

use Elasticsearch\ClientBuilder;

class Crossroads_Elasticsearch_Helper_Data extends Mage_Core_Helper_Abstract {
    const CONFIG_NODES                  = "crossroads_elasticsearch/cluster/nodes";
    const CONFIG_ENABLED                = "crossroads_elasticsearch/cluster/enabled";
    const CONFIG_PRODUCT_INDEX          = "crossroads_elasticsearch/index/product";
    const CONFIG_INDEX_SHARDS           = "crossroads_elasticsearch/index/num_shards";
    const CONFIG_INDEX_REPLICAS         = "crossroads_elasticsearch/index/num_replicas";
    const CONFIG_FACET_COMBINE_OPERATOR = "crossroads_elasticsearch/products/facet_combine_operator";
    const CONFIG_PRODUCT_RECOMMENDED    = "crossroads_elasticsearch/products/num_recommended";
    const CONFIG_DAYS_CONSIDERED        = "crossroads_elasticsearch/recommendations/days_considered";
    const CONFIG_MIN_ORDERS             = "crossroads_elasticsearch/recommendations/minimum_orders";
    const CONFIG_GROUP_MULTIPLIER       = "crossroads_elasticsearch/recommendations/matching_customer_group_mult";

    /**
     * Contains a list of category names, translated to the appropriate store.
     */
    const FIELD_CATEGORIES  = "xes_categories";
    /**
     * Contains a list of the positions of the product in different categories, no relation to any specific category.
     */
    const FIELD_POSITIONS   = "xes_positions";
    /**
     * Constains a list of category id paths, like the path column from magento categories.
     */
    const FIELD_PATHS       = "xes_paths";
    /**
     * Constains a list of entity ids which are usually bought together.
     */
    const FIELD_RECOMMENDED = "xes_recommended";
    /**
     * Constains a list of SKUs which are usually bought together.
     */
    const FIELD_RECOMMENDED_SKUS = "xes_recommended_skus";
    /**
     * Contains the number of times this product has been purchased.
     */
    const FIELD_PURCHASES   = "xes_purchases";
    /**
     * Contains a scaled popularity score.
     */
    const FIELD_POPULARITY  = "xes_popularity";
    /**
     * Contains a boolean flag telling if the item is in stock or not.
     */
    const FIELD_IN_STOCK    = "xes_in_stock";
    /**
     * Contains the total number of items in stock for the given product (including children).
     */
    const FIELD_STOCK       = "xes_stock";

    const FIELD_CUSTOMER_ID       = "customer_id";
    const FIELD_CUSTOMER_GROUP_ID = "customer_group_id";


    /**
     * Event fired for each created and prepared product query (Crossroads_Elasticsearch_Model_Query_Products).
     *
     * NOTE: This event should not affect the ranking of products, only filter what is available.
     *
     * Params:
     *
     *  * query: The query, can be modified.
     *  * store: The current magento store
     */
    const EVENT_PRODUCTQUERY_PREPARE             = "crossroads_elasticsearch_productquery_prepare";
    /**
     * Event fired for each created and prepared recommended product query
     * (Crossroads_Elasticsearch_Model_Query_Products_Recommended).
     *
     * Contrary to the basic prepare this query is allowed to affect product ranking.
     *
     * Params:
     *
     *  * query: The query, can be modified.
     *  * store: The current magento store
     */
    const EVENT_PRODUCTQUERY_PREPARE_RECOMMENDED = "crossroads_elasticsearch_productquery_prepare_recommended";

    /**
     * Returns a list of servers for the Elasticsearch cluster.
     *
     * @param  Mage_Core_Model_Store
     * @return Array<string>
     */
    protected function getServers($store) {
        return array_map('trim', explode(",", $store->getConfig(self::CONFIG_NODES)));
    }

    /**
     * Returns a list of stores with elastic search enabled.
     *
     * @param  Array<string>   List of entity types which require an index
     * @return Array<Mage_Core_Model_Store>
     */
    public function getStores($types) {
        return array_filter(Mage::app()->getStores(), function($store) use($types) {
            return $this->enabledInStore($types, $store);
        });
    }

    /**
     * Returns true if the supplied store has elastic search enabled and configured for
     * all the supplied entity types.
     *
     * @param  Array<string>         List of entity types which require an index
     * @param  Mage_Core_Model_Store
     * @return boolean
     */
    public function enabledInStore($types, $store) {
        $ok = $store->getConfig(self::CONFIG_ENABLED);

        if(in_array(Mage_Catalog_Model_Product::ENTITY, $types)) {
            $ok = $ok && $store->getConfig(self::CONFIG_PRODUCT_INDEX);
        }

        return $ok;
    }

    public function createClient() {
        // TODO: Username, password and so on
        $logger = ClientBuilder::defaultLogger("var/log/elasticsearch.log");

        return ClientBuilder::create()
            ->setHosts($this->getServers(Mage::app()->getStore()))
            // We have to allow bad JSON serialization since we run this on older PHP versions
            ->allowBadJSONSerialization()
            ->setLogger($logger)
            ->build();
    }

    public function toLanguageAnalyzer($localeCode) {
        // Commented lines are unsupported languages by default ElasticSearch installations
        switch($localeCode) {
        //case "af_ZA":
        //case "sq_AL":
        case "ar_DZ":
        case "ar_EG":
        case "ar_KW":
        case "ar_MA":
        case "ar_SA":
            return "arabic";
        //case "az_AZ":
        //case "be_BY":
        //case "bn_BD":
        //case "bs_BA":
        case "bg_BG":
            return "bulgarian";
        case "ca_ES":
            return "catalan";
        //case "zh_CN":
        //case "zh_HK":
        //case "zh_TW":
        //case "hr_HR":
        case "cs_CZ":
            return "czech";
        case "da_DK":
            return "danish";
        case "nl_NL":
            return "dutch";
        case "en_AU":
        case "en_CA":
        case "en_IE":
        case "en_NZ":
        case "en_GB":
        case "en_US":
            return "english";
        //case "et_EE":
        //case "fil_PH":
        case "fi_FI":
            return "finnish";
        case "fr_CA":
        case "fr_FR":
            return "french";
        case "gl_ES":
            return "galician";
        //case "ka_GE":
        case "de_AT":
        case "de_DE":
        case "de_CH":
            return "german";
        case "el_GR":
            return "greek";
        //case "gu_IN":
        //case "he_IL":
        case "hi_IN":
            return "hindi";
        case "hu_HU":
            return "hungarian";
        //case "is_IS":
        //case "id_ID":
        case "it_IT":
        case "it_CH":
            return "italian";
        //case "ja_JP":
        //case "km_KH":
        //case "ko_KR":
        //case "lo_LA":
        case "lv_LV":
            return "latvian";
        case "lt_LT":
            return "lithuanian";
        //case "mk_MK":
        //case "ms_MY":
        //case "mn_MN":
        case "nb_NO":
        case "nn_NO":
            return "norwegian";
        case "fa_IR":
            return "persian";
        //case "pl_PL":
        case "pt_BR":
        case "pt_PT":
            return "portuguese";
        case "ro_RO":
            return "romanian";
        case "ru_RU":
            return "russian";
        //case "sr_RS":
        //case "sk_SK":
        //case "sl_SI":
        case "es_AR":
        case "es_CL":
        case "es_CO":
        case "es_CR":
        case "es_MX":
        case "es_PA":
        case "es_PE":
        case "es_ES":
        case "es_VE":
            return "spanish";
        //case "sw_KE":
        case "sv_SE":
            return "swedish";
        case "th_TH":
            return "thai";
        case "tr_TR":
            return "turkish";
        //case "uk_UA":
        //case "vi_VN":
        //case "cy_GB":
        default:
            return false;
        }
    }

    /**
     * Fetches the given product entities with the given page-size, ordered in the same
     * order as the supplied entity ids.
     *
     * @param  Array<integer>
     * @param  integer
     * @return Array<Mage_Catalog_Model_Product>
     */
    public function fetchProducts($entityIds, $pageSize, $visibility) {
        $coll = Mage::getModel('catalog/product')
            ->getCollection()
            ->addAttributeToSelect("*")
            ->addMinimalPrice()
            ->addAttributeToFilter(
                'visibility',
                ['in' => [
                    Mage_Catalog_Model_Product_Visibility::VISIBILITY_BOTH,
                    $visibility,
                ]]
            )
            ->addFieldToFilter("entity_id", ["in"=> $entityIds]);

        $products = [];
        $items    = $coll->getItems();

        foreach($entityIds as $id) {
            if(array_key_exists($id, $items)) {
                $products[] = $items[$id];
            }
            else {
                Mage::log("Crossroads_Elasticsearch: Failed to fetch product id {$id} for listing");
            }
        }

        return array_slice($products, 0, $pageSize);
    }

    /**
     * @param  Array  Data from elastic search aggregation query
     * @param  Array<Attribute>
     * @param  Array<string, string>
     * @param  Array<string, Array<string>>
     * @param  integer
     */
     public function transformFacets($data, $attributes, $params, $bucketOrdering, $numProducts) {
        return $this->sortFacets($bucketOrdering, $this->mapFacets($data, $attributes, function($f, $attr) use($numProducts, $params) {
            // Filters which do not match all the products of the filter
            $notMatchingAll = array_filter($f["buckets"], function($b) use($f) {
                return $f["count"] != $b["count"];
            });

            return (count($notMatchingAll) > 1 || array_key_exists($f["key"], $params)) &&
                $f["count"] / $numProducts >= $attr->getMinCardinalityFrac();
        }));
     }

    /**
     * Maps an aggregation response from ElasticSearch to an API-suitable format, ordering will be
     * the same as the order from $attributes.
     *
     * Input is an aggregation response, with optional `key`--count for the counts of the
     * different attributes.
     *
     * @param  Array
     * @param  Array<Crossroads_Elasticsearch_Model_Attribute>
     * @param  callable<Facet, Attribute>
     * @return Array
     */
    public function mapFacets($aggregations, $attributes, $filterFn = null) {
        $facets = [];

        foreach($attributes as $attr) {
            if($attr->isNumber()) {
                // TODO: Implement number support
                continue;
            }

            if(array_key_exists($attr->getCode(), $aggregations) && $attr->hasKeywordField()) {
                $a = $aggregations[$attr->getCode()]["data"];

                if( ! $a["count"]["value"]) {
                    continue;
                }

                $facet = [
                    "title"    => $attr->getFrontendLabel(),
                    "key"      => $attr->getCode(),
                    "count"    => array_key_exists("count", $a) ? $a["count"]["value"] : null,
                    "expand"   => $attr->getExpanded(),
                    "position" => $attr->getPosition(),
                    "buckets"  => array_values(array_map(function($b) {
                         return [
                             "key"   => $b["key"],
                             "count" => $b["doc_count"],
                         ];
                    }, $a["values"]["buckets"])),
                ];

                if( ! $filterFn || $filterFn($facet, $attr)) {
                    $facets[] = $facet;
                }
            }
        }

        return $facets;
    }

    public function sortFacets($keyOrder, $facets) {
        return array_map(function($f) use($keyOrder) {
            return $this->sortFacet($keyOrder, $f);
        }, $facets);
    }

    /**
     * @param  Array<string, Array<string>> hash of attribute_code => [attribute_value] in order
     * @param  Facet data
     * @return Facet data
     */
    public function sortFacet($keyOrder, $facet) {
        if( ! array_key_exists($facet["key"], $keyOrder)) {
            return $facet;
        }

        $order = $keyOrder[$facet["key"]];

        usort($facet["buckets"], function($a, $b) use($order) {
            $aPos = array_search($a["key"], $order);
            $bPos = array_search($b["key"], $order);

            return ($aPos === false ? 99999 : $aPos) - ($bPos === false ? 9999 : $bPos);
        });

        return $facet;
    }

    public function productPostDataPrepare($event) {
        $store        = Mage::app()->getStore();
        $pHelper      = Mage::helper("API/product");
        $product      = $event->getProduct();
        $preparedData = $event->getPreparedData();
        $numRecommend = (int)$store->getConfig(self::CONFIG_PRODUCT_RECOMMENDED);

        if($numRecommend < 1) {
            $preparedData->setData("xesRecommended", []);

            return;
        }

        $prodQuery = $this->createProductRecommendedQuery($store, Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_CATALOG)
            ->setSkus([$product->getSku()]);

        if($preparedData->getData("relatedProducts")) {
            // Skip related products
            $prodQuery->addFilter([
                "bool" => [
                    "must_not" => [
                        "terms" => [
                            "sku" => array_map(function($p) {
                                return $p["sku"];
                            }, $preparedData->getData("relatedProducts")),
                        ]
                    ]
                ]
            ]);
        }

        $query = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->setPageSize($numRecommend)
            ->setQuery($prodQuery);

        $result = $this->createClient()->search($query->toRequest());

        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        $recommended = $this->fetchProducts($entityIds, $numRecommend, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG);

        $preparedData->setData("xesRecommended", array_map([$pHelper, "prepareListProduct"], $recommended));
    }

    public function createProductQuery(Mage_Core_Model_Store $store, $visibility) {
        return $this->prepareProductQuery(Mage::getModel("Crossroads_Elasticsearch/query_products"), $store, $visibility);
    }

    /**
     * Sets common settings for a product query, does not affect product ranking (only filters allowed).
     */
    public function prepareProductQuery(Crossroads_Elasticsearch_Model_Query_Products $query, Mage_Core_Model_Store $store, $visibility) {
        $sess  = Mage::getSingleton("customer/session");
        $query = $query->setVisibility($visibility)
                       ->setStore($store);

        if($sess->isLoggedIn()) {
            $query->setCustomerId($sess->getCustomerId());

            // Limit products based on customer ids
            // TODO: Maybe move to an event?
            $query->addFilter(["bool" => [
                "should" => [
                    ["term" => ["elasticsearch_limit_customers" => $sess->getCustomerId()]],
                    ["bool" => ["must_not" => ["exists" => ["field" => "elasticsearch_limit_customers"]]]],
                ],
                "minimum_should_match" => 1,
            ]]);
        }
        else {
            // TODO: Maybe move to an event?
            // If we do not have a logged in user we still must exclude all products which require a user
            $query->addFilter(["bool" => ["must_not" => ["exists" => ["field" => "elasticsearch_limit_customers"]]]]);
        }

        Mage::dispatchEvent(self::EVENT_PRODUCTQUERY_PREPARE, [
            "store" => $store,
            "query" => $query,
        ]);

        return $query;
    }


    public function createProductRecommendedQuery($store, $visibility, $groupId = null, $complete = false) {
        return $this->prepareProductRecommendedQuery(Mage::getModel("Crossroads_Elasticsearch/query_products_recommended"), $store, $visibility, $groupId, $complete);
    }

    /**
     * Prepares a recommended query, can affect ranking.
     */
    public function prepareProductRecommendedQuery(Crossroads_Elasticsearch_Model_Query_Products_Recommended $query, Mage_Core_Model_Store $store, $visibility, $groupId = null, $complete = false) {
        $sess  = Mage::getSingleton("customer/session");
        $q = $this->prepareProductQuery($query, $store, $visibility)
                  ->setMinimumOrders($store->getConfig(self::CONFIG_MIN_ORDERS))
                  ->setGaussScale($store->getConfig(self::CONFIG_DAYS_CONSIDERED) / 2);

        if($groupId === null && $sess->isLoggedIn()) {
            $groupId = $sess->getCustomerGroupId();
        }

        if($groupId) {
            $query->addBoostFilter([ "customerGroupId" => $groupId ], $store->getConfig(self::CONFIG_GROUP_MULTIPLIER));
        }

        Mage::dispatchEvent(self::EVENT_PRODUCTQUERY_PREPARE_RECOMMENDED, [
            "store" => $store,
            "query" => $query,
        ]);

        return $q;
    }
}