<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;

require("Model/QueryParser.php");

/**
 * @covers Crossroads_Elasticsearch_Model_QueryParser
 */
class Crossroads_Elasticsearch_Model_QueryParserTests extends TestCase {
    public function testBasic() {
        $p = new Crossroads_Elasticsearch_Model_QueryParser();

        $this->assertEquals("", $p->repair(""));
        $this->assertEquals(" ", $p->repair(" "));
        $this->assertEquals("foo", $p->repair("foo"));
        $this->assertEquals("test", $p->repair("test"));
    }

    public function testReserved() {
        $p = new Crossroads_Elasticsearch_Model_QueryParser();

        $this->assertEquals("\: \t\r\n\f\\\\\\(\\)\"+-/^~[]\"", $p->repair(": \t\r\n\f\\()\"+-/^~[]"));
        $this->assertEquals("\: \t\r\n\f\\\\\\(\\)+\-\/\^\~\[\]", $p->repair(": \t\r\n\f\\()+-/^~[]"));
        $this->assertEquals("+\(-foo\)", $p->repair("+(-foo)"));
    }

    public function testOperators() {
        $p = new Crossroads_Elasticsearch_Model_QueryParser();

        $this->assertEquals("\NOT", $p->repair("NOT"));
        $this->assertEquals("\AND", $p->repair("AND"));
        $this->assertEquals("\OR",  $p->repair("OR"));
        $this->assertEquals("not",  $p->repair("not"));
        $this->assertEquals("and",  $p->repair("and"));
        $this->assertEquals("or",   $p->repair("or"));
        $this->assertEquals("NOT foo", $p->repair("NOT foo"));
        $this->assertEquals("\AND foo", $p->repair("AND foo"));
        $this->assertEquals("\OR foo",  $p->repair("OR foo"));
        $this->assertEquals("foo \NOT", $p->repair("foo NOT"));
        $this->assertEquals("foo \AND", $p->repair("foo AND"));
        $this->assertEquals("foo \OR",  $p->repair("foo OR"));
        $this->assertEquals("foo NOT bar", $p->repair("foo NOT bar"));
        $this->assertEquals("foo AND bar", $p->repair("foo AND bar"));
        $this->assertEquals("foo OR bar",  $p->repair("foo OR bar"));
        $this->assertEquals(" NOT foo", $p->repair(" NOT foo"));
        $this->assertEquals(" \AND foo", $p->repair(" AND foo"));
        $this->assertEquals(" \OR foo",  $p->repair(" OR foo"));
    }

    public function testFuzz() {
        $p = new Crossroads_Elasticsearch_Model_QueryParser();

        $this->assertEquals("\\~", $p->repair("~"));
        $this->assertEquals("\\\~", $p->repair("\~"));
        $this->assertEquals("\~foo", $p->repair("~foo"));
        $this->assertEquals("foo~", $p->repair("foo~"));
        $this->assertEquals("foo~5", $p->repair("foo~5"));
        $this->assertEquals("foo~0.5", $p->repair("foo~0.5"));
        $this->assertEquals("foo ~", $p->repair("foo ~"));
        $this->assertEquals("foo ~ 5", $p->repair("foo ~ 5"));
    }

    public function testWeird() {
        $p = new Crossroads_Elasticsearch_Model_QueryParser();

        $this->assertEquals("8=\)99802342haohaw\~\~08y\^pp8y9fpawweåi¨'*\"aad¨^\"", $p->repair("8=)99802342haohaw~~08y^pp8y9fpawweåi¨'*\"aad¨^"));
    }
}
