<?php

// TODO: Remove? Only used by standard search
class Crossroads_Elasticsearch_Model_Search {
    protected $store;

    protected $productAttributes;
    protected $localeCode;
    protected $queryString;
    protected $filters = [];

    protected $page     = 1;
    protected $pageSize = 20;

    protected $sort = [
        ["_score" => ["order" => "desc"]]
    ];

    public function __construct() {
        $this->productAttributes = Mage::getResourceModel("Crossroads_Elasticsearch/attribute")->getIndexableAttributes(Mage_Catalog_Model_Product::ENTITY, ["sku"]);
    }

    public function setStore($store) {
        $this->store = $store;

        return $this;
    }

    public function getStore() {
        return $this->store ?: Mage::app()->getStore();
    }

    public function setLocaleCode($code) {
        $this->localeCode = $code;

        return $this;
    }

    public function getLocaleCode() {
        return $this->localeCode ?: Mage::app()->getLocale()->getLocaleCode();
    }

    public function setpage($page) {
        $this->page = $page;

        return $this;
    }

    public function getPage() {
        return max($this->page, 1);
    }

    public function setPageSize($pageSize) {
        $this->pageSize = $pageSize;

        return $this;
    }

    public function getPageSize() {
        return max($this->pageSize, 1);
    }

    protected function getOffset() {
        return ($this->getPage() - 1) * $this->getPageSize();
    }

    public function setQueryString($queryString) {
        $p = new Crossroads_Elasticsearch_Model_QueryParser();

        $this->queryString = $p->repair($queryString);

        return $this;
    }

    public function getQueryString() {
        return $this->queryString;
    }

    /**
     * @param  array<string => string>
     * @return Crossroads_Elasticsearch_Model_Search
     */
    public function setFilters($filters) {
        $this->filters = $filters;

        return $this;
    }

    public function getFilters() {
        return $this->filters;
    }

    /**
     * Adds a new value to sort by, before any existing sorts. If the field is
     * "relevance", then it will change the basic score sort order.
     *
     * @param  string
     * @param  string   desc or asc
     * @return Crossroads_Elasticsearch_Model_Search
     */
    public function addSort($field, $dir = "desc") {
        if($field === "relevance") {
            $sort = array_filter($this->sort, function($s) {
                return !array_key_exists("_score", $s);
            });

            array_unshift($sort, [
                "_score" => [
                    "order" => $dir
                ]
            ]);

            $this->sort = $sort;

            return $this;
        }

        foreach($this->productAttributes as $attr) {
            if($attr->getCode() !== $field || ! $attr->getUsedForSortBy()) {
                continue;
            }

            array_unshift($this->sort, [
                $field => [
                    "order" => $dir
                ]
            ]);
        }

        return $this;
    }

    public function setSort($sort) {
        $this->sort = $sort;

        return $this;
    }

    public function getSort() {
        return $this->sort;
    }

    public function createQuery() {
        $store     = $this->getStore();
        $indexName = $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX);

        // TODO: Text filters
        // TODO: Int ranges
        $q = [
            "index" => $indexName,
            "type"  => "product",
            "body"  => [
                "size"         => $this->getPageSize(),
                "from"         => $this->getOffset(),
                "sort"         => $this->getSort(),
                "query"        => [
                    "bool" => [
                        "must"   => $this->getQuery(),
                        "filter" => $this->getFilter(),
                    ]
                ],
                "aggregations" => $this->getSearchAggregations(),
            ]
        ];

        return $q;
    }

    protected function getQuery() {
        return [
            "query_string" => [
                "query"                => $this->queryString,
                "fields"               => $this->getSearchFields(),
                // TODO: Configurable?
                "fuzzy_prefix_length"  => 3,
                // TODO: Configurable?
                "phrase_slop"          => 1,
                // TODO: Configurable?
                "minimum_should_match" => "100%",
                // TODO: Configurable?
                "default_operator"     => "AND",
                "locale"               => $this->getLocaleCode(),
            ]
        ];
    }

    protected function getSearchFields() {
        return call_user_func_array("array_merge", array_values(array_map(function($a) {
            return $a->getSearchFields($this->getLocaleCode());
        }, $this->productAttributes)));
    }

    protected function getSearchAggregations() {
        $aggs = [];
        // TODO: Skip the filter for the aggregate count
        // TODO: Add categories

        foreach($this->productAttributes as $attr) {
            // Check if the keyword really got indexed
            if($attr->hasKeywordField()) {
                $aggs[$attr->getCode()] = [
                    "terms" => [
                        "field" => $attr->getKeywordField()
                    ]
                ];
            }

            // TODO: Integer ranges
            if($attr->isNumber()) {
                $aggs[$attr->getCode()] = [
                    "stats" => [
                        "field" => $attr->getCode()
                    ]
                ];
            }
        }

        return empty($aggs) ? new stdClass() : $aggs;
    }

    protected function getFilter() {
        $filters = [];

        foreach($this->productAttributes as $attr) {
            if(array_key_exists($attr->getCode(), $this->filters)) {
                $data = $this->filters[$attr->getCode()];

                if($attr->hasKeywordField()) {
                    empty($filters["terms"]) && $filters["terms"] = [];

                    $filters["terms"][$attr->getCode().".keyword"] = is_array($data) ? array_values($data) : [$data];
                }

                // TODO: Integer ranges and exact matches
            }
        }

        return empty($filters) ? new stdClass() : $filters;
    }
}
