<?php

class Crossroads_Elasticsearch_Products_FacetedController extends Crossroads_API_Controller_Resource {
    /**
     * Parameters:
     *  * params:   Key-value query parameters used for filter
     *  * response: The response body Varien_Object
     */
    const EVENT_PRODUCTS_FACETED_RESPONSE = "crossroads_elasticsearch_products_faceted_response";

    /**
     * Parameters:
     *  * data:   Varien_Object with sorted value, set to true if event has sorted.
     *  * params: Request parameters
     *  * query:  The elasticsearch query to sort.
     */
    const EVENT_SORT = "crossroads_elasticsearch_products_faceted_sort";

    /**
     *
     * @apiParam {Integer} [limit=20]
     * @apiParam {Integer} [page=0]
     * @apiParam {String}  [sort]   Multiple sort-columns can be used in order with most significant first, default is _score (and will be added last as a discriminator, no need to add this manually)
     * @apiParam {Boolean} [aggregate=false]
     */
    public function getAll() {
        $helper = Mage::helper("Crossroads_Elasticsearch");
        $store  = Mage::app()->getStore();
        $req    = $this->getRequest();

        if( ! $helper->enabledInStore([Mage_Catalog_Model_Product::ENTITY], $store)) {
            return [403, [ "message" => "Elasticsearch extension is disabled for this store" ]];
        }

        $pHelper    = Mage::helper("API/product");
        $pageSize   = min((int)$req->getQuery("limit", "20"), 100);
        $page       = max((int)$req->getQuery("page", "1"), 1);
        $aggregate  = trim(strtolower($req->getQuery("aggregate", "false"))) === "true";
        $resource   = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes = $resource->getFilterableProductAttributes($store);
        $bucketOrd  = $resource->getSortedAttributeValues($attributes, $store);
        $client     = $helper->createClient();
        $params     = array_filter(array_diff_key($req->getParams(), [
            "limit"     => true,
            "page"      => true,
            "sort"      => true,
            "aggregate" => true,
        ]));
        $q = Mage::getModel("Crossroads_Elasticsearch/query_products")
                ->setVisibility(Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_CATALOG)
                ->setStore($store);

        if(Mage::getSingleton("customer/session")->isLoggedIn()) {
            $q->setCustomerId(Mage::getSingleton("customer/session")->getCustomerId());
        }

        $clone = clone $q;
        $query = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->addSource([ "sku", "name" ])
            ->setPageSize($pageSize)
            ->setPage($page)
            ->setQuery($clone->setFilter($attributes, $params));

        if($aggregate) {
            $query->setAggregations(Mage::getModel("Crossroads_Elasticsearch/query_aggregation_attribute")
                ->setAttributes($attributes)
                ->setParameters($params)
                ->setExcludeQueryConstructor(function($paramKey) use($q, $attributes, $params) {
                    $clone = clone $q;

                    return $clone->setFilter($attributes, array_diff_key($params, [ $paramKey => true ]));
                }));
        }

        if($req->getQuery("sort")) {
            $obj = new Varien_Object([
                "sorted" => false,
            ]);

            Mage::dispatchEvent(self::EVENT_SORT, [
                "query"   => $query,
                "flags"   => $obj,
                "request" => $req,
                "sort"    => $req->getQuery("sort"),
            ]);

            if( ! $obj->getSorted()) {
                foreach(array_reverse(array_filter(array_map("trim", explode(",", $req->getQuery("sort"))))) as $sort) {
                    $query->addAttributeSort($attributes, $sort);
                }
            }
        }

        $result = $client->search($query->toRequest());

        $numProducts = $result["hits"]["total"];
        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        $resp = new Varien_Object([
            "_time"    => $result["took"],
            "total"    => $numProducts,
            "products" => array_map([$pHelper, "prepareListProduct"],
                $helper->fetchProducts($entityIds, $pageSize, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG)),
            "facets"   => $aggregate && array_key_exists("aggregations", $result) ? $helper->transformFacets($result["aggregations"], $attributes, $params, $bucketOrd, $numProducts) : null,
        ]);

        Mage::dispatchEvent(self::EVENT_PRODUCTS_FACETED_RESPONSE, [
            "params"   => $params,
            "response" => $resp,
        ]);

        return [200, $resp->getData()];
    }

    public function getItem($id) {
        return [404];
    }
}
