<?php

class Crossroads_Elasticsearch_Model_Resource_Product_Instock extends Crossroads_Elasticsearch_Model_Resource_Abstract {
    protected function _construct() {
        $this->_setResource("core_read");
    }

    public function addMapping($event) {
        $mapping = $event->getMapping();

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_IN_STOCK, [
            "type"       => "boolean",
            // Default to in stock
            "null_value" => true,
        ]);

        $mapping->setData(Crossroads_Elasticsearch_Helper_Data::FIELD_STOCK, [
            "type"       => "integer",
            "null_value" => 0,
        ]);
    }

    public function indexAll($event) {
        $manager = $event->getManager();
        $indices = $event->getIndices();
        $stores  = $event->getStores();
        $types   = $event->getTypes();
        $sql     = $this->getReadConnection();
        $query   = $this->createInStockConfigurable($sql, $stores);
        $stmt    = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    public function indexProducts($event) {
        $productIds = $event->getProductIds();
        $manager    = $event->getManager();
        $indices    = $event->getIndices();
        $stores     = $event->getStores();
        $types      = $event->getTypes();
        $sql        = $this->getReadConnection();
        $query      = $this->createInStockConfigurable($sql, $stores, $productIds);
        $stmt       = $sql->query($query);

        $this->loadData($indices, $types, $manager, $stmt);
    }

    protected function loadData($indices, $types, $manager, $stmt) {
        foreach($stmt as $row) {
            // Perform partial updates
            $manager->queue([ "update" => [
                "_index"   => $indices[$row["store_id"]],
                "_routing" => (string)$row["entity_id"],
                "_type"    => "product",
                "_id"      => $row["entity_id"],
            ]], [ "doc" => [
                Crossroads_Elasticsearch_Helper_Data::FIELD_IN_STOCK => $row["stock"] > 0,
                Crossroads_Elasticsearch_Helper_Data::FIELD_STOCK    => (int)$row["stock"],
            ]]);
        }
    }

    public function createInStockConfigurable($sql, $stores, $productIds = null) {
        $statusExpr = Mage::getStoreConfigFlag(Mage_CatalogInventory_Model_Stock_Item::XML_PATH_MANAGE_STOCK) ?
            $sql->getCheckSql("cisi.use_config_manage_stock = 0 AND cisi.manage_stock = 0", 1, "cisi.is_in_stock") :
            $sql->getCheckSql("cisi.use_config_manage_stock = 0 AND cisi.manage_stock = 1", "cisi.is_in_stock", 1);
        $childExpr = Mage::getStoreConfigFlag(Mage_CatalogInventory_Model_Stock_Item::XML_PATH_MANAGE_STOCK) ?
            $sql->getCheckSql("i.use_config_manage_stock = 0 AND i.manage_stock = 0", 1, "i.is_in_stock") :
            $sql->getCheckSql("i.use_config_manage_stock = 0 AND i.manage_stock = 1", "i.is_in_stock", 1);
        $valueExpr  = $sql->getCheckSql("p.type_id IN ('configurable', 'bundle')", 0, $statusExpr);

        $select = $sql->select()
            ->from(["s" => $this->createEntityIdStoreQuery($sql, $stores, $productIds)], ["entity_id", "store_id"])
            ->join(["p" => $this->getTable("catalog/product")], "p.entity_id = s.entity_id", ["type_id"]);

        $select = $select
            ->join(["cis" => $this->getTable("cataloginventory/stock")], "", [])
            ->joinLeft(["cisi" => $this->getTable("cataloginventory/stock_item")], "cisi.stock_id = cis.stock_id AND cisi.product_id = p.entity_id", [])
            ->joinLeft(["l" => $this->getTable("catalog/product_super_link")], "l.parent_id = p.entity_id", [])
            ->joinLeft(["le" => $this->getTable("catalog/product")], "le.entity_id = l.product_id", [])
            ->joinLeft(["i" => $this->getTable("cataloginventory/stock_item")], "i.product_id = l.product_id AND cis.stock_id = i.stock_id", [])
            ->columns([
                "stock" => "SUM(".$valueExpr.") + SUM(COALESCE(".$childExpr.", 0))"
            ]);

        return $select->group(["s.store_id", "s.entity_id"]);
    }
}
