<?php

class Crossroads_Elasticsearch_Model_Resource_Attribute extends Mage_Core_Model_Resource_Db_Abstract {
    protected function _construct() {
        $this->_setResource("core_read");
    }

    protected function createSelect($sql, $storeId = null) {
        return $sql->select()
            ->from(["ca" => $this->getTable("catalog/eav_attribute")], [
                "attribute_id",
                "is_filterable",
                "is_filterable_in_search",
                "is_visible_in_advanced_search",
                "is_searchable",
                "is_html" => $sql->getCheckSql("ca.is_html_allowed_on_front OR ca.is_wysiwyg_enabled", "1", "0"),
                "used_for_sort_by",
                "boost" => $sql->getCheckSql("ca.crossroads_elasticsearch_boost > 0", "ca.crossroads_elasticsearch_boost", "1"),
                "operator" => "crossroads_elasticsearch_operator",
                "autocomplete" => "crossroads_elasticsearch_autocomplete",
                "min_cardinality" => "crossroads_elasticsearch_min_cardinality",
                "expanded" => "crossroads_elasticsearch_expanded",
                "max_options" => "crossroads_elasticsearch_max_options",
            ])
            ->join(["ea" => $this->getTable("eav/attribute")], "ca.attribute_id = ea.attribute_id", [
                "attribute_code",
                "backend_type",
                "backend_model",
                "source_model",
                "frontend_input",
                "default_value",
            ])
            ->join(["et" => $this->getTable("eav/entity_type")], "ea.entity_type_id = et.entity_type_id", [])
            ->joinLeft(["eal" => $this->getTable("eav/attribute_label")], sprintf("eal.attribute_id = ea.attribute_id AND eal.store_id = '%d'", (int)$storeId ?: 0), [])
            ->columns(["frontend_label" => $sql->getIfNullSql(
                "eal.value",
                "ea.frontend_label"
            )]);
    }

    protected function convertResults($stmt) {
        $resp = [];

        while($row = $stmt->fetch()) {
            $resp[] = new Crossroads_Elasticsearch_Model_Attribute($row);
        }

        return array_combine(array_map(function($a) { return $a->getId(); }, $resp), $resp);
    }

    public function getSearchableProductAttributes($store) {
        $sql = $this->getReadConnection();

        return $this->convertResults($sql->query($this->createSelect($sql, $store->getId())
            ->joinLeft([ "attr_status" => $this->getTable("Crossroads_Elasticsearch/attribute_status") ], "attr_status.attribute_id = ca.attribute_id", ["keyword_field", "locale_fields"])
            ->where("ca.is_searchable > 0 OR ca.is_visible_in_advanced_search > 0")
            ->where("et.entity_type_code = ?", Mage_Catalog_Model_Product::ENTITY)
            ->order(new Zend_Db_Expr("ca.position = 0, ca.position ASC"))));
    }

    public function getFilterableProductAttributes($store) {
        $sql = $this->getReadConnection();

        return $this->convertResults($sql->query($this->createSelect($sql, $store->getId())
            ->joinLeft([ "attr_status" => $this->getTable("Crossroads_Elasticsearch/attribute_status") ], "attr_status.attribute_id = ca.attribute_id", ["keyword_field", "locale_fields"])
            ->where("ca.is_filterable > 0 OR ca.is_filterable_in_search > 0 OR ca.used_for_sort_by > 0")
            ->where("et.entity_type_code = ?", Mage_Catalog_Model_Product::ENTITY)
            ->order(new Zend_Db_Expr("ca.position = 0, ca.position ASC"))));
    }

    /**
     * Returns a list of Attributes for the given entity code, mapped by their attribute id.
     *
     * @param  string
     * @return Array<Crossroads_Elasticsearch_Model_Attribute>
     */
    public function getIndexableAttributes($entityCode, $includedStaticAttributes = []) {
        $sql = $this->getReadConnection();

        $select = $this->createSelect($sql)
            ->where("ca.is_filterable > 0 OR ca.is_searchable > 0 OR ca.is_filterable_in_search > 0 OR ca.is_visible_in_advanced_search > 0 OR used_for_sort_by > 0 OR attribute_code = 'elasticsearch_limit_customers'")
            ->where("et.entity_type_code = ?", $entityCode);

        if( ! empty($includedStaticAttributes)) {
            $select = $select->where("ea.backend_type <> 'static' OR ea.attribute_code IN (?)", $includedStaticAttributes);
        }
        else {
            $select = $select->where("ea.backend_type <> 'static'");
        }

        return $this->convertResults($sql->query($select));
    }

    /**
     * Lists all attributes which are to be autocompleted. Will by default only return the ones
     * which have already mappings created (reindex creates them).
     *
     * @param  Mage_Core_Model_Store
     * @param  boolean True to list all attributes
     */
    public function getAutocompleteAttributes($store, $includeNonMapped = false) {
        $sql   = $this->getReadConnection();
        $query = $this->createSelect($sql, $store->getId())
            ->where("(ca.is_filterable > 0 OR ca.is_filterable_in_search > 0 OR ca.used_for_sort_by > 0) AND ca.crossroads_elasticsearch_autocomplete > 0")
            ->where("et.entity_type_code = ?", Mage_Catalog_Model_Product::ENTITY)
            ->order(new Zend_Db_Expr("ca.position = 0, ca.position ASC"));

        if( ! $includeNonMapped) {
            $query->join([ "attr_status" => $this->getTable("Crossroads_Elasticsearch/attribute_status") ], "attr_status.attribute_id = ca.attribute_id AND attr_status.completion_field = 1", []);
        }

        return $this->convertResults($sql->query($query));
    }

    /**
     * Creates a Look-Up-Table for attributes with the backend model eav/entity_attribute_backend_array.
     *
     * Keys are, in order; store_id, attribute_id, option_id.
     *
     * @param  Array<Attribute>
     * @return Array<int, Array<int, Array<int, string>>>
     */
    public function getMultiselectLUT($attributes) {
        $sql = $this->getReadConnection();

        $ids = array_map(function($a) { return $a->getId(); }, array_filter($attributes, function($a) {
            return $a->getBackendModel() === "eav/entity_attribute_backend_array";
        }));

        $data   = [];
        $result = $sql->query($sql->select()
            ->from([ "a" => $this->getTable("eav/attribute") ], ["attribute_id"])
            ->from([ "s" => $this->getTable("core/store") ], ["store_id"])
            ->join([ "ao" => $this->getTable("eav/attribute_option") ], "a.attribute_id = ao.attribute_id", ["option_id"])
            ->joinLeft([ "aov_global" => $this->getTable("eav/attribute_option_value") ], "ao.option_id = aov_global.option_id AND aov_global.store_id = 0", [])
            ->joinLeft([ "aov_store" => $this->getTable("eav/attribute_option_value") ], "ao.option_id = aov_store.option_id AND aov_store.store_id = s.store_id", [])
            ->columns([ "value" => $sql->getIfNullSql(
                    $sql->quoteIdentifier('aov_store.value'),
                    $sql->quoteIdentifier('aov_global.value')
                )
            ])
            ->group(["a.attribute_id", "s.store_id", "ao.option_id"]));

        foreach($result as $row) {
            $attr_id   = (int)$row["attribute_id"];
            $store_id  = (int)$row["store_id"];
            $option_id = (int)$row["option_id"];

            if( ! array_key_exists($store_id, $data)) {
                $data[$store_id] = [];
            }

            if( ! array_key_exists($attr_id, $data[$store_id])) {
                $data[$store_id][$attr_id] = [];
            }

            $data[$store_id][$attr_id][$option_id] = $row["value"];
        }

        return $data;
    }

    public function getSortedAttributeValues($attributes, $store) {
        $sql = $this->getReadConnection();

        $ids = array_map(function($a) { return $a->getId(); }, array_filter($attributes, function($a) {
            return $a->getSourceModel() === "eav/entity_attribute_source_table" ||
                in_array($a->getFrontendInput(), ["select", "multiselect"]) && !$a->getSourceModel();
        }));

        $data   = [];
        $result = $sql->query($sql->select()
            ->from([ "a" => $this->getTable("eav/attribute") ], ["attribute_code"])
            ->join([ "ao" => $this->getTable("eav/attribute_option") ], "a.attribute_id = ao.attribute_id", ["option_id"])
            ->joinLeft([ "aov_global" => $this->getTable("eav/attribute_option_value") ], "ao.option_id = aov_global.option_id AND aov_global.store_id = 0", [])
            ->joinLeft([ "aov_store" => $this->getTable("eav/attribute_option_value") ], sprintf("ao.option_id = aov_store.option_id AND aov_store.store_id = %d", $store->getId()), [])
            ->columns([ "value" => $sql->getIfNullSql(
                    $sql->quoteIdentifier('aov_store.value'),
                    $sql->quoteIdentifier('aov_global.value')
                )
            ])
            ->where("a.attribute_id IN (?)", $ids)
            ->where("ao.sort_order <> 0")
            ->group(["a.attribute_code", "ao.option_id"])
            ->order("ao.sort_order ASC"));

        foreach($result as $row) {
            if(empty($data[$row["attribute_code"]])) {
                $data[$row["attribute_code"]] = [];
            }

            $data[$row["attribute_code"]][] = $row["value"];
        }

        return $data;
    }
}
