<?php

class Crossroads_Elasticsearch_Model_Attribute {
    const MYSQL_DATE_FORMAT = "yyyy-MM-dd HH:mm:ss";

    protected $attribute_id;
    protected $attribute_code;
    protected $backend_type;
    protected $backend_model;
    protected $source_model;
    protected $frontend_input;
    protected $default_value;
    protected $is_searchable;
    protected $is_filterable;
    protected $is_filterable_in_search;
    protected $is_visible_in_advanced_search;
    /**
     * If this attribute contains HTML (is_html_allowed_on_front OR is_wysiwyg_enabled).
     *
     * @var boolean
     */
    protected $is_html;
    protected $usedForSortBy;
    protected $boost;
    protected $operator;
    /**
     * Minimum cardinality as a percentage.
     *
     * @var integer
     */
    protected $min_cardinality;
    protected $expanded;
    protected $max_options;
    protected $frontend_label;
    protected $autocomplete;
    /**
     * If this attribute has an indexed keyword field.
     */
    protected $keyword_field = null;
    protected $locale_fields = null;

    public function __construct($data) {
        $this->attribute_id                  = $data["attribute_id"];
        $this->attribute_code                = $data["attribute_code"];
        $this->backend_type                  = $data["backend_type"];
        $this->backend_model                 = $data["backend_model"];
        $this->source_model                  = $data["source_model"];
        $this->frontend_input                = $data["frontend_input"];
        $this->default_value                 = $data["default_value"];
        $this->is_searchable                 = $data["is_searchable"];
        $this->is_filterable                 = $data["is_filterable"];
        $this->is_filterable_in_search       = $data["is_filterable_in_search"];
        $this->is_visible_in_advanced_search = $data["is_visible_in_advanced_search"];
        $this->is_html                       = $data["is_html"];
        $this->usedForSortBy                 = $data["used_for_sort_by"];
        $this->boost                         = $data["boost"];
        $this->operator                      = $data["operator"];
        $this->frontend_label                = $data["frontend_label"];
        $this->autocomplete                  = $data["autocomplete"];
        $this->min_cardinality               = $data["min_cardinality"];
        $this->expanded                      = $data["expanded"];
        $this->max_options                   = $data["max_options"];

        if(array_key_exists("keyword_field", $data)) {
            $this->keyword_field = (bool)$data["keyword_field"];
        }

        if(array_key_exists("locale_fields", $data)) {
            $this->locale_fields = array_filter(explode(",", $data["locale_fields"]));
        }
    }

    public function getId() {
        return $this->attribute_id;
    }

    public function getCode() {
        return $this->attribute_code;
    }

    public function getBackendType() {
        return $this->backend_type;
    }

    public function getBackendModel() {
        return $this->backend_model;
    }

    public function getSourceModel() {
        return $this->source_model;
    }

    public function getFrontendInput() {
        return $this->frontend_input;
    }

    public function getDefaultValue() {
        return $this->default_value;
    }

    public function getIsSearchable() {
        return $this->is_searchable;
    }

    public function getIsFilterable() {
        return $this->is_filterable;
    }

    public function getIsFilterableInSearch() {
        return $this->is_filterable_in_search;
    }

    public function getIsVisibleInAdvancedSearch() {
        return $this->is_visible_in_advanced_search;
    }

    /**
     * Returns true if the field contains HTML which needs to be stripped/parsed before indexing.
     *
     * @return boolean
     */
    public function getIsHtml() {
        return $this->is_html;
    }

    /**
     * Returns true if the field is used for sorting.
     *
     * @return boolean
     */
    public function getUsedForSortBy() {
        // Keyword field required for sorting
        if( ! is_null($this->keyword_field) && ! $this->keyword_field) {
            return false;
        }

        return $this->usedForSortBy;
    }

    /**
     * Returns the boost-factor for matches in the given field during full-text search.
     *
     * @return integer
     */
    public function getBoost() {
        return $this->boost;
    }

    /**
     * Returns the operator for combining multiple filters using this attribute.
     *
     * @return string  ElasticSearch bool-query operator (must, should)
     */
    public function getOperator() {
        return $this->operator;
    }

    /**
     * Returns the frontend label for this attribute.
     *
     * @return string
     */
    public function getFrontendLabel() {
        return $this->frontend_label;
    }

    /**
     * Returns true if we are able to perform autocomplete on this field.
     *
     * @return boolean
     */
    public function getAutocomplete() {
        return $this->autocomplete;
    }

    /**
     * Returns the minimum cardinality as a fraction.
     *
     * @return double
     */
    public function getMinCardinalityFrac() {
        return (double)$this->min_cardinality / 100;
    }

    /**
     * Returns true if this attribute facet should be expanded by default.
     *
     * @return boolean
     */
    public function getExpanded() {
        return (boolean)$this->expanded;
    }

    public function getMaxOptions() {
        return (int)$this->max_options;
    }

    /**
     * Returns true if the attribute is a Magento enumeration, where one or more values are selected
     * from a drop-down.
     *
     * @return boolean
     */
    public function isEnum() {
        return in_array($this->frontend_input, ["select", "multiselect"]);
    }

    /**
     * Returns true if the data contained in this attribute is text.
     *
     * @return boolean
     */
    public function isText() {
        return $this->getMappingType() === "text";
    }

    public function isNumber() {
        return $this->getMappingType() === "integer";
    }

    public function isStatic() {
        return $this->getBackendType() === "static";
    }

    public function getMapping($locale) {
        $data   = [
            "type"    => $this->getMappingType(),
            // "copy_to" => $this->getCode(),
            "fields"  => $this->getMappingFields($locale)
        ];

        if($this->getBackendType() === "datetime") {
            $data["format"] = self::MYSQL_DATE_FORMAT;
        }

        return $data;
    }

    public function getSearchFields($locale) {
        if($this->getIsSearchable() && $this->isText()) {
            $boost = ($this->boost > 1 ? "^".(int)$this->boost : "");

            return ($this->isStatic() || ! in_array($locale, $this->locale_fields ?: [])) ? [$this->getCode().$boost] : [$this->getCode().$boost, $this->getCode().".".$locale.$boost];
        }

        return [];
    }

    /**
     * Returns true if this attribute has a keyword field.
     *
     * @return boolean
     */
    public function hasKeywordField() {
        if( ! is_null($this->keyword_field)) {
            return $this->keyword_field;
        }

        return ($this->getIsFilterableInSearch() || $this->getIsFilterable()) && $this->isText();
    }

    public function getKeywordField() {
        if(Mage::getIsDeveloperMode() && ! ($this->hasKeywordField() || $this->getUsedForSortBy())) {
            throw new Exception(sprintf("%s: Filterable/sortable field is not enabled for '%s'.", __METHOD__, $this->getCode()));
        }

        return $this->getCode().".keyword";
    }

    public function getTrigramField() {
        if(Mage::getIsDeveloperMode() && ! $this->autocomplete) {
            throw new Exception(sprintf("%s: Autocomplete is not enabled for '%s'.", __METHOD__, $this->getCode()));
        }

        return $this->getCode().".trigram";
    }

    public function getCompletionField() {
        if(Mage::getIsDeveloperMode() && ! $this->autocomplete) {
            throw new Exception(sprintf("%s: Autocomplete is not enabled for '%s'.", __METHOD__, $this->getCode()));
        }

        return $this->getCode().".completion";
    }

    protected function getMappingType() {
        switch($this->getBackendType()) {
        case "datetime":
            return "date";
        case "decimal":
            return "double";
        case "int":
            if( ! $this->isEnum()) {
                return "integer";
            }
            // We let it continue to text/varchar on purpose if the frontend is select/multiselect since it
            // is an option, which means that the data is text
        case "text":
        case "varchar":
            return "text";
        case "static":
            return $this->getFrontendInput();
        default:
            throw new Exception(sprintf("%s: Unknown attribute backend type '%s'.", __METHOD__, $this->getBackendType()));
        }
    }

    protected function getMappingFields($locale) {
        $fields = [];

        if($this->isText()) {
            $fields[$locale] = [
                "type"     => "text",
                "analyzer" => Mage::helper("Crossroads_Elasticsearch")->toLanguageAnalyzer($locale),
                // "copy_to"  => "_all",
                "store"    => true,
            ];
        }

        if($this->hasKeywordField() || $this->getUsedForSortBy()) {
            $fields["keyword"] = [
                "type"         => "keyword",
                "ignore_above" => 256
            ];
        }

        if($this->getAutocomplete()) {
            $fields["trigram"] = [
                "analyzer" => "trigram",
                "type"     => "text"
            ];
            $fields["completion"] = [
                "analyzer" => "standard",
                "type"     => "completion"
            ];
        }

        return $fields;
    }
}
