<?php

class Crossroads_Elasticsearch_Products_RecommendedController extends Crossroads_API_Controller_Resource {
    /**
     * Event fired when constructing the query for recommending products.
     *
     * Params:
     *
     *  * query:  The elasticsearch query
     *  * params: A Varien_Object of the parameters, remaining values will be used for filtering
     */
    const EVENT_RECOMMENDED_QUERY = "crossroads_elasticsearch_products_recommended_query";

    protected function getSkus() {
        $skus     = [];
        $helper   = Mage::helper("Crossroads_Elasticsearch");
        $cart     = $this->getCheckoutCart();
        $orders   = Mage::getResourceModel("Crossroads_Elasticsearch/order");
        $customer = Mage::getSingleton('customer/session')->getCustomer();
        $list     = array_values(array_filter(array_map("trim", explode(",", $this->getRequest()->getQuery("skus")))));

        foreach($list as $sku) {
            switch($sku) {
            case "CART":
                foreach($cart->getQuote()->getAllItems() as $i) {
                    $skus[] = $i->getSku();
                }

                break;
            case "HISTORY":
                foreach($orders->getOrderHistory($customer->getEntityId()) as $s) {
                    $skus[] = $s;
                }

                break;
            default:
                $skus[] = $sku;
            }
        }

        return $skus;
    }

    /**
     *
     * @apiParam {Integer} [limit=20]
     * @apiParam {Integer} [page=1]
     * @apiParam {Boolean} [aggregate=false]
     * @apiParam {String}  [skus] Comma-separated list of SKUs, `HISTORY` and
     *                             `CART` are special SKUs expanding to multiple SKUs
     * @apiParam {Boolean} [complete=false] If the query should attempt to complete the list of SKUs
     *                                      instead of just listing popular products
     */
    public function getAll() {
        $helper = Mage::helper("Crossroads_Elasticsearch");
        $store  = Mage::app()->getStore();
        $req    = $this->getRequest();

        if( ! $helper->enabledInStore([Mage_Catalog_Model_Product::ENTITY], $store)) {
            return [403, [ "message" => "Elasticsearch extension is disabled for this store" ]];
        }

        $pHelper    = Mage::helper("API/product");
        $pageSize   = min((int)$req->getQuery("limit", "20"), 100);
        $page       = max((int)$req->getQuery("page", "1"), 1);
        $aggregate  = trim(strtolower($req->getQuery("aggregate"))) === "true";
        $complete   = trim(strtolower($req->getQuery("complete"))) === "true";
        $groupId    = $req->getQuery("customer_group_id") ?: Mage::getSingleton("customer/session")->getCustomerGroupId() ?: 0;
        $resource   = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes = $resource->getFilterableProductAttributes($store);
        $bucketOrd  = $resource->getSortedAttributeValues($attributes, $store);
        $client     = $helper->createClient();
        $params     = new Varien_Object(array_filter(array_diff_key($this->getRequest()->getParams(), [
            "limit"             => true,
            "page"              => true,
            "skus"              => true,
            "aggregate"         => true,
            "complete"          => true,
            "customer_group_id" => true,
        ])));

        $skus     = $this->getSkus();

        $prodQuery = Mage::getModel("Crossroads_Elasticsearch/query_products_recommended")
            ->setStore($store)
            ->setComplete($complete)
            ->setVisibility(Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_CATALOG)
            ->setMinimumOrders($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_MIN_ORDERS))
            ->setGaussScale($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_DAYS_CONSIDERED) / 2)
            ->setSkus($skus);

        if(Mage::getSingleton("customer/session")->isLoggedIn()) {
            $prodQuery->setCustomerId(Mage::getSingleton("customer/session")->getCustomerId());
        }

        if($groupId) {
            $prodQuery->addBoostFilter([ "customerGroupId" => $groupId ], $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_GROUP_MULTIPLIER));
        }

        Mage::dispatchEvent(self::EVENT_RECOMMENDED_QUERY, [
            "params" => $params,
            "query"  => $prodQuery,
        ]);

        $params = $params->getData();
        $clone  = clone $prodQuery;
        $query  = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->addSource([ "sku", "name" ])
            ->setPageSize($pageSize)
            ->setPage($page)
            //->addSort(Crossroads_Elasticsearch_Helper_Data::FIELD_POSITIONS, "asc", [ "mode" => "avg" ])
            ->setQuery($clone->setFilter($attributes, $params));

        if($aggregate) {
            $query->setAggregations(Mage::getModel("Crossroads_Elasticsearch/query_aggregation_attribute")
                ->setAttributes($attributes)
                ->setParameters($params)
                ->setExcludeQueryConstructor(function($paramKey) use($prodQuery, $attributes, $params) {
                    $clone = clone $prodQuery;

                    return $clone->setFilter($attributes, array_diff_key($params, [ $paramKey => true ]));
                }));
        }

        if($req->getQuery("sort")) {
            $query->addAttributeSort($attributes, $req->getQuery("sort"));
        }

        $result = $client->search($query->toRequest());

        $numProducts = $result["hits"]["total"];
        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        return [200, [
            "_time"    => $result["took"],
            "total"    => $numProducts,
            "products" => array_map([$pHelper, "prepareListProduct"],
                $helper->fetchProducts($entityIds, $pageSize, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG)),
            "facets"   => $aggregate ? $helper->transformFacets($result["aggregations"], $attributes, $params, $bucketOrd, $numProducts) : null,
        ]];
    }

    /**
     * @api {get} /products_recommended/:productUrl   Show product and all of its recommended products
     * @apiName  productsRecommendedSpecific
     * @apiGroup Recommendations
     *
     * @apiParam {string}   productUrl The urlKey of the recommendation source product
     * @apiParam {boolean}  [aggregate=false]  If to show the aggregations
     * @apiParam {booelan}  [product=false]    If to show the product
     * @apiParam {integer}  [limit=20]      Maximum number of recommended products per page
     * @apiParam {integer}  [page=1]      Page to show
     *
     * @apiSuccess {integer}  total      Total number of recommendations
     * @apiSuccess {Object}   [product]  The recommendation source product, null if `product` parameter is false
     * @apiSuccess {Object[]} products   The recommended products
     * @apiSuccess {Object[]} [facets]   Facets for the recommended products, null if `aggregate` is false
     */
    public function getItem($id) {
        $helper     = Mage::helper("Crossroads_Elasticsearch");
        $req        = $this->getRequest();
        $store      = Mage::app()->getStore();
        $pHelper    = Mage::helper("API/product");
        $pageSize   = min((int)$req->getQuery("limit", "20"), 100);
        $page       = max((int)$req->getQuery("page", "1"), 1);
        $aggregate  = trim(strtolower($req->getQuery("aggregate"))) === "true";
        $showProd   = trim(strtolower($req->getQuery("product"))) === "true";
        $resource   = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes = $resource->getFilterableProductAttributes($store);
        $bucketOrd  = $resource->getSortedAttributeValues($attributes, $store);
        $params     = array_filter(array_diff_key($this->getRequest()->getParams(), [
            "limit"     => true,
            "page"      => true,
            "product"   => true,
            "aggregate" => true,
        ]));
        $rewrite    = Mage::getModel("core/url_rewrite")
            ->setStoreId(Mage::app()->getStore()->getId())
            ->loadByRequestPath($id);

        $isRedirect  = stripos($rewrite->getOptions(), "R") !== false;
        $isPermaneht = stripos($rewrite->getOptions(), "P") !== false;

        if($isRedirect || $isPermaneht) {
            return [404];
        }

        if( ! $rewrite->getProductId()) {
            return [404];
        }

        $product = Mage::getModel('catalog/product')->load($rewrite->getProductId());

        if( ! $product || ! $product->getId() || ! $product->isVisibleInSiteVisibility()) {
            return [404];
        }

        $prodQuery = Mage::getModel("Crossroads_Elasticsearch/query_products_recommended")
            ->setStore($store)
            ->setVisibility(Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_CATALOG)
            ->setMinimumOrders($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_MIN_ORDERS))
            ->setGaussScale($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_DAYS_CONSIDERED) / 2)
            ->setSkus([$product->getSku()]);

        $clone = clone $prodQuery;
        $query = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->setPageSize($pageSize)
            ->setPage($page)
            ->setQuery($clone->setFilter($attributes, $params));

        if($aggregate) {
            $query->setAggregations(Mage::getModel("Crossroads_Elasticsearch/query_aggregation_attribute")
                ->setAttributes($attributes)
                ->setParameters($params)
                ->setExcludeQueryConstructor(function($paramKey) use($prodQuery, $attributes, $params) {
                    $clone = clone $prodQuery;

                    return $clone->setFilter($attributes, array_diff_key($params, [ $paramKey => true ]));
                }));
        }

        $result = $helper->createClient()->search($query->toRequest());

        $numProducts = $result["hits"]["total"];
        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        $recommended = $helper->fetchProducts($entityIds, $pageSize, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG);

        return [200, [
            "_time"    => $result["took"],
            "total"    => $numProducts,
            "product"  => $showProd ? $pHelper->prepareProductDetail($product) : null,
            "products" => array_map([$pHelper, "prepareListProduct"], $recommended),
            "facets"   => $aggregate ? $helper->transformFacets($result["aggregations"], $attributes, $params, $bucketOrd, $numProducts) : null,
        ]];
    }
}
