<?php

class Crossroads_Elasticsearch_SearchController extends Crossroads_API_Controller_Resource {
    /**
     * @apiParam {Integer} [limit=20]
     * @apiParam {Integer} [page=0]
     * @apiParam {String}  query
     * @apiParam {Boolean} [aggregate=false]
     */
    public function getAll() {
        $req         = $this->getRequest();
        $searchQuery = $req->getParam("query");
        $pageSize    = min((int)$req->getQuery("limit", "20"), 100);
        $page        = max((int)$req->getQuery("page", "1"), 1);
        $aggregate   = trim(strtolower($req->getQuery("aggregate"))) === "true";

        if ( ! $searchQuery) {
            // TODO: Error code
            throw Crossroads_API_ResponseException::create(400, "Missing 'query' parameter.", null);
        }

        $store       = Mage::app()->getStore();
        $client      = Mage::helper("Crossroads_Elasticsearch")->createClient();
        $queryParser = Mage::getModel("Crossroads_Elasticsearch/query_parser");
        $params      = array_diff_key($req->getParams(), [
            "query"     => true,
            "limit"     => true,
            "page"      => true,
            "aggregate" => true,
        ]);
        $helper      = Mage::helper("Crossroads_Elasticsearch");
        $pHelper     = Mage::helper("API/product");
        $resource   = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes = $resource->getSearchableProductAttributes($store);
        $bucketOrd  = $resource->getSortedAttributeValues($attributes, $store);

        $parsedQuery = $queryParser->setAttributes($attributes)
            ->setLocaleCode(Mage::app()->getLocale()->getLocaleCode())
            ->parseQuery($searchQuery);

        $q = Mage::getModel("Crossroads_Elasticsearch/query_products")
                ->setStore($store)
                ->setVisibility(Crossroads_Elasticsearch_Model_Query_Products::VISIBILITY_SEARCH)
                ->setSearchQuery($parsedQuery);

        if(Mage::getSingleton("customer/session")->isLoggedIn()) {
            $q->setCustomerId(Mage::getSingleton("customer/session")->getCustomerId());
        }

        $clone = clone $q;
        $query = Mage::getModel("Crossroads_Elasticsearch/query")
            ->setIndex($store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX), "product")
            ->addSource([ "sku", "name" ])
            ->setPageSize($pageSize)
            ->setPage($page)
            ->setQuery($clone->setFilter($attributes, $params));

        if($aggregate) {
            $query->setAggregations(Mage::getModel("Crossroads_Elasticsearch/query_aggregation_attribute")
                ->setAttributes($attributes)
                ->setParameters($params)
                ->setExcludeQueryConstructor(function($paramKey) use($attributes, $q, $params) {
                    $clone = clone $q;

                    return $clone->setFilter($attributes, array_diff_key($params, [ $paramKey => true ]));
                }));
        }

        if($req->getQuery("sort")) {
            $query->addAttributeSort($attributes, $req->getQuery("sort"));
        }

        $result = $client->search($query->toRequest());

        $numProducts = $result["hits"]["total"];
        $entityIds   = array_map(function($hit) {
            return $hit["_id"];
        }, $result["hits"]["hits"]);

        return [200, [
            "_time"    => $result["took"],
            "total"    => $result["hits"]["total"],
            "products" => array_map([$pHelper, "prepareListProduct"],
                $helper->fetchProducts($entityIds, $pageSize, Mage_Catalog_Model_Product_Visibility::VISIBILITY_IN_CATALOG)),
            "facets"   => $aggregate ? $helper->transformFacets($result["aggregations"], $attributes, $params, $bucketOrd, $numProducts) : null,
        ]];
    }

    public function getItem($id) {
        switch($id) {
        case "suggest":
            return $this->doSuggest();
        default:
            return [404];
        }
    }

    protected function doSuggest() {
        $searchQuery = $this->getRequest()->getParam("query");
        $limit       = min((int)$this->getRequest()->getParam("limit") ?: 10, 20);
        $store       = Mage::app()->getStore();
        $attributes  = Mage::getResourceModel("Crossroads_Elasticsearch/attribute")->getAutocompleteAttributes($store);
        $client      = Mage::helper("Crossroads_Elasticsearch")->createClient();
        $indexName   = $store->getConfig(Crossroads_Elasticsearch_Helper_Data::CONFIG_PRODUCT_INDEX);

        if(empty($attributes)) {
            return [200, [
                "suggestions" => [],
                "_reason"     => "No available attributes.",
            ]];
        }

        $result = $client->search([
            "index" => $indexName,
            "type"  => "product",
            "body"  => [
                "_source" => [
                    "_score"
                ],
                "suggest" => array_merge([
                    "text" => $searchQuery,
                ], array_map(function($attr) use($limit, $searchQuery) {
                    return [
                        "prefix"     => $searchQuery,
                        "completion" => [
                            "field"  => $attr->getCompletionField(),
                            "skip_duplicates" => true,
                            "fuzzy" => [
                                "fuzziness"  => "AUTO",
                                "min_length" => 5,
                            ],
                            "size" => $limit,
                        ],
                    ];
                }, array_combine(array_map(function($a) { return $a->getCode(); }, $attributes), $attributes))),
            ]
        ]);

        return [200, $this->prepareAutocomplete($attributes, $result, $limit)];
    }

    protected function prepareAutocomplete($attributes, $data, $limit) {
        $response    = [];
        $keyedAttrs  = array_combine(array_map(function($a) { return $a->getCode(); }, $attributes), $attributes);
        $filterAttrs = array_filter($keyedAttrs, function($a) { return $a->getIsFilterableInSearch() && $a->isText(); });

        foreach($data["suggest"] as $key => $result) {
            foreach($result as $inner) {
                foreach($inner["options"] as $opt) {
                    $response[] = [
                        "text"  => $opt["text"],
                        "score" => $opt["_score"],
                        "key"   => array_key_exists($key, $filterAttrs) ? $key : "query",
                        "label" => array_key_exists($key, $filterAttrs) ? $filterAttrs[$key]->getFrontendLabel() : null,
                    ];
                }
            }
        }

        usort($response, function($a, $b) {
            // Reverse sort, higher score is better
            return $b["score"] - $a["score"];
        });

        return [
            "suggestions" => $response,
        ];
    }
}
