<?php

class Crossroads_Elasticsearch_Model_Resource_Product extends Crossroads_Elasticsearch_Model_Resource_Abstract {
    const EVENT_REINDEX_ALL      = "crossroads_elasticsearch_index_reindex_all";
    const EVENT_REINDEX_PRODUCTS = "crossroads_elasticsearch_index_reindex_products";

    protected function _construct() {
        $this->_setResource("core_read");
    }

    /**
     * Reindexes all product data for configured stores.
     */
    public function reindexAll() {
        $sql        = $this->getReadConnection();
        $attrModel  = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes = $attrModel->getIndexableAttributes(Mage_Catalog_Model_Product::ENTITY);
        $attrLUT    = $attrModel->getMultiselectLUT($attributes);
        $stores     = Mage::helper("Crossroads_Elasticsearch")->getStores([ Mage_Catalog_Model_Product::ENTITY ]);
        $client     = Mage::helper("Crossroads_Elasticsearch")->createClient();
        $index      = Mage::getModel("Crossroads_Elasticsearch/product_index");
        $indices    = [];
        $storeCount = [];

        Mage::log(sprintf("Crossroads_Elasticsearch: Indexing all products for stores [%s].", implode(", ", array_map(function($s) { return $s->getId(); }, $stores))));

        foreach($stores as $store) {
            $indices[$store->getId()]    = $index->createIndex($store, $attributes, $client);
            $storeCount[$store->getId()] = 0;
        }

        $query     = $sql->query($this->createProductQuery($sql, $stores, $attributes));
        $resultSet = new Crossroads_Elasticsearch_Model_Resource_Product_DataIterator($query, $attributes, $attrLUT);
        $manager   = new Crossroads_Elasticsearch_Model_BatchManager($client);

        foreach($resultSet as $product) {
            $manager->queue([ "index" => [
                "_index"   => $indices[$product["store_id"]],
                "_routing" => (string)$product["entity_id"],
                "_type"    => "product",
                "_id"      => (string)$product["entity_id"],
            ]], $product);

            $storeCount[$product["store_id"]]++;
        }

        // TODO: Send product ids
        Mage::dispatchEvent(self::EVENT_REINDEX_ALL, [
            "manager" => $manager,
            "indices" => $indices,
            "stores"  => $stores,
        ]);

        $manager->sendBuffer();

        // Debug log for all
        foreach($stores as $store) {
            Mage::log(sprintf("Crossroads_Elasticsearch: Inserted %d products into index %s for store %d.", $storeCount[$store->getId()], $indices[$store->getId()], $store->getId()));
        }

        if($manager->hasErrors()) {
            Mage::log(sprintf("Crossroads_Elasticsearch: Got %d errors while indexing.", count($manager->getErrors())));
        }

        $manager->clear();

        foreach($stores as $store) {
            $index->updateAlias($store, $indices[$store->getId()], $client);
        }

        Mage::log(sprintf("Crossroads_Elasticsearch: Finished indexing all products for stores [%s].", implode(", ", array_map(function($s) { return $s->getId(); }, $stores))));
    }

    /**
     * Ensures an index exists, then pushes all products to it.
     *
     * @param  Array<integer>|null
     */
    public function indexProducts($productIds = null) {
        $indices    = [];
        $storeCount = [];
        $indexedIds = [];
        $attrModel  = Mage::getResourceModel("Crossroads_Elasticsearch/attribute");
        $attributes = $attrModel->getIndexableAttributes(Mage_Catalog_Model_Product::ENTITY);
        $attrLUT    = $attrModel->getMultiselectLUT($attributes);
        $stores     = Mage::helper("Crossroads_Elasticsearch")->getStores([ Mage_Catalog_Model_Product::ENTITY ]);
        $sql        = $this->getReadConnection();
        $client     = Mage::helper("Crossroads_Elasticsearch")->createClient();
        $index      = Mage::getModel("Crossroads_Elasticsearch/product_index");

        Mage::log("Crossroads_Elasticsearch: " . (empty($productIds) ? "Indexing all products" : sprintf("Indexing products with ids [%s]", implode(", ", $productIds))));

        foreach($stores as $store) {
            $indexName = $index->ensureAlias($store, $attributes, $client);

            $indices[$store->getId()]    = $indexName;
            $storeCount[$store->getId()] = 0;
            $indexedIds[$store->getId()] = [];
        }

        $query     = $sql->query($this->createProductQuery($sql, $stores, $attributes, $productIds));
        $resultSet = new Crossroads_Elasticsearch_Model_Resource_Product_DataIterator($query, $attributes, $attrLUT);
        $manager   = new Crossroads_Elasticsearch_Model_BatchManager($client);

        foreach($resultSet as $product) {
            $manager->queue([
                "index" => [
                    "_index"   => $indices[$product["store_id"]],
                    "_routing" => (string)$product["entity_id"],
                    "_type"    => "product",
                    "_id"      => (string)$product["entity_id"],
                ]
            ], $product);

            $storeCount[$product["store_id"]]++;
            $indexedIds[$product["store_id"]][] = $product["entity_id"];
        }

        Mage::dispatchEvent(self::EVENT_REINDEX_PRODUCTS, [
            "manager"     => $manager,
            "indices"     => $indices,
            "stores"      => $stores,
            "product_ids" => array_unique(call_user_func_array("array_merge", $indexedIds)),
        ]);

        $manager->sendBuffer();

        // Debug log for all
        foreach($stores as $store) {
            Mage::log(sprintf("Crossroads_Elasticsearch: Inserted %d products into index %s for store %d with %d errors.", $storeCount[$store->getId()], $indices[$store->getId()], $store->getId()));
        }

        if($manager->hasErrors()) {
            Mage::log(sprintf("Crossroads_Elasticsearch: Got %d errors while indexing.", count($manager->getErrors())));
        }

        $manager->clear();

        if( ! empty($productIds)) {
            foreach($stores as $store) {
                $toDelete = array_diff($productIds, $indexedIds[$store->getId()]);

                foreach($toDelete as $productId) {
                    $manager->queue([ "delete" => [
                        "_index" => $indices[$store->getId()],
                        "_routing" => (string)$productId,
                        "_type"  => "product",
                        "_id"    => $productId
                    ]]);
                }
            }

            $manager->sendBuffer();

            if($manager->hasErrors()) {
                Mage::log(sprintf("Crossroads_Elasticsearch: Got %d errors while indexing.", count($manager->getErrors())));
            }
        }
    }

    /**
     * Deletes an array of entity_ids from elasticsearch index.
     *
     * @param  Array<integer>|null
     */
    public function deleteProducts($productIds = null) {
        $attributes = Mage::getResourceModel("Crossroads_Elasticsearch/attribute")->getIndexableAttributes(Mage_Catalog_Model_Product::ENTITY);
        $stores     = Mage::helper("Crossroads_Elasticsearch")->getStores([ Mage_Catalog_Model_Product::ENTITY ]);
        $client     = Mage::helper("Crossroads_Elasticsearch")->createClient();
        $indices    = [];

        foreach($stores as $store) {
            // TODO: This one is not necessary, exclude stores which do not have an index
            $indices[$store->getId()] = $this->ensureAlias($store, $attributes, $client);
        }

        $manager = new Crossroads_Elasticsearch_Model_BatchManager($client);

        foreach($productIds as $productId) {
            foreach($stores as $store) {
                $manager->queue([ "delete" => [
                    "_index"   => $indices[$store->getId()],
                    "_routing" => $productId,
                    "_type"    => "product",
                    "_id"      => $productId
                ]]);
            }
        }

        $manager->sendBuffer();
    }

    /**
     * Creates a query fetching products based on a list of stores and attributes, optionally filtered by product ids.
     *
     * It will fetch a row for each (store, attribute) combination with the following base attributes included:
     *
     *  * store_id
     *  * website_id
     *  * entity_id
     *  * type_id
     *  * sku
     *  * created_at
     *  * updated_at
     *
     * The list of attributes require the following data for each attribute:
     *
     *  * attribute_id
     *  * backend_type
     *
     * @param  Zend_Db_Adapter_Abstract
     * @param  Array<Mage_Core_Model_Store>
     * @param  Array<Attribute>
     * @param  Array<integer>
     * @return Zend_Db_Select
     */
    protected function createProductQuery($sql, $stores, $attributes, $productIds = null) {
        $select = $sql->select()
            ->from(["s" => $this->createEntityIdStoreQuery($sql, $stores, $productIds)], ["entity_id", "store_id", "website_id", "visibility"])
            ->join(["p" => $this->getTable("catalog/product")], "p.entity_id = s.entity_id", ["type_id", "sku", "created_at", "updated_at"])
            ->from(["a" => $this->getTable("catalog/eav_attribute")], ["attribute_id"]);

        $select = array_reduce($this->getAttributeTypes($attributes), function($select, $type) use($sql) {
            $global = $type."__global";
            $store  = $type."__store";

            return $select->joinLeft(
                [$global => $this->getValueTable("catalog/product", $type)],
                sprintf("%1\$s.entity_id = p.entity_id AND %1\$s.attribute_id = a.attribute_id AND %1\$s.store_id = 0", $sql->quoteIdentifier($global)),
                []
            )
            ->joinLeft(
                [$store => $this->getValueTable("catalog/product", $type)],
                sprintf("%1\$s.entity_id = p.entity_id AND %1\$s.attribute_id = a.attribute_id AND %1\$s.store_id = s.store_id", $sql->quoteIdentifier($store)),
                []
            )
            ->columns([
                $type => $sql->getIfNullSql(
                    $sql->quoteIdentifier($store.'.value'),
                    $sql->quoteIdentifier($global.'.value')
                )
            ]);
        }, $select);

        // Join option values, these are stored in int columns, having frontend_input as some kind of select
        // We always have 'int' here, see `getAttributeTypes`
        $select = $select->joinLeft(
                ["eao" => $this->getTable("eav/attribute_option")],
                sprintf("%1\$s.attribute_id = a.attribute_id AND (%2\$s.value = %1\$s.option_id OR %3\$s.value = %1\$s.option_id)", $sql->quoteIdentifier("eao"), $sql->quoteIdentifier("int__global"), $sql->quoteIdentifier("int__store")),
                []
            )
            ->joinLeft(
                ["eaov__global" => $this->getTable("eav/attribute_option_value")],
                sprintf("%1\$s.option_id = %2\$s.option_id AND %1\$s.store_id = 0", $sql->quoteIdentifier("eaov__global"), $sql->quoteIdentifier("eao")),
                []
            )
            ->joinLeft(
                ["eaov__store" => $this->getTable("eav/attribute_option_value")],
                sprintf("%1\$s.option_id = %2\$s.option_id AND %1\$s.store_id = s.store_id", $sql->quoteIdentifier("eaov__store"), $sql->quoteIdentifier("eao")),
                []
            )
            ->columns([
                "option_value" => $sql->getIfNullSql(
                    $sql->quoteIdentifier('eaov__store.value'),
                    $sql->quoteIdentifier('eaov__global.value')
                ),
                // We include the id so we can decide if we have duplicates or not in a multiselect
                "option_value_id" => $sql->getIfNullSql(
                    $sql->quoteIdentifier('eaov__store.value_id'),
                    $sql->quoteIdentifier('eaov__global.value_id')
                )
            ]);

        $select = $select->where("a.attribute_id IN (?)", array_map(function($a) { return $a->getId(); }, $attributes))
            ->group(["p.entity_id", "s.store_id", "a.attribute_id", "eaov__global.value_id", "eaov__store.value_id"]);

        // This order is necessary since we do not want to load the whole result set into memory at the same time
        // it will enable us to assemble one product entity at a time
        return $select->order([
            "p.entity_id ASC",
            "s.store_id ASC",
            "a.attribute_id ASC"
        ]);
    }

    /**
     * Returns a list of attribute backend types, always includes the 'int' type.
     *
     * @param  Array<Crossroads_Elasticsearch_Model_Attribute>
     * @return Array<string>
     */
    protected function getAttributeTypes($attributes) {
        return array_unique(array_merge(["int"], array_map(function($a) { return $a->getBackendType(); }, $attributes)));
    }
}
