<?php

/**
 * Abstract resource for resources with a store-specific sub-table which contains overrides.
 */
abstract class Crossroads_DynamicCategories_Model_Resource_Abstract extends Mage_Core_Model_Resource_Db_Abstract {
    protected $_storeEntity   = "";
    protected $_storeEntityPk = "";
    protected $_storeEntityFk = "store_id";

    public function getStoreTable() {
        if(empty($this->_storeEntity)) {
            Mage::throwException(Mage::helper("core")->__("Empty store entity name"));
        }

        return $this->getTable($this->_storeEntity);
    }

    public function getStoreEntityPk() {
        if(empty($this->_storeEntityPk)) {
            Mage::throwException(Mage::helper("core")->__("Empty store entity primary key"));
        }

        return $this->_storeEntityPk;
    }

    public function getStoreEntityFk() {
        if(empty($this->_storeEntityFk)) {
            Mage::throwException(Mage::helper("core")->__("Empty store entity foreign key"));
        }

        return $this->_storeEntityFk;
    }

    /**
     * Retrieve select object for load object data
     *
     * @param string $field
     * @param mixed $value
     * @param Mage_Core_Model_Abstract $object
     * @return Zend_Db_Select
     */
    public function _getLoadSelect($field, $value, $object) {
        $read   = $this->_getReadAdapter();
        $field  = $read->quoteIdentifier("e." . $field);
        $select = $read->select()
            ->from(["e" => $this->getMainTable() ])
            ->where($field . "=?", $value);

        if($object->getStoreId()) {
            $columns = [];
            $cols    = $read->describeTable($this->getStoreTable());
            $skip    = [$this->getStoreEntityFk(), $this->getStoreEntityPk()];

            foreach(array_keys($cols) as $key) {
                if( ! in_array($key, $skip)) {
                    $columns[$key] = new Zend_Db_Expr(sprintf(
                        "COALESCE(%s, %s)",
                        $read->quoteIdentifier("s.".$key),
                        $read->quoteIdentifier("e.".$key)
                    ));
                }
            }

            $columns["store_id"] = new Zend_Db_Expr($read->quoteInto("?", $object->getStoreId()));

            $select->joinLeft([ "s" => $this->getStoreTable() ],
                $read->quoteInto($read->quoteIdentifier("s.".$this->getStoreEntityFk())." = ?", $object->getStoreId()).
                " AND ".
                $read->quoteIdentifier("e.".$this->getIdFieldName())." = ".$read->quoteIdentifier("s.".$this->getStoreEntityPk()), [])
                ->columns($columns);
        }

        return $select;
    }

    public function loadStoreAttributeValues($object) {
        if( ! $object->getId() || ! $object->getStoreId()) {
            return [];
        }

        $read       = $this->_getReadAdapter();
        $idField    = $read->quoteIdentifier("s.".$this->getStoreEntityPk());
        $storeField = $read->quoteIdentifier("s.".$this->getStoreEntityFk());
        $select     = $read->select()
            ->from(["s" => $this->getStoreTable()])
            ->where($idField    ."=?", $object->getId())
            ->where($storeField ."=?", $object->getStoreId());

        return $read->fetchRow($select) ?: [];
    }

    public function loadDefaultAttributeValues($object) {
        if( ! $object->getId()) {
            return [];
        }

        $read   = $this->_getReadAdapter();
        $field  = $read->quoteIdentifier("e.".$this->getIdFieldName());
        $select = $read->select()
            ->from(["e" => $this->getMainTable()])
            ->where($field . "=?", $object->getId());

        return $read->fetchRow($select) ?: [];
    }

    /**
     * Override of Mage_Core_Model_Resource_Db_Abstract::save() to incorporate parts of
     * Mage_Core_Eav_Model_Entity_Resource_Abstract::save().
     */
    public function save(Mage_Core_Model_Abstract $object) {
        if ($object->isDeleted()) {
            return $this->delete($object);
        }

        $this->_serializeFields($object);
        $this->_beforeSave($object);
        $this->_checkUnique($object);

        $this->_processSaveData($object, $this->_collectSaveData($object));

        $this->unserializeFields($object);
        $this->_afterSave($object);

        return $this;
    }

    protected function _processSaveData(Varien_Object $object, $tableData) {
        $data  = $tableData["main"];
        $write = $this->_getWriteAdapter();

        if( ! is_null($object->getId()) && ( ! $this->_useIsObjectNew || ! $object->isObjectNew())) {
            // Update
            $condition = $write->quoteInto($this->getIdFieldName()."=?", $object->getId());

            if($this->_isPkAutoIncrement) {
                unset($data[$this->getIdFieldName()]);

                $write->update($this->getMainTable(), $data, $condition);
            }
            else {
                $select = $write->select()
                    ->from($this->getMainTable(), [$this->getIdFieldName()])
                    ->where($condition);

                if($write->fetchOne($select) !== false) {
                    unset($data[$this->getIdFieldName()]);

                    if( ! empty($data)) {
                        $write->update($this->getMainTable(), $data, $condition);
                    }
                }
                else {
                    $write->insert($this->getMainTable(), $data);
                }
            }

            $this->_updateStoreData($object, $tableData["store"]);
        }
        else {
            // Insert
            if($this->_isPkAutoIncrement) {
                unset($data[$this->getIdFieldName()]);
            }

            $write->insert($this->getMainTable(), $data);

            $object->setId($write->lastInsertId($this->getMainTable()));

            if($this->_useIsObjectNew) {
                $object->isObjectNew(false);
            }

            $this->_updateStoreData($object, $tableData["store"]);
        }
    }

    protected function _updateStoreData(Varien_Object $object, $data) {
        $write      = $this->_getWriteAdapter();
        $condition  = $write->quoteInto($this->getStoreEntityPk()."=? AND ".$this->getStoreEntityFk()."=?", $object->getId(), $object->getStoreId());
        $updateData = $data;

        unset($updateData[$this->getStoreEntityPk()]);
        unset($updateData[$this->getStoreEntityFk()]);

        if(empty($updateData)) {
            $write->delete($this->getStoreTable(), $condition);
        }
        else {
            $select = $write->select()
                ->from($this->getStoreTable(), [$this->getStoreEntityPk()])
                ->where($condition);

            if($write->fetchOne($select) !== false) {
                $write->update($this->getStoreTable(), $updateData, $condition);
            }
            else {
                $write->insert($this->getStoreTable(), $data);
            }
        }
    }

    protected function _collectSaveData(Varien_Object $object) {
        $idField    = $this->getIdFieldName();
        $mainTable  = [];
        $storeTable = [];
        $write      = $this->_getWriteAdapter();
        $isNew      = !$object->getId();

        if($object->getStoreId()) {
            $baseFields  = $write->describeTable($this->getMainTable());
            $storeFields = $write->describeTable($this->getStoreTable());

            foreach($baseFields as $key => $field) {
                // Only write it if it is a global value or the object is new and the column is not nullable
                if($object->hasData($key) && ($isNew && ! empty($field["NULLABLE"]) || ! array_key_exists($key, $storeFields))) {
                    $fieldValue = $object->getData($key);

                    if ($fieldValue instanceof Zend_Db_Expr) {
                        $mainTable[$key] = $fieldValue;
                    } else {
                        if (null !== $fieldValue) {
                            $mainTable[$key] = $write->prepareColumnValue($field, $this->_prepareTableValueForSave($fieldValue, $field["DATA_TYPE"]));
                        }
                        elseif( ! empty($field["NULLABLE"])) {
                            $mainTable[$key] = null;
                        }
                    }
                }
            }

            foreach($storeFields as $key => $field) {
                if($object->hasData($key)) {
                    $fieldValue = $object->getData($key);

                    if ($fieldValue instanceof Zend_Db_Expr) {
                        $storeTable[$key] = $fieldValue;
                    } else {
                        if (null !== $fieldValue) {
                            $storeTable[$key] = $write->prepareColumnValue($field, $this->_prepareTableValueForSave($fieldValue, $field["DATA_TYPE"]));
                        }
                        elseif( ! empty($field["NULLABLE"])) {
                            $storeTable[$key] = null;
                        }
                    }
                }
            }

            $storeTable[$this->getStoreEntityPk()] = $object->getData($this->getIdFieldName());
            $storeTable[$this->getStoreEntityFk()] = $object->getStoreId();
        }
        else {
            $mainTable = $this->_prepareDataForTable($object, $this->getMainTable());
        }

        return [
            "main"  => $mainTable,
            "store" => $storeTable,
        ];
    }
}
