<?php

class Crossroads_DibsD2_Helper_Data extends Mage_Core_Helper_Abstract {

    /**
     * Event triggered when payment status has been updated for a given order from callback.
     *
     * Params:
     *  * order          Mage_Sales_Model_Order
     *  * payment        Payment instance for order
     *  * invoice_status New invoice status, one of the Crossroads_Dibs_Helper_Data::INVOICE_STATUS_* constants.
     *  * invoice_no     Dibs invoice number
     */
    const EVENT_POST_ORDER_STATUS_UPDATE = "crossroads_dibsd2_post_order_status_update";
    const CONFIG_PATH_MIN_ORDER_TOTAL = "payment/DibsD2/min_order_total";
    const CONFIG_PATH_MERCHANT = "payment/DibsD2/merchant";
    const CONFIG_PATH_API_USER = "payment/DibsD2/api_user";
    const CONFIG_PATH_API_PASSWORD = "payment/DibsD2/api_password";
    const PAYMENT_METHOD_CODE = "DibsD2";

    /**
     * For list of parameters, se: http://tech.dibspayment.com/D2/Hosted/Output_parameters/Return_parameters
     */
    const FIELD_PAYMENT_STATE = "DibsD2_payment_state";
    const FIELD_TRANSACT = "DibsD2_transact";
    const FIELD_STATUSCODE = "DibsD2_statuscode"; // Se: http://tech.dibspayment.com/nodeaddpage/toolboxstatuscodes
    const FIELD_ACQUIRER = "DibsD2_acquirer";
    const FIELD_CARDEXPDATE = "DibsD2_cardexpdate";
    const FIELD_CARDNOMASK = "DibsD2_cardnomask";
    const FIELD_CARDPREFIX = "DibsD2_cardprefix";
    const FIELD_CARDCOUNTRY = "DibsD2_cardcountry";
    const FIELD_FORM_DATA = "DibsD2_form_data";
    const FIELD_CALLBACK_DATA = "DibsD2_callback_data";

    // Payment states
    const STATE_AUTHORIZATION_WAITING = 1;
    const STATE_AUTHORIZATION_APPROVED = 2;
    const STATE_AUTHORIZATION_DECLINED = 3;
    const STATE_PAYMENT_WAITING = 11;
    const STATE_PAYMENT_APPROVED = 12;
    const STATE_PAYMENT_DECLINED = 13;
    const STATE_PAYMENT_CANCELED = 21;
    const STATE_PAYMENT_REFYNDED = 22;
    const STATE_NOT_APPLICABLE = 91;
    const STATE_PAYMENT_ERROR = 99;

    const LOGGING_FILENAME_NORMAL = "dibsd2.normal.log";
    const LOGGING_FILENAME_RESULTS = "dibsd2.results.log";
    const DEBUGGING_MODE = 1;

    protected $_payment_fields = array();
    protected $_payment_states = array();

    public function __construct()
    {
        $this->_payment_states = array(
            self::STATE_AUTHORIZATION_WAITING => 'Waiting for authorization',
            self::STATE_AUTHORIZATION_APPROVED => 'Authorization approved',
            self::STATE_AUTHORIZATION_DECLINED => 'Authorization declined',
            self::STATE_PAYMENT_WAITING => 'Waiting for confirmation',
            self::STATE_PAYMENT_APPROVED => 'Payment approved',
            self::STATE_PAYMENT_DECLINED => 'Payment declined',
            self::STATE_PAYMENT_CANCELED => 'Payment canceled',
            self::STATE_PAYMENT_REFYNDED => 'Payment refunded',
            self::STATE_NOT_APPLICABLE => 'Not a payment status',
            self::STATE_PAYMENT_ERROR => 'Payment error'
        );
    }

    /**
     * Registers the payment method with the Crossroads_API.
     */
    public function registerPayment(Varien_Event_Observer $observer)
    {
        $observer->getRegistry()->addMethod(new Crossroads_DibsD2_Model_Payment());
    }

    /**
     * Fetches all DibsD2 specific fields
     *
     * @return array
     */
    public function getPaymentFields()
    {
        if (empty($this->_payment_fields)) {
            $this->_payment_fields = array(
                self::FIELD_TRANSACT,
                self::FIELD_ACQUIRER,
                self::FIELD_CARDEXPDATE,
                self::FIELD_CARDNOMASK,
                self::FIELD_CARDPREFIX,
                self::FIELD_CARDCOUNTRY,
                self::FIELD_STATUSCODE,
            );
        }
        return $this->_payment_fields;
    }

    public function getLastTransaction($payment)
    {
        $trn = Mage::getModel('sales/order_payment_transaction')
                ->getCollection()
                ->addAttributeToFilter('order_id', array('eq' => $payment->getOrder()->getEntityId()))
                ->setOrder('created_at', Varien_Data_Collection::SORT_ORDER_DESC)
                ->getFirstItem();
        return $payment->getTransaction($trn->getTxnId());
    }

    public function getTranslatedPaymentStateByCode($code)
    {
        if (array_key_exists($code, $this->_payment_states)) {
            return $this->_payment_states[$code];
        } else {
            return "Invalid payment state [{$code}]";
        }
    }

    public function getIsTest($storeId = null)
    {
        if (is_null($storeId)) {
            $storeId = Mage::app()->getStore()->getId();
        }
        return (self::DEBUGGING_MODE ? true : false) && (Mage::getStoreConfig('payment/DibsD2/test', $storeId) ? true : false);
    }

    public function getReadableCardNo($prefix, $cardmask)
    {
        if (!empty($prefix) && !empty($cardmask)) {
            $pattern = 'xxxx xxxx xxxx xxxx';
            $cardPrefix = str_pad($prefix, 6, 'x');
            $cardNo = $cardPrefix . substr(str_pad($cardmask, 16, 'x'), 6);
            return vsprintf(str_replace('x', '%s', $pattern), str_split($cardNo));
        }
        return '';
    }

    public function getShallowArray($inputKey, $inputVal)
    {
        $output = [];
        if (is_array($inputVal)) {
            foreach ($inputVal as $newKey => $newVal) {
                if (empty($inputKey)) {
                    $realKey = $newKey;
                } else {
                    $realKey = "{$inputKey} - {$newKey}";
                }
                $output = array_merge($output, $this->getShallowArray($realKey, $newVal));
            }
        } elseif (!empty($inputKey)) {
            $output[$inputKey] = $inputVal;
        }
        return $output;
    }

    public function sendFraudEmail($order, $transactionInfo)
    {
        $incrementId = $order->getIncrementId();
        $subject = "Payment of order {$incrementId} has been marked as suspect by Dibs";
        $body = "Transaction contains the following information:\n";
        foreach ($this->getShallowArray('', $transactionInfo) as $key => $val) {
            $body .= "{$key} : {$val}\n";
        }
        $body .= "\n\n(detta email skickades till IT och kundtjänst)";
        mail("it@crossroads.se", $subject, $body);
        mail("kundtjanst@crossroads.se", $subject, $body);
    }

}
