<?php

class Crossroads_DibsD2_Model_Dibsd2 extends Mage_Payment_Model_Method_Abstract {

    protected $_code = Crossroads_DibsD2_Helper_Data::PAYMENT_METHOD_CODE;
    // Properties from mage model
    protected $_isGateway = true;
    // Required to provide the ability to call the API, without this capture() would not be called
    protected $_isOnline = true;
    // We do not perform separate reservations
    protected $_canUseForMultishipping = false;
    protected $_canUseInternal = true;
    protected $_canAuthorize = true;
    // We do not allow refunds in Magento, this process is performed outside of Magento
    protected $_canRefund = false;
    // We need to be able to review payment since we send it to Dibs for review.
    protected $_canReviewPayment = true;
    protected $_canCapture = true;
    protected $_canVoid = true;
    protected $_canCancel = true;
    protected $_canFetchTransactionInfo = true;
    protected $_infoBlockType = "DibsD2/info";
    protected $_formBlockType = "DibsD2/form";
    // Set by authorize() and read by getOrderPlaceRedirectUrl()
    protected $_last_increment_id = null;

    /**
     * Check whether payment method can be used
     *
     * @param  Mage_Sales_Model_Quote|null
     * @return bool
     */
    public function isAvailable($quote = null)
    {
        $minimumOrderTotal = Mage::getStoreConfig(Crossroads_DibsD2_Helper_Data::CONFIG_PATH_MIN_ORDER_TOTAL, $quote ? $quote->getStoreId() : null);

        return (!$quote || $quote->getBaseGrandTotal() >= $minimumOrderTotal) && parent::isAvailable($quote);
    }

    /**
     * Authorization function for the payment, executed before the order is approved.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  float
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function authorize(Varien_Object $payment, $amount)
    {
        $order = $payment->getOrder();
        $storeId = $order->getStoreId();
        $incrementId = $order->getIncrementId();
        $this->_last_increment_id = $incrementId;

        $payment->setTransactionId($incrementId);
        $payment->setIsTransactionPending(true);
        $payment->setIsTransactionClosed(false);

        $billingAddress = $order->getBillingAddress();
        $APIHelper = Mage::helper('DibsD2/API');

        $street = $billingAddress->getStreet();
        if (!is_array($street)) {
            $street = [$street];
        }
        $data = [
            "merchant" => Mage::getStoreConfig('payment/DibsD2/merchant', $storeId),
            "orderid" => $incrementId,
            "amount" => intval($amount * 100),
            "currency" => $APIHelper->getNumericCurrencyId($order->getBaseCurrencyCode()),
            "billingFirstName" => $billingAddress->getFirstname(),
            "billingLastName" => $billingAddress->getLastname(),
            "billingAddress" => $street[0],
            "billingAddress2" => empty($street[1]) ? '' : $street[1],
            "billingPostalCode" => $billingAddress->getPostcode(),
            "billingPostalPlace" => $billingAddress->getCity(),
            "accepturl" => $APIHelper->getAcceptUrl($payment),
            "cancelurl" => $APIHelper->getCancelUrl($payment),
            "callbackurl" => $APIHelper->getCallbackUrl($payment),
            "notifyurl" => $APIHelper->getNotifyUrl($payment),
            "lang" => Mage::getStoreConfig('payment/DibsD2/lang', $storeId),
        ];

        $decorator = Mage::getStoreConfig('payment/DibsD2/decorator', $storeId);
        if (!empty($decorator) && $decorator !== 'custom') {
            $data["decorator"] = $decorator;
        }

        $account = Mage::getStoreConfig('payment/DibsD2/account', $storeId);
        if (!empty($account)) {
            $data["account"] = $account;
        }

        $capturenow = Mage::getStoreConfig('payment/DibsD2/capturenow', $storeId);
        if (!empty($capturenow)) {
            $data["capturenow"] = $capturenow;
        }

        $paytype = Mage::getStoreConfig('payment/DibsD2/paytype', $storeId);
        if (!empty($paytype)) {
            $data["paytype"] = $paytype;
        }

        $test = Mage::getStoreConfig('payment/DibsD2/test', $storeId);
        if (!empty($test)) {
            $data["test"] = $test;
        }

        $settings = [
            'merchant' => ['md5_prio' => 1],
            'orderid' => ['md5_prio' => 2],
            'currency' => ['md5_prio' => 3],
            'amount' => ['md5_prio' => 4]
        ];
        $data["md5key"] = Mage::helper('DibsD2/API')->getMD5Key($storeId, $settings, $data);

        $payment->setTransactionAdditionalInfo([
            Crossroads_DibsD2_Helper_Data::FIELD_TRANSACT => $incrementId,
            Crossroads_DibsD2_Helper_Data::FIELD_FORM_DATA => $data
        ], null);
        $payment->save();

        return $this;
    }

    /**
     * Captures amount from Dibs from a previous reservation made using authorize.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @param  float
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function capture(Varien_Object $payment, $amount)
    {
        // todo: Do an actual capture?
        $payment->setIsPaid(true);
        return $this;
    }

    /**
     * Cancels authorization with Dibs.
     *
     * @param  Mage_Sales_Model_Order_Payment
     * @return Mage_Payment_Model_Method_Abstract
     */
    public function cancel(Varien_Object $payment)
    {
        // If we got here from callback, we already have confirmation that payment is canceled.
        // No need to do API-call to Dibs to check for payment status
        if (!$this->getInHandleCallback()) {
            // Order canceled manually.
            // Do not call Dibs for now, just continue and cancel order
        }

        $payment->setIsTransactionApproved(false);
        $payment->setIsTransactionPending(false);
        $payment->setIsTransactionClosed(true);

        // Add transaction and set order state with proper messages
        $payment->addTransaction(
                Mage_Sales_Model_Order_Payment_Transaction::TYPE_VOID,
                null,
                true,
                $payment->getMessage() ? $payment->getMessage() : Mage::helper('DibsD2')->__('Authorization canceled by customer.')
        )->save();

        // Fix for stupid Mage_Sales_Model_Order_Payment::$_transactionsLookup not updating when doing addTransaction()
        $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_VOID);

        $order = $payment->getOrder();
        $order->setState(
                Mage_Sales_Model_Order::STATE_CANCELED,
                true,
                $payment->getMessage() ? $payment->getMessage() : Mage::helper('DibsD2')->__('Order canceled.')
        );

        return $this;
    }

    public function void(Varien_Object $payment)
    {
        return $this->cancel($payment);
    }

    public function acceptPayment(Mage_Payment_Model_Info $payment)
    {
        $auth = $payment->lookupTransaction(null, Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH);

        if (!$auth || !$auth->getTxnId()) {
            Mage::throwException("Crossroads_DibsD2: Failed to obtain authorization id for order '{$payment->getOrder()->getIncrementId()}' when accepting payment.");
        }

        $payment->setIsTransactionApproved(true);
        $payment->setIsTransactionClosed(true);
        $payment->setStatus(Mage_Payment_Model_Method_Abstract::STATUS_APPROVED);

        return true;
    }

    public function getOrderPlaceRedirectUrl()
    {

        $url = Mage::app()->getStore()->getBaseUrl(Mage_Core_Model_Store::URL_TYPE_WEB);
        $url .= 'dibsd2/redirect/index/';

        $incrementId = $this->_last_increment_id;
        if (!empty($incrementId)) {
            $url .= 'incrementid/' . urlencode($incrementId) . '/';
        }
        return $url;
    }

    public function fetchTransactionInfo(Mage_Payment_Model_Info $payment, $transactionId)
    {
        // Fix for not ending up in an infinite loop when capturing payment from Dibs callback
        if ($this->getInHandleCallback()) {
            return;
        }

        // For test, override values here:
        if (false) {
            $this->handleCallback(
                    $payment,
                    ['*' => 'Someone went bananas and the cucumber escaped!']
            );
            return array();
        }

        $order = $payment->getOrder();
        $storeId = $order->getStoreId();
        $incrementId = $order->getIncrementId();
        $APIModel = Mage::getModel('DibsD2/API');

        // If $transactionId == $incrementId we only made the call to Dibs but never got a callback.
        // The first transaction saved have the same transaction id (txn_id) as increment_id from order.
        if ($transactionId === $incrementId) {

            $transaction = Mage::helper("DibsD2")->getLastTransaction($payment);
            $formData = $transaction->getAdditionalInformation(Crossroads_DibsD2_Helper_Data::FIELD_FORM_DATA);

            if (empty($formData)) {
                $ex = new Crossroads_DibsD2_Exception("DibsD2::fetchTransactionInfo() Missing form data.");
                $ex->setExtendedInfo([
                    'store_id' => $storeId,
                    'increment_id' => $incrementId,
                    'additional_information' => $transaction->getAdditionalInformation()
                ]);
                throw $ex;
            }

            $params1 = [
                'amount' => $formData['amount'],
                'currency' => $formData['currency'],
                'merchant' => $formData['merchant'],
                'orderid' => $formData['orderid']
            ];
            /**
             * We should get the following data from Dibs:
             * [approvalcode] => string
             * [status] => int
             * [transact] => int
             * OR just a string with some error message
             */
            $result1 = $APIModel->postToDIBS($storeId, 'transinfo', $params1);

            if (empty($result1['translated']['transact'])) {
                throw new Crossroads_DibsD2_Exception("DibsD2::fetchTransactionInfo() Got no 'transact' from Dibs.");
            }
            $transactionId = $result1['translated']['transact'];
        }

        $params2 = [
            'merchant' => $formData['merchant'],
            'transact' => $transactionId
        ];
        /**
         * This function return a single value stating the status of the transaction:
         */
        $result2 = $APIModel->postToDIBS($storeId, 'transstatus', $params2);
        $realStatuscode = trim($result2['raw']);

        $params3 = [
            'merchant' => Mage::getStoreConfig('payment/DibsD2/merchant', $storeId),
            'transact' => $transactionId
        ];
        /**
         * We should get the same full amount of data as the a normal system callback from Dibs
         */
        $result3 = $APIModel->postToDIBS($storeId, 'callback', $params3);

        if (empty($result3['translated'])) {
            $ex = new Crossroads_DibsD2_Exception("DibsD2::fetchTransactionInfo() Got no result from API-call to 'callback'.");
            $ex->setExtendedInfo([
                'result' => $result3
            ]);
            throw $ex;
        }

        // Replace statuscode in callback-data, since that code reflects what we got when callback was originally made
        $result3['translated']['statuscode'] = $realStatuscode;

        // For test, override values here:
        if (false) {
            $result3['translated']['statuscode'] = '3';
        }

        $this->handleCallback($payment, $result3['translated']);

        return parent::fetchTransactionInfo($payment, $transactionId);
    }

    public function handleCallback($payment, $params, $doVerify = true)
    {
        // This value is set to stop infinite looping since some system calls are made to functions already called by callback from Dibs.
        $this->setInHandleCallback(true);

        $order = $payment->getOrder();
        $storeId = $order->getStoreId();
        $orderState = $order->getState();
        $incrementId = $order->getIncrementId();

        $APIModel = Mage::getModel('DibsD2/API');
        if ($doVerify) {
            $APIModel->verifyCallback($storeId, $params);
        }

        $transactionId = $params['transact'];
        $statuscode = $params['statuscode'];
        $paymentState = Mage::helper('DibsD2/API')->getPaymentStateByCode($statuscode);

        $transaction = Mage::helper("DibsD2")->getLastTransaction($payment);
        $oldPaymentState = $transaction->getAdditionalInformation(Crossroads_DibsD2_Helper_Data::FIELD_PAYMENT_STATE);
        $oldTransactionId = $transaction->getAdditionalInformation(Crossroads_DibsD2_Helper_Data::FIELD_TRANSACT);
        $oldStatuscode = $transaction->getAdditionalInformation(Crossroads_DibsD2_Helper_Data::FIELD_STATUSCODE);
        $oldTransactionType = $transaction->getTxnType();
        $oldTxnId = $transaction->getTxnId();

        // Only continue if we have new data.
        if ($paymentState == $oldPaymentState && $transactionId == $oldTransactionId && $statuscode == $oldStatuscode) {
            return;
        }

        $newTransInfo = [
            Crossroads_DibsD2_Helper_Data::FIELD_PAYMENT_STATE => $paymentState,
            Crossroads_DibsD2_Helper_Data::FIELD_TRANSACT => $transactionId,
            Crossroads_DibsD2_Helper_Data::FIELD_STATUSCODE => array_key_exists('statuscode', $params) ? $params['statuscode'] : '',
            Crossroads_DibsD2_Helper_Data::FIELD_ACQUIRER => array_key_exists('acquirer', $params) ? $params['acquirer'] : '',
            Crossroads_DibsD2_Helper_Data::FIELD_CARDEXPDATE => array_key_exists('cardexpdate', $params) ? $params['cardexpdate'] : '',
            Crossroads_DibsD2_Helper_Data::FIELD_CARDNOMASK => array_key_exists('cardnomask', $params) ? $params['cardnomask'] : '',
            Crossroads_DibsD2_Helper_Data::FIELD_CARDPREFIX => array_key_exists('cardprefix', $params) ? $params['cardprefix'] : '',
            Crossroads_DibsD2_Helper_Data::FIELD_CARDCOUNTRY => array_key_exists('cardcountry', $params) ? $params['cardcountry'] : '',
            Crossroads_DibsD2_Helper_Data::FIELD_FORM_DATA => $transaction->getAdditionalInformation(Crossroads_DibsD2_Helper_Data::FIELD_FORM_DATA),
            Crossroads_DibsD2_Helper_Data::FIELD_CALLBACK_DATA => $params
        ];

        switch ($paymentState) {
            case Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_WAITING:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_AUTHORIZATION_WAITING", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                if ($orderState !== Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                    Mage::log(
                            "Got message that payment is awaiting authorization but order is not 'pending' but '{$orderState}' instead.",
                            LOG_ERR,
                            Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL,
                            true
                    );
                    // todo: Tell someone about this?
                } else {
                    // Just save new data and wait for capture at a later time
                    foreach ($newTransInfo as $key => $val) {
                        $transaction->setAdditionalInformation($key, $val);
                    }
                    $transaction->save();
                }
                break;

            case Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_APPROVED:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_AUTHORIZATION_APPROVED", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                if ($orderState !== Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                    // todo: Tell someone about this?
                    throw new Crossroads_DibsD2_Exception("DibsD2::handleCallback() Wrong order state '{$orderState}' when payment authorization was approved.");
                }

                // Just save new data and wait for capture at a later time
                foreach ($newTransInfo as $key => $val) {
                    $transaction->setAdditionalInformation($key, $val);
                }
                $transaction->save();
                break;

            case Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_DECLINED:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_AUTHORIZATION_DECLINED", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                if ($orderState !== Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                    // todo: Tell someone about this?
                    throw new Crossroads_DibsD2_Exception("DibsD2::handleCallback() Wrong order state '{$orderState}' when payment authorization was declined.");
                }

                // Update current transaction
                foreach ($newTransInfo as $key => $val) {
                    $transaction->setAdditionalInformation($key, $val);
                }
                $transaction->save();

                // Prepare new transaction
                $payment->setParentTransactionId($oldTxnId);
                $payment->setTransactionId($transactionId);
                $payment->setShouldCloseParentTransaction(true);
                $payment->setTransactionAdditionalInfo($newTransInfo, null);
                $payment->setMessage(Mage::helper('DibsD2')->__('Authorization declined by Dibs.'));
                $payment->cancel();
                $order->save();
                break;

            case Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_WAITING:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_PAYMENT_WAITING", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                if ($orderState !== Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                    // todo: Tell someone about this?
                    throw new Crossroads_DibsD2_Exception("DibsD2::handleCallback() Wrong order state '{$orderState}' when payment capture is pending.");
                }

                // Just save callback data and wait for capture at a later time
                foreach ($newTransInfo as $key => $val) {
                    $transaction->setAdditionalInformation($key, $val);
                }
                $transaction->save();
                break;

            case Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_APPROVED:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_PAYMENT_APPROVED", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                if ($orderState !== Mage_Sales_Model_Order::STATE_PAYMENT_REVIEW) {
                    // todo: Tell someone about this?
                    throw new Crossroads_DibsD2_Exception("DibsD2::handleCallback() Wrong order state '{$orderState}' when payment capture is approved.");
                }

                // If not done previously: close authorization transaction
                if ($oldTransactionType == Mage_Sales_Model_Order_Payment_Transaction::TYPE_AUTH && $payment->getIsTransactionPending()) {
                    $payment->setIsTransactionPending(false);
                    $payment->setTransactionAdditionalInfo(Crossroads_DibsD2_Helper_Data::FIELD_CALLBACK_DATA, $params);
                    $payment->accept();
                    $order->save();
                }

                // Prepare new transaction
                $payment->setParentTransactionId($oldTxnId);
                $payment->setTransactionId($transactionId);
                $payment->setIsTransactionApproved(true);
                $payment->setIsTransactionPending(false);
                $payment->setIsTransactionClosed(true);
                $payment->setTransactionAdditionalInfo($newTransInfo, null);

                $invoice = Mage::getModel('sales/service_order', $order)->prepareInvoice();
                if ($invoice->getTotalQty() > 0) {
                    $invoice->setRequestedCaptureCase(Mage_Sales_Model_Order_Invoice::CAPTURE_ONLINE);
                    $invoice->register();
                    $transactionSave = Mage::getModel('core/resource_transaction');
                    $transactionSave->addObject($invoice)->addObject($order);
                    $transactionSave->save();
                    $order->save();

                    $order->sendNewOrderEmail();

                    Mage::dispatchEvent('crossroads_order_payment_complete', [ "order" => $order ]);
                } else {
                    Mage::log(
                            "Error invoicing order {$incrementId} (no products to invoice in order?)",
                            LOG_ERR,
                            Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL,
                            true
                    );
                    throw new Crossroads_DibsD2_Exception("Error invoicing order {$incrementId} (no products to invoice in order?)");
                }

                break;

            case Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_DECLINED:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_PAYMENT_DECLINED", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }

                // Update current transaction
                $payment->setTransactionAdditionalInfo(Crossroads_DibsD2_Helper_Data::FIELD_CALLBACK_DATA, $params);
                $payment->save();

                // Prepare new transaction
                $payment->setParentTransactionId($oldTxnId);
                $payment->setTransactionId($transactionId);
                $payment->setShouldCloseParentTransaction(true);
                $payment->setTransactionAdditionalInfo($newTransInfo, null);
                $payment->setMessage(Mage::helper('DibsD2')->__('Payment declined by Dibs.'));
                $payment->cancel();
                $order->save();
                break;

            case Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_CANCELED:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_PAYMENT_CANCELED", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }

                // Update current transaction
                $payment->setTransactionAdditionalInfo(Crossroads_DibsD2_Helper_Data::FIELD_CALLBACK_DATA, $params);
                $payment->save();

                // Prepare new transaction
                $payment->setParentTransactionId($oldTxnId);
                $payment->setTransactionId($transactionId);
                $payment->setShouldCloseParentTransaction(true);
                $payment->setTransactionAdditionalInfo($newTransInfo, null);
                $payment->setMessage(Mage::helper('DibsD2')->__('Payment canceled by customer.'));
                $payment->cancel();
                $order->save();
                break;

            case Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:STATE_NOT_APPLICABLE", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                // Do not change anything
                break;

            default:
                if (Crossroads_DibsD2_Helper_Data::DEBUGGING_MODE) {
                    Mage::log("handleCallback:invalid state", LOG_DEBUG, Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_NORMAL, true);
                }
                // todo: Tell someone about this?
                throw new Crossroads_DibsD2_Exception("DibsD2::handleCallback() Unknown payment state '{$paymentState}'");
                break;
        }
    }

}
