<?php

class Crossroads_DibsD2_Model_API extends Mage_Core_Model_Abstract
{
    /**
     * Sends a set of parameters to a DIBS API function
     * @param int $storeId
     * @param string $method The name of the target payment function, e.g. AuthorizeCard
     * @param array $params A set of parameters to be posted in key => value format
     * @return array
     */
    public function postToDIBS($storeId, $method, $params)
    {
        $output = ['raw' => '', 'translated' => []];

        $APIUser = Mage::getStoreConfig(Crossroads_DibsD2_Helper_Data::CONFIG_PATH_API_USER, $storeId);
        $APIPassword = Mage::getStoreConfig(Crossroads_DibsD2_Helper_Data::CONFIG_PATH_API_PASSWORD, $storeId);

        $postUrl = '';

        switch ($method) {
            case 'cancel': // Cancel authorization
                $postUrl = "https://{$APIUser}:{$APIPassword}@payment.architrade.com/cgi-adm/cancel.cgi";
                break;
            case 'capture': // Capture ticket (or authorization)
                $postUrl = "https://payment.architrade.com/cgi-bin/capture.cgi";
                break;
            case 'delticket': // Delete ticket
                $postUrl = "https://{$APIUser}:{$APIPassword}@payment.architrade.com/cgi-adm/delticket.cgi";
                break;
            case 'refund': // Refund previously captured transaction
                $postUrl = "https://{$APIUser}:{$APIPassword}@payment.architrade.com/cgi-adm/refund.cgi";
                break;
            case 'ticket_auth': // Authorize ticket
                $postUrl = "https://payment.architrade.com/cgi-ssl/ticket_auth.cgi";
                break;
            case 'callback': // Get information about callback for transaction
                $postUrl = "https://{$APIUser}:{$APIPassword}@payment.architrade.com/cgi-adm/callback.cgi";
                break;
            case 'cardtype': // Get information about cardtype for transaction
                $postUrl = "https://payment.architrade.com/cardtype.pml";
                break;
            case 'confirmtransact': // Get confirmation that the transaction exists by returning the amount authorized
                $postUrl = "https://payment.architrade.com/cgi-bin/confirmtransact.cgi";
                break;
            case 'payinfo': // Get information about transaction
                $postUrl = "https://{$APIUser}:{$APIPassword}@payment.architrade.com/cgi-adm/payinfo.cgi";
                break;
            case 'transinfo': // Get transaction information
                $postUrl = "https://payment.architrade.com/cgi-bin/transinfo.cgi";
                break;
            case 'transstat': // Get transaction status
                $postUrl = "http://payment.architrade.com/cgi-bin/transstat.cgi";
                break;
            case 'transstatus': // Get transaction status
                $postUrl = "http://payment.architrade.com/transstatus.pml";
                break;
            default:
                throw new Crossroads_DibsD2_Exception("DibsD2::postToDIBS() Unknown method: '{$method}'");
                break;
        }

        if (!empty($postUrl)) {

            // Verify that we have required fields
            $this->verifyDataToPost($storeId, $method, $params);

            $postData = http_build_query($params, '', '&', PHP_QUERY_RFC3986);

            //Use Curl to communicate with the server.
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $postUrl);
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
            curl_setopt($ch, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 15);
            $output['raw'] = curl_exec($ch);

            //Check for errors in the Curl operation
            if (curl_errno($ch) != 0) {
                $ex = new Crossroads_DibsD2_Exception("DibsD2:postToDIBS() Curl error");
                $ex->setExtendedInfo([
                    'curl_errno' => curl_errno($ch),
                    'curl_error' => curl_error($ch),
                    'curl_getinfo' => curl_getinfo($ch)
                ]);
                throw $ex;
            }
            curl_close($ch);

            /**
             * Sometimes we get single value or error text from Dibs
             * We need to check for this before we try to use parse_str
             * As of now we use "=" as identifier for url encoded parameter string
             */
            if (strpos($output['raw'], '=')) {
                parse_str($output['raw'], $output['translated']);
            } else {
                // Use special format for single return value
                $output['translated'] = ['*' => urldecode($output['raw'])];
            }

            $this->verifyDataFromDibs($storeId, $method, $output['translated']);

            Mage::log(
                    "postToDIBS(storeId:{$storeId}, method:'{$method}', params:'{$postData}'):\n" . print_r($output, true),
                    LOG_DEBUG,
                    Crossroads_DibsD2_Helper_Data::LOGGING_FILENAME_RESULTS,
                    true
            );

        }


        return $output;
    }

    /**
     * Verify data about to be posted to Dibs.
     * Also calculate MD5 if it is missing (hence the "&" on $params)
     * @param int $storeId
     * @param string $method The name of the target payment function, e.g. AuthorizeCard
     * @param array $params A set of parameters to be posted in key => value format
     * @return void
     */
    public function verifyDataToPost($storeId, $method, &$params)
    {
        $paramsSettings = $this->getParamSettingsOut($method);
        $calculateMD5 = false;

        foreach ($paramsSettings as $fieldKey => $fieldSetting) {
            if (array_key_exists($fieldKey, $params)) {
                switch ($fieldSetting['type']) {
                    case 'string':
                        if (array_key_exists('fixed', $fieldSetting)) {
                            $params[$fieldKey] = $fieldSetting['fixed'];
                        } else {
                            $params[$fieldKey] = "{$params[$fieldKey]}";
                        }
                        break;
                    case 'int':
                        $params[$fieldKey] = intval($params[$fieldKey]);
                        break;
                    case 'md5':
                        // If we already have md5 value, no need to calculate it
                        break;
                    default:
                        throw new Crossroads_DibsD2_Exception("DibsD2::verifyDataToPost() Unknown type '{$fieldSetting['type']}' for field '{$fieldKey}'");
                        break;
                }
            } elseif ($fieldSetting['required']) {
                if ($fieldSetting['type'] == 'md5') {
                    $calculateMD5 = true;
                } else {
                    throw new Crossroads_DibsD2_Exception("DibsD2::verifyDataToPost() Missing required field '{$fieldKey}'");
                }
            }
        }

        if ($calculateMD5) {
            $params['md5key'] = Mage::helper('DibsD2/API')->getMD5Key($storeId, $paramsSettings, $params);
        }
    }

    /**
     * Get outgoing parameter settings for specified payment function
     * @param string $method The name of the target payment function, e.g. AuthorizeCard
     * @return array
     */
    public function getParamSettingsOut($method)
    {
        $paramsTemplate = array();

        switch ($method) {
            case 'cancel': // Cancel authorization
                $paramsTemplate = array(
                    'md5key' => ['required' => true, 'type' => 'md5'],
                    'merchant' => ['required' => true, 'type' => 'int', 'md5_prio' => 1],
                    'orderid' => ['required' => true, 'type' => 'string', 'md5_prio' => 2],
                    'transact' => ['required' => true, 'type' => 'int', 'md5_prio' => 3],
                    'textreply' => ['required' => true, 'type' => 'string', 'fixed' => 'true'],
                    'account' => ['required' => false, 'type' => 'string'],
                    'fullreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true']
                );
                break;
            case 'capture': // Capture ticket (or authorization)
                $paramsTemplate = array(
                    'md5key' => ['required' => true, 'type' => 'md5'],
                    'merchant' => ['required' => true, 'type' => 'int', 'md5_prio' => 1],
                    'orderid' => ['required' => true, 'type' => 'string', 'md5_prio' => 2],
                    'transact' => ['required' => true, 'type' => 'int', 'md5_prio' => 3],
                    'amount' => ['required' => true, 'type' => 'int', 'md5_prio' => 4],
                    'close' => ['required' => false, 'type' => 'string', 'fixed' => 'true'],
                    'force' => ['required' => false, 'type' => 'string', 'fixed' => 'true'],
                    'splitpay' => ['required' => false, 'type' => 'string', 'fixed' => 'true'],
                    'account' => ['required' => false, 'type' => 'string'],
                    'textreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true'],
                    'fullreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true']
                );
                break;
            case 'delticket': // Delete ticket
                $paramsTemplate = array(
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'ticket' => ['required' => true, 'type' => 'int'],
                    'account' => ['required' => false, 'type' => 'string'],
                    'fullreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true']
                );
                break;
            case 'refund': // Refund previously captured transaction
                $paramsTemplate = array(
                    'md5key' => ['required' => true, 'type' => 'md5'],
                    'merchant' => ['required' => true, 'type' => 'int', 'md5_prio' => 1],
                    'orderid' => ['required' => true, 'type' => 'string', 'md5_prio' => 2],
                    'transact' => ['required' => true, 'type' => 'int', 'md5_prio' => 3],
                    'amount' => ['required' => true, 'type' => 'int', 'md5_prio' => 4],
                    'account' => ['required' => false, 'type' => 'string'],
                    'textreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true'],
                    'fullreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true']
                );
                break;
            case 'ticket_auth': // Authorize ticket
                $paramsTemplate = array(
                    'md5key' => ['required' => true, 'type' => 'md5'],
                    'merchant' => ['required' => true, 'type' => 'int', 'md5_prio' => 1],
                    'orderid' => ['required' => true, 'type' => 'string', 'md5_prio' => 2],
                    'ticket' => ['required' => true, 'type' => 'int', 'md5_prio' => 3],
                    'currency' => ['required' => true, 'type' => 'int', 'md5_prio' => 4],
                    'amount' => ['required' => true, 'type' => 'int', 'md5_prio' => 5],
                    'account' => ['required' => false, 'type' => 'string'],
                    'textreply' => ['required' => true, 'type' => 'string', 'fixed' => 'true'],
                    'fullreply' => ['required' => false, 'type' => 'string', 'fixed' => 'true'],
                    'capturenow' => ['required' => false, 'type' => 'string', 'fixed' => 'yes'],
                    'test' => ['required' => false, 'type' => 'int', 'fixed' => 1]
                );
                break;
            case 'callback': // Get information about callback for transaction
                $paramsTemplate = array(
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'transact' => ['required' => true, 'type' => 'int']
                );
                break;
            case 'cardtype': // Get information about cardtype for transaction
                $paramsTemplate = array(
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'transact' => ['required' => true, 'type' => 'int'],
                    'account' => ['required' => false, 'type' => 'string']
                );
                break;
            case 'confirmtransact': // Get confirmation that the transaction exists by returning the amount authorized
                $paramsTemplate = array(
                    'account' => ['required' => false, 'type' => 'string'],
                    'currency' => ['required' => true, 'type' => 'int'],
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'orderid' => ['required' => true, 'type' => 'string'],
                    'transact' => ['required' => true, 'type' => 'int']
                );
                break;
            case 'payinfo': // Get information about transaction
                $paramsTemplate = array(
                    'transact' => ['required' => true, 'type' => 'int']
                );
                break;
            case 'transinfo': // Get transaction information
                $paramsTemplate = array(
                    'account' => ['required' => false, 'type' => 'string'],
                    'amount' => ['required' => true, 'type' => 'int'],
                    'currency' => ['required' => true, 'type' => 'int'],
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'orderid' => ['required' => true, 'type' => 'string']
                );
                break;
            case 'transstat': // Get transaction status
                $paramsTemplate = array(
                    'account' => ['required' => false, 'type' => 'string'],
                    'amount' => ['required' => true, 'type' => 'int'],
                    'currency' => ['required' => true, 'type' => 'int'],
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'orderid' => ['required' => true, 'type' => 'string'],
                    'transact' => ['required' => true, 'type' => 'int']
                );
                break;
            case 'transstatus': // Get transaction status
                $paramsTemplate = array(
                    'account' => ['required' => false, 'type' => 'string'],
                    'merchant' => ['required' => true, 'type' => 'int'],
                    'transact' => ['required' => true, 'type' => 'int']
                );
                break;
            default:
                throw new Crossroads_DibsD2_Exception("DibsD2::getParamSettingsOut() Unknown method '{$method}'");
                break;
        }

        return $paramsTemplate;
    }

    /**
     * Translate data from Dibs.
     * Also calculate MD5 and check if it matches the one sent from Dibs
     * @param int $storeId
     * @param string $method The name of the target payment function, e.g. AuthorizeCard
     * @param array $params A set of parameters to be posted in key => value format
     * @return array
     */
    public function verifyDataFromDibs($storeId, $method, $params)
    {
        $paramsSettings = $this->getParamSettingsIn($method);

        // Only throw exception if $method is supposed return named parameters.
        // Since the default '*' always is present, we check for > 1 parameter
        if (array_key_exists('*', $params) && count($paramsSettings) > 1) {
            throw new Crossroads_DibsD2_Exception("DibsD2::verifyDataFromDibs() API call to Dibs responded with: {$params['*']}");
        }

        $MD5Field = null;
        foreach ($paramsSettings as $fieldKey => $fieldSetting) {
            if (array_key_exists($fieldKey, $params)) {
                switch ($fieldSetting['type']) {
                    case 'string':
                        $params[$fieldKey] = "{$params[$fieldKey]}";
                        break;
                    case 'int':
                        if (!is_numeric($params[$fieldKey])) {
                            throw new Crossroads_DibsD2_Exception("DibsD2::verifyDataFromDibs() Value for field '{$fieldKey}' is not a number '{$params[$fieldKey]}'");
                        }
                        $params[$fieldKey] = intval($params[$fieldKey]);
                        break;
                    case 'md5':
                        $MD5Field = $fieldKey;
                        break;
                    default:
                        break;
                }
            } elseif ($fieldSetting['required']) {
                throw new Crossroads_DibsD2_Exception("DibsD2::verifyDataFromDibs() Missing required field '{$fieldKey}'");
            }
        }

        if ($MD5Field !== null) {
            $ourCalculatedMD5 = Mage::helper('DibsD2/API')->getMD5Key($storeId, $paramsSettings, $params);
            if ($params[$MD5Field] != $ourCalculatedMD5) {
                $ex = new Crossroads_DibsD2_Exception("DibsD2::verifyDataFromDibs() MD5 mismatch");
                $ex->setExtendedInfo([
                    'store_id' => $storeId,
                    'method' => $method,
                    'params' => $params
                ]);
                throw $ex;
            }
        }
    }

    /**
     * Get incomming parameter settings for specified payment function
     * @param string $method The name of the target payment function, e.g. AuthorizeCard
     * @return array
     */
    public function getParamSettingsIn($method)
    {
        $paramsTemplate = array();

        switch ($method) {
            case 'cancel': // Cancel authorization
                $paramsTemplate = array(
                    'cardtype' => ['required' => false, 'type' => 'string'],
                    'result' => ['required' => true, 'type' => 'int'],
                    'status' => ['required' => true, 'type' => 'string'],
                    'transact' => ['required' => false, 'type' => 'int'],
                    'reason' => ['required' => false, 'type' => 'string']
                );
                break;
            case 'capture': // Capture ticket (or authorization)
                $paramsTemplate = array(
                    'cardtype' => ['required' => false, 'type' => 'string'],
                    'result' => ['required' => false, 'type' => 'int'],
                    'status' => ['required' => false, 'type' => 'string'],
                    'transact' => ['required' => false, 'type' => 'int'],
                    'message' => ['required' => false, 'type' => 'string'],
                    'reason' => ['required' => false, 'type' => 'string']
                );
                break;
            case 'delticket': // Delete ticket
                $paramsTemplate = array(
                    'status' => ['required' => false, 'type' => 'string'],
                    'message' => ['required' => false, 'type' => 'string']
                );
                break;
            case 'refund': // Refund previously captured transaction
                $paramsTemplate = array(
                    'result' => ['required' => false, 'type' => 'int'],
                );
                break;
            case 'ticket_auth': // Authorize ticket
                $paramsTemplate = array(
                    'approvalcode' => ['required' => false, 'type' => 'string'],
                    // Dibs is missing information on how to calculate md5 for ticket_auth
                    'authkey' => ['required' => false, 'type' => 'string'],
                    'cardtype' => ['required' => false, 'type' => 'string'],
                    'fee' => ['required' => false, 'type' => 'int'],
                    'transact' => ['required' => false, 'type' => 'int'],
                    'status' => ['required' => false, 'type' => 'string'],
                    'reason' => ['required' => false, 'type' => 'string']
                );
                break;
            case 'callback': // Get information about callback for transaction
                // This method needs special handling, se verifyCallback()
                // Just accept whatever Dibs sent.
                $paramsTemplate = array();
                break;
            case 'cardtype': // Get information about cardtype for transaction
                $paramsTemplate = array();
                break;
            case 'confirmtransact': // Get confirmation that the transaction exists by returning the amount authorized
                $paramsTemplate = array();
                break;
            case 'payinfo': // Get information about transaction
                $paramsTemplate = array(
                    'actioncode' => ['required' => false, 'type' => 'string'],
                    'amount' => ['required' => false, 'type' => 'int'],
                    'approvalcode' => ['required' => false, 'type' => 'string'],
                    'batch' => ['required' => false, 'type' => 'int'],
                    'currency' => ['required' => false, 'type' => 'string'],
                    'fee' => ['required' => false, 'type' => 'int'],
                    'orderid' => ['required' => false, 'type' => 'string'],
                    'status' => ['required' => false, 'type' => 'int'],
                    'transact' => ['required' => false, 'type' => 'int']
                );
                break;
            case 'transinfo': // Get transaction information
                $paramsTemplate = array(
                    'approvalcode' => ['required' => false, 'type' => 'string'],
                    'status' => ['required' => false, 'type' => 'int'],
                    'transact' => ['required' => false, 'type' => 'int']
                );
                break;
            case 'transstat': // Get transaction status
                $paramsTemplate = array();
                break;
            case 'transstatus': // Get transaction status
                $paramsTemplate = array();
                break;
            default:
                throw new Crossroads_DibsD2_Exception("DibsD2::getParamSettingsIn() Unknown method '{$method}'");
                break;
        }

        // Add generic setting for error messages from Dibs as they come without "url parameter" format.
        $paramsTemplate['*'] = ['required' => false, 'type' => 'string'];

        return $paramsTemplate;
    }

    public function verifyCallback($storeId, &$params)
    {
        if (array_key_exists('*', $params)) {
            throw new Crossroads_DibsD2_Exception("DibsD2::verifyCallback() API call to Dibs responded with: {$params['*']}");
        }
        $MD5Field = null;
        $paramsSettings = array(
            // Required parameters
            'authkey' => ['required' => true, 'type' => 'md5'],
            'transact' => ['required' => true, 'type' => 'int', 'md5_prio' => 1],
            'amount' => ['required' => true, 'type' => 'int', 'md5_prio' => 2],
            'currency' => ['required' => true, 'type' => 'string', 'md5_prio' => 3],
            'statuscode' => ['required' => true, 'type' => 'int'],
            // Optional parameters (if customer cancels payment, no card data is present)
            'acquirer' => ['required' => false, 'type' => 'string'],
            'cardexpdate' => ['required' => false, 'type' => 'string'],
            'cardnomask' => ['required' => false, 'type' => 'string'],
            'cardprefix' => ['required' => false, 'type' => 'string'],
            'cardcountry' => ['required' => false, 'type' => 'string'],
            'agreement' => ['required' => false, 'type' => 'int'],
            'approvalcode' => ['required' => false, 'type' => 'string'],
            'cardId' => ['required' => false, 'type' => 'string'],
            'checksum' => ['required' => false, 'type' => 'string'],
            'fee' => ['required' => false, 'type' => 'int'],
            'orderid' => ['required' => false, 'type' => 'string'],
            'paytype' => ['required' => false, 'type' => 'string'],
            'severity' => ['required' => false, 'type' => 'int'],
            'suspect' => ['required' => false, 'type' => 'string'],
            'threeDstatus' => ['required' => false, 'type' => 'int'],
        );

        foreach ($paramsSettings as $fieldKey => $fieldSetting) {
            if (array_key_exists($fieldKey, $params)) {
                switch ($fieldSetting['type']) {
                    case 'string':
                        $params[$fieldKey] = "{$params[$fieldKey]}";
                        break;
                    case 'int':
                        if (!is_numeric($params[$fieldKey])) {
                            throw new Crossroads_DibsD2_Exception("DibsD2::verifyCallback() value for field '{$fieldKey}' is not a number '{$params[$fieldKey]}'");
                        }
                        $params[$fieldKey] = intval($params[$fieldKey]);
                        break;
                    case 'md5':
                        $MD5Field = $fieldKey;
                        break;
                    default:
                        break;
                }
            } elseif ($fieldSetting['required']) {
                throw new Crossroads_DibsD2_Exception("DibsD2::verifyCallback() missing required field '{$fieldKey}'");
            }
        }

        if ($MD5Field !== null) {
            $ourCalculatedMD5 = Mage::helper('DibsD2/API')->getMD5Key($storeId, $paramsSettings, $params);
            if ($params[$MD5Field] != $ourCalculatedMD5) {
                $ex = new Crossroads_DibsD2_Exception("DibsD2::verifyCallback() MD5 mismatch");
                $ex->setExtendedInfo([
                    'store_id' => $storeId,
                    'params' => $params
                ]);
                throw $ex;
            }
        }
    }

}