<?php

class Crossroads_DibsD2_Helper_API extends Mage_Core_Helper_Abstract {

    const PAYMENT_URL = "https://payment.architrade.com/paymentweb/start.action";

    protected $_status_codes = array();

    public function __construct()
    {
        $this->_status_codes = array(
            'invalid' => array(
                'short' => 'invalid',
                'description' => 'Invalid information.',
                'extra' => 'Something must have gone wrong since payment is missing crucial information.',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_WAITING
            ),
            'missing_callback' => array(
                'short' => 'missing callback',
                'description' => 'No callback from DIBS received.',
                'extra' => 'We have not yet gotten any callback from DIBS. If the customer just closes the payment window or there are network problems, this might happen.',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_WAITING
            ),
            '0' => array(
                'short' => 'transaction inserted',
                'description' => 'The transaction has been sent to the acquirer, but no response was received.',
                'extra' => 'If this status is returned, the affiliated acquirer should be contacted and asked if the transaction has been authorized and DIBS support should be contacted with the status of the authorization to release the transaction.',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_WAITING
            ),
            '1' => array(
                'short' => 'declined',
                'description' => 'The transaction is declined by acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_DECLINED
            ),
            '2' => array(
                'short' => 'authorization approved',
                'description' => 'The transaction is approved by acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_AUTHORIZATION_APPROVED
            ),
            '3' => array(
                'short' => 'capture sent to acquirer',
                'description' => 'The transaction capture request is sent to the acquirer and DIBS awaits response.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_WAITING
            ),
            '4' => array(
                'short' => 'capture declined by acquirer',
                'description' => 'The transaction capture is declined by the acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_DECLINED
            ),
            '5' => array(
                'short' => 'capture completed',
                'description' => 'The transaction capture is approved by the acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_APPROVED
            ),
            '6' => array(
                'short' => 'authorization deleted',
                'description' => 'The transaction authorization is deleted at the acquirer. If the transaction is still alive a reauthorization can be made.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_DECLINED
            ),
            '7' => array(
                'short' => 'capture balanced',
                'description' => 'The transaction is balanced.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '8' => array(
                'short' => 'partially refunded and balanced',
                'description' => 'The transaction is balanced and partially refunded.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '9' => array(
                'short' => 'refund sent to acquirer',
                'description' => 'The refund request is sent to the acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '10' => array(
                'short' => 'refund declined',
                'description' => 'The refund is declined by the acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '11' => array(
                'short' => 'refund completed',
                'description' => 'The refund is approved by the acquirer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '12' => array(
                'short' => 'capture pending',
                'description' => 'The transaction capture is pending.',
                'extra' => 'The capture is done in batches and this transaction is in one. If nothing goes horribly wrong, payment is going to be done. If needed, check current payment status at Dibs.',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_APPROVED
            ),
            '13' => array(
                'short' => '"ticket" transaction',
                'description' => 'The transaction is a "ticket" transaction.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '14' => array(
                'short' => 'deleted "ticket" transaction',
                'description' => 'The "ticket" transaction is deleted.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '15' => array(
                'short' => 'refund pending',
                'description' => 'The transaction refund is pending.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '16' => array(
                'short' => 'waiting for shop approval',
                'description' => 'The transaction is waiting for the shop to approve the transaction.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_WAITING
            ),
            '17' => array(
                'short' => 'declined by DIBS',
                'description' => 'The transaction is declined by DIBS.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_DECLINED
            ),
            '18' => array(
                'short' => 'multicap transaction open',
                'description' => 'The transaction is a multicapture transaction and is still active.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '19' => array(
                'short' => 'multicap transaction closed',
                'description' => 'The transaction is a multicapture transaction and is not active.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_NOT_APPLICABLE
            ),
            '26' => array(
                'short' => 'postponed',
                'description' => 'The transaction outcome is not yet decided by the acquirer, DIBS will continue to poll the transaction status.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_WAITING
            ),
            '99' => array(
                'short' => 'canceled',
                'description' => 'The order has been canceled by customer.',
                'extra' => '',
                'payment_state' => Crossroads_DibsD2_Helper_Data::STATE_PAYMENT_CANCELED
            )
        );
    }

    public function getPaymentStateByCode($code)
    {
        $info = $this->getPaymentInformationByCode($code);
        return $info['payment_state'];
    }

    public function getPaymentInformationByCode($code)
    {
        if (array_key_exists("{$code}", $this->_status_codes)) {
            return $this->_status_codes["{$code}"];
        } else {
            return $this->_status_codes['invalid'];
        }
    }

    public function getMissingStatus()
    {
        return $this->_default_codes['missing_status'];
    }

    public function getNumericCurrencyId($currency)
    {
        switch(strtolower($currency)) {
            case 'nok':
                return 578;
            case 'dkk':
                return 208;
            case 'sek':
                return 752;
            case 'eur':
            default:
                return 978;
            
        }
    }

    public function getMD5Key($storeId, $paramsSettings, $params)
    {
        $paramsToUse = array();
        foreach ($paramsSettings as $key => $val) {
            if (!empty($val['md5_prio'])) {
                if (array_key_exists($val['md5_prio'], $paramsToUse)) {
                    $ex = new Crossroads_DibsD2_Exception("DibsD2::getMD5Key() Duplicate md5_prio '{$val['md5_prio']}' when calculating md5.");
                    $ex->setExtendedInfo([
                        'settings' => $paramsSettings,
                        'params' => $params,
                    ]);
                    throw $ex;
                }
                $paramsToUse[$val['md5_prio']] = $key;
            }
        }

        $parameterString = '';
        foreach ($paramsToUse as $key) {
            if (array_key_exists($key, $params)) {
                $parameterString .= "{$key}={$params[$key]}&";
            } else {
                $ex = new Crossroads_DibsD2_Exception("DibsD2::getMD5Key() Missing required field '{$key}' when calculating md5.");
                $ex->setExtendedInfo([
                    'settings' => $paramsSettings,
                    'params' => $params,
                    'required' => $paramsToUse
                ]);
                throw $ex;
            }
        }
        if (!empty($parameterString)) {
            $parameterString = substr($parameterString, 0, -1); // Remove trailing "&"
        } else {
            $ex = new Crossroads_DibsD2_Exception("DibsD2::getMD5Key() No parameterString was created.");
            $ex->setExtendedInfo([
                'settings' => $paramsSettings,
                'params' => $params,
                'required' => $paramsToUse
            ]);
            throw $ex;
        }

        $md5key1 = Mage::getStoreConfig('payment/DibsD2/md5key1', $storeId);
        $md5key2 = Mage::getStoreConfig('payment/DibsD2/md5key2', $storeId);

        return md5($md5key2 . md5($md5key1 . $parameterString));
    }

    public function getAcceptUrl($payment)
    {
        $order = $payment->getOrder();
        $params = array(
            'incrementid' => $order->getIncrementId()
        );
        return $this->_formatUrl($order->getStoreId(), 'accept', $params);

    }

    public function getCancelUrl($payment)
    {
        $order = $payment->getOrder();
        $params = array(
            'incrementid' => $order->getIncrementId()
        );
        return $this->_formatUrl($order->getStoreId(), 'cancel', $params);
    }

    public function getCallbackUrl($payment)
    {
        $order = $payment->getOrder();
        $params = array(
            'incrementid' => $order->getIncrementId()
        );
        return $this->_formatUrl($order->getStoreId(), 'system', $params);
    }

    public function getNotifyUrl($payment)
    {
        $order = $payment->getOrder();
        $params = array(
            'incrementid' => $order->getIncrementId()
        );
        return $this->_formatUrl($order->getStoreId(), 'system', $params);
    }

    protected function _formatUrl($storeId, $action, $params)
    {
        $url = Mage::app()->getStore($storeId)->getBaseUrl(Mage_Core_Model_Store::URL_TYPE_WEB);
        $url .= "dibsd2/callback/{$action}/";
        foreach ($params as $key => $val) {
            $url .= urlencode($key) . '/' . urlencode($val) . '/';
        }
        return $url;
    }

}
