<?php

declare(strict_types=1);

use GraphQL\Type\Definition\ResolveInfo;

use MageQL\Registry;
use MageQL\Type\AbstractBuilder;
use MageQL\Type\FieldBuilder;

class Crossroads_Collector_Model_Graphql_Payment extends MageQL_Sales_Model_Payment_Abstract {
    public function getCode(): string {
        return "Crossroads_Collector";
    }

    public function getMethodTypeName(): string {
        return "CollectorInvoice";
    }

    /**
     * @return Array<FieldBuilder>
     */
    public function getPaymentMethodFields(Registry $registry): array {
        return [];
    }

    public function getQuotePaymentMutationFieldBuilder(Registry $registry): FieldBuilder {
        $builder = parent::getQuotePaymentMutationFieldBuilder($registry);

        $builder->addArgument("ssn", $this->argument("String!", "The social security number of the customer"));
        $builder->addArgument("isCompany", $this->argument("Boolean", "If the customer is a company"));

        return $builder;
    }

    public function getQuotePaymentFields(Registry $registry): array {
        return [
            "socialSecurityNumber" => $this->field("String", "SSN of customer")
                ->setResolver([$this, "resolveSSN"]),
            "isCompany" => $this->field("Boolean", "If the customer is a company or not")
                ->setResolver([$this, "resolveIsCompany"]),
        ];
    }

    public function getMutateSetQuotePaymentFields(Registry $registry): array {
        return [
            "result" => $this->field(
                $this->getMutationResultType()."!",
                "Result of attempting to set the payment method to Collector Checkout"
            )
                // Simple result, we either succeed or fail
                ->setResolver("MageQL\\forwardResolver"),
            "socialSecurityNumber" => $this->field("String", "SSN of customer")
                ->setResolver([$this, "resolveSSN"]),
            "isCompany" => $this->field("Boolean", "If the customer is a company or not")
                ->setResolver([$this, "resolveIsCompany"]),
        ];
    }

    public function resolveSSN(): ?bool {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();
        $payment = $quote->getPayment();
        $isCompany = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_IS_COMPANY);

        return is_null($isCompany) ? null : (boolean)$isCompany;
    }

    public function resolveIsCompany(): ?string {
        $model = Mage::getSingleton("mageql_sales/quote");
        $quote = $model->getQuote();
        $payment = $quote->getPayment();
        $ssn = $payment->getAdditionalInformation(Crossroads_Collector_Helper_Data::FIELD_SSN);

        return $ssn && is_string($ssn) && strlen($ssn) > 0 ? $ssn : null;
    }

    public function mutateSetPaymentMethod(Mage_Sales_Model_Quote $quote, array $args, $ctx, ResolveInfo $info) {
        $model = Mage::getSingleton("mageql_sales/quote");
        $session = Mage::getSingleton("checkout/session");

        try {
            $paymentData = [
                "socialSecurityNumber" => $args["ssn"],
            ];

            if(array_key_exists("isCompany", $args)) {
                $paymentData["isCompany"] = $args["isCompany"];
            }

            $model->importPaymentData($quote, "Crossroads_Collector", $paymentData);
            $model->saveSessionQuote();

            // Reset old session data in case we had a previous order
            $session->setLastOrderId(null);
            $session->setLastQuoteId(null);
            $session->setLastSuccessQuoteId(null);
            $session->setLastOrderId(null);
            $session->setLastRealOrderId(null);

            return self::SUCCESS;
        }
        catch(Mage_Core_Exception $e) {
            return $this->translatePaymentError($e);
        }
    }
}
