<?php

class Crossroads_Collector_Helper_Data extends Mage_Core_Helper_Abstract
{
    /**
     * Event triggered when payment status has been updated for a given order from callback.
     *
     * Params:
     *  * order          Mage_Sales_Model_Order
     *  * payment        Payment instance for order
     *  * invoice_status New invoice status, one of the Crossroads_Collector_Helper_Data::INVOICE_STATUS_* constants.
     *  * invoice_no     Collector invoice number
     */
    const EVENT_POST_ORDER_STATUS_UPDATE = "crossroads_collector_post_order_status_update";

    const QUOTE_FIELD_SSN = "socialSecurityNumber";

    const CONFIG_PATH_MIN_ORDER_TOTAL       = "payment/Crossroads_Collector/min_order_total";
    const CONFIG_PATH_ORDER_STATUS_FALLBACK = "payment/Crossroads_Collector/order_status_fallback";

    /**
     * Field containing the Collector invoice id.
     */
    const FIELD_INVOICE_ID         = "collector_invoice_id";
    /**
     * Field containing the Collector invoice status code.
     */
    const FIELD_INVOICE_STATUS     = "collector_invoice_status";
    /**
     * Field containing social security number (if individual) or organization number (if company).
     */
    const FIELD_SSN                = "collector_ssn";
    /**
     * Field containing boolean, if true the fallback account was used for the transaction.
     */
    const FIELD_USING_FALLBACK     = "collector_using_fallback";
    /**
     * Store id used by collector.
     */
    const FIELD_STORE_ID           = "collector_store_id";
    /**
     * True if it is a company order.
     */
    const FIELD_IS_COMPANY         = "collector_is_company";
    /**
     * Payment reference number once invoice is activated.
     */
    const FIELD_PAYMENT_REFERENCE  = "collector_payment_reference";

    /**
     * On hold for credit check.
     */
    const INVOICE_STATUS_ON_HOLD     = 0;
    /**
     * Passed preliminary credit check.
     */
    const INVOICE_STATUS_PRELIMINARY = 1;
    /**
     * Invoice is activated
     */
    const INVOICE_STATUS_ACTIVATED   = 2;
    /**
     * Credit check rejected.
     */
    const INVOICE_STATUS_REJECTED    = 5;
    /**
     * The invoice needs to be digitally signed by end customer before the invoice can be activated.
     */
    const INVOICE_STATUS_SIGNING     = 6;

    /**
     * Registers the payment method with the Crossroads_API.
     */
    public function registerPayment(Varien_Event_Observer $observer) {
        $observer->getRegistry()->addMethod(new Crossroads_Collector_Model_Payment());
    }

    /**
     * Returns true if the supplied org-nr or social security number is an organization number.
     *
     * An organization number has a month of 20 or greater.
     *
     * @param  string
     * @return boolean
     */
    public function isOrganizationNumber($nr) {
        // Drop everything which is not a number
        $nr = preg_replace('/[^\d]/', '', $nr);

        // If the org-nr is more than 10 numbers long, assume it is in the format YYYYMMDDXXXX,
        // otherwise it is YYMMDDXXXX
        return ((int) substr($nr, strlen($nr) > 10 ? 4 : 2, 2)) >= 20;
    }

    /**
     * Retrieves the Collector Bank Store Id magento configuration key to be used for a transaction.
     *
     * @param  boolean  If we are attempting the norisk fallback
     * @param  boolean  If we are performing a transaction with a company customer
     * @return string
     */
    public function getStoreIdKey($fallback, $isCompany) {
        if($fallback) {
            if($isCompany) {
                return "payment/Crossroads_Collector/fallback_company_store_id";
            }
            else {
                return "payment/Crossroads_Collector/fallback_store_id";
            }
        }
        else {
            if($isCompany) {
                return "payment/Crossroads_Collector/company_store_id";
            }
            else {
                return "payment/Crossroads_Collector/store_id";
            }
        }
    }

    /**
     * Validate the social security number if we use our own payment method.
     */
    public function checkoutPostVerifyQuote(Varien_Event_Observer $observer) {
        $quote = $observer->getQuote();

        if($quote->getPayment()->getMethod() !== "Crossroads_Collector") {
            return;
        }

        $ssn = $quote->getPayment()->getAdditionalInformation(self::FIELD_SSN);

        if( ! $ssn) {
            throw Crossroads_API_ResponseException::create(400, "Missing required paymentMethodData.socialSecurityNumber", null, 12008);
        }
    }

    /**
     * Updates an order given a request from Collector.
     *
     * Expected query-parameters:
     *  * InvoiceNo:     Collector invoice number
     *  * OrderNo:       Magento increment id order number
     *  * InvoiceStatus: New integer status from Collector
     *
     * @param  Zend_Controller_Request
     * @param  Zend_Controller_Response_Http
     * @return Zend_Controller_Response_Http
     */
    public function handleOrderStatusUpdate($request, $response) {
        $invoiceNo     = $request->getParam("InvoiceNo");
        $orderNo       = $request->getParam("OrderNo");
        $invoiceStatus = $request->getParam("InvoiceStatus");

        if(empty($invoiceNo) || empty($orderNo) || empty($invoiceStatus)) {
            return $response->setHttpResponseCode(400)
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([ "message" => "Missing query parameters" ]));
        }

        // Magento does not include the Zend code for managing HTTP Basic Authentication,
        // so we roll our own
        $authData = explode(" ", trim($request->getHeader("Authorization")), 2);

        if(count($authData) != 2) {
            return $response->setHttpResponseCode(401)
                ->setHeader("Content-Type", "application/json", true)
                ->setHeader("WWW-Authenticate", "Basic realm=\"Collector Endpoint\"", true)
                ->setBody(json_encode([ "message" => "Malformated Authorization header" ]));
        }

        if(strtolower($authData[0]) !== "basic") {
            return $response->setHttpResponseCode(401)
                ->setHeader("Content-Type", "application/json", true)
                ->setHeader("WWW-Authenticate", "Basic realm=\"Collector Endpoint\"", true)
                ->setBody(json_encode([ "message" => "Requires basic authentication" ]));
        }

        $decoded = base64_decode($authData[1]);

        if( ! $decoded) {
            return $response->setHttpResponseCode(401)
                ->setHeader("Content-Type", "application/json", true)
                ->setHeader("WWW-Authenticate", "Basic realm=\"Collector Endpoint\"", true)
                ->setBody(json_encode([ "message" => "Invalid base64-data in Authorization header" ]));
        }

        $decodedSplit = explode(":", $decoded, 2);

        if(count($decodedSplit) != 2) {
            return $response->setHttpResponseCode(401)
                ->setHeader("Content-Type", "application/json", true)
                ->setHeader("WWW-Authenticate", "Basic realm=\"Collector Endpoint\"", true)
                ->setBody(json_encode([ "message" => "Invalid data in Authorization header" ]));
        }

        list($username, $password) = $decodedSplit;

        $collector = Mage::getModel("Crossroads_Collector/collector");
        $order     = Mage::getModel("sales/order");

        $order->loadByIncrementId($orderNo);

        if( ! $order || ! $order->getIncrementId()) {
            return $this->sendData([404]);
        }

        $payment     = $order->getPayment();
        $method      = $payment->getMethodInstance();
        $store       = Mage::app()->getStore($order->getStoreId());
        $reqUsername = $store->getConfig("payment/Crossroads_Collector/notification_username");
        $reqPassword = $store->getConfig("payment/Crossroads_Collector/notification_password");

        if($username !== $reqUsername || $password !== $reqPassword) {
            return $response->setHttpResponseCode(403)
                ->setHeader("Content-Type", "application/json", true)
                ->setHeader("WWW-Authenticate", "Basic realm=\"Collector Endpoint\"", true)
                ->setBody(json_encode([ "message" => "Invalid username or password" ]));
        }

        $method->setStore($order->getStoreId());

        if($payment->getMethod() !== "Crossroads_Collector") {
            return $response->setHttpResponseCode(404)
                ->setHeader("Content-Type", "application/json", true)
                ->setBody(json_encode([ "message" => "Invalid payment method for Collector handleOrderStatusUpdate." ]));
        }


        Mage::log(sprintf("Crossroads_Collector: Updating payment status for order %s (invoice %s) to %s.", $order->getIncrementId(), $invoiceNo, $invoiceStatus));

        $method->updatePaymentStatus($payment, $invoiceNo, $invoiceStatus);

        $payment->save();
        $order->save();

        Mage::dispatchEvent(self::EVENT_POST_ORDER_STATUS_UPDATE, [
            "order"          => $order,
            "payment"        => $payment,
            "invoice_status" => $invoiceStatus,
            "invoice_no"     => $invoiceNo
        ]);

        Mage::log(sprintf("Crossroads_Collector: Updated payment status for order %s (invoice %s) to %s.", $order->getIncrementId(), $invoiceNo, $invoiceStatus));

        return $response->setHttpResponseCode(200);
    }
}
